/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.azure.eventgrid;

import java.util.Map;

import org.apache.camel.Category;
import org.apache.camel.Component;
import org.apache.camel.Consumer;
import org.apache.camel.Processor;
import org.apache.camel.Producer;
import org.apache.camel.spi.EndpointServiceLocation;
import org.apache.camel.spi.UriEndpoint;
import org.apache.camel.spi.UriParam;
import org.apache.camel.support.DefaultEndpoint;
import org.apache.camel.util.ObjectHelper;

/**
 * Send events to Azure Event Grid topics.
 */
@UriEndpoint(firstVersion = "4.17.0", scheme = "azure-eventgrid", title = "Azure Event Grid",
             syntax = "azure-eventgrid:topicEndpoint", producerOnly = true, category = {
                     Category.CLOUD, Category.MESSAGING },
             headersClass = EventGridConstants.class)
public class EventGridEndpoint extends DefaultEndpoint implements EndpointServiceLocation {

    @UriParam
    private EventGridConfiguration configuration;

    public EventGridEndpoint(final String uri, final Component component, final EventGridConfiguration configuration) {
        super(uri, component);
        this.configuration = configuration;
    }

    @Override
    public Producer createProducer() throws Exception {
        return new EventGridProducer(this);
    }

    @Override
    public Consumer createConsumer(Processor processor) throws Exception {
        throw new UnsupportedOperationException("Consumer is not supported for Azure Event Grid component");
    }

    /**
     * The component configurations
     */
    public EventGridConfiguration getConfiguration() {
        return configuration;
    }

    public void setConfiguration(EventGridConfiguration configuration) {
        this.configuration = configuration;
    }

    @Override
    public String getServiceUrl() {
        if (ObjectHelper.isNotEmpty(configuration.getTopicEndpoint())) {
            return configuration.getTopicEndpoint();
        }
        return null;
    }

    @Override
    public String getServiceProtocol() {
        return "eventgrid";
    }

    @Override
    public Map<String, String> getServiceMetadata() {
        if (configuration.getTopicEndpoint() != null) {
            return Map.of("topicEndpoint", configuration.getTopicEndpoint());
        }
        return null;
    }
}
