﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/clouddirectory/CloudDirectoryRequest.h>
#include <aws/clouddirectory/CloudDirectory_EXPORTS.h>
#include <aws/clouddirectory/model/TypedLinkSpecifier.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CloudDirectory {
namespace Model {

/**
 */
class DetachTypedLinkRequest : public CloudDirectoryRequest {
 public:
  AWS_CLOUDDIRECTORY_API DetachTypedLinkRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DetachTypedLink"; }

  AWS_CLOUDDIRECTORY_API Aws::String SerializePayload() const override;

  AWS_CLOUDDIRECTORY_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the directory where you want to detach the
   * typed link.</p>
   */
  inline const Aws::String& GetDirectoryArn() const { return m_directoryArn; }
  inline bool DirectoryArnHasBeenSet() const { return m_directoryArnHasBeenSet; }
  template <typename DirectoryArnT = Aws::String>
  void SetDirectoryArn(DirectoryArnT&& value) {
    m_directoryArnHasBeenSet = true;
    m_directoryArn = std::forward<DirectoryArnT>(value);
  }
  template <typename DirectoryArnT = Aws::String>
  DetachTypedLinkRequest& WithDirectoryArn(DirectoryArnT&& value) {
    SetDirectoryArn(std::forward<DirectoryArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Used to accept a typed link specifier as input.</p>
   */
  inline const TypedLinkSpecifier& GetTypedLinkSpecifier() const { return m_typedLinkSpecifier; }
  inline bool TypedLinkSpecifierHasBeenSet() const { return m_typedLinkSpecifierHasBeenSet; }
  template <typename TypedLinkSpecifierT = TypedLinkSpecifier>
  void SetTypedLinkSpecifier(TypedLinkSpecifierT&& value) {
    m_typedLinkSpecifierHasBeenSet = true;
    m_typedLinkSpecifier = std::forward<TypedLinkSpecifierT>(value);
  }
  template <typename TypedLinkSpecifierT = TypedLinkSpecifier>
  DetachTypedLinkRequest& WithTypedLinkSpecifier(TypedLinkSpecifierT&& value) {
    SetTypedLinkSpecifier(std::forward<TypedLinkSpecifierT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_directoryArn;

  TypedLinkSpecifier m_typedLinkSpecifier;
  bool m_directoryArnHasBeenSet = false;
  bool m_typedLinkSpecifierHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudDirectory
}  // namespace Aws
