﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iotsitewise/IoTSiteWiseRequest.h>
#include <aws/iotsitewise/IoTSiteWise_EXPORTS.h>
#include <aws/iotsitewise/model/ListAssetModelPropertiesFilter.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace IoTSiteWise {
namespace Model {

/**
 */
class ListAssetModelPropertiesRequest : public IoTSiteWiseRequest {
 public:
  AWS_IOTSITEWISE_API ListAssetModelPropertiesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListAssetModelProperties"; }

  AWS_IOTSITEWISE_API Aws::String SerializePayload() const override;

  AWS_IOTSITEWISE_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The ID of the asset model. This can be either the actual ID in UUID format,
   * or else <code>externalId:</code> followed by the external ID, if it has one. For
   * more information, see <a
   * href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/object-ids.html#external-id-references">Referencing
   * objects with external IDs</a> in the <i>IoT SiteWise User Guide</i>.</p>
   */
  inline const Aws::String& GetAssetModelId() const { return m_assetModelId; }
  inline bool AssetModelIdHasBeenSet() const { return m_assetModelIdHasBeenSet; }
  template <typename AssetModelIdT = Aws::String>
  void SetAssetModelId(AssetModelIdT&& value) {
    m_assetModelIdHasBeenSet = true;
    m_assetModelId = std::forward<AssetModelIdT>(value);
  }
  template <typename AssetModelIdT = Aws::String>
  ListAssetModelPropertiesRequest& WithAssetModelId(AssetModelIdT&& value) {
    SetAssetModelId(std::forward<AssetModelIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token to be used for the next set of paginated results.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListAssetModelPropertiesRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to return for each paginated request. If not
   * specified, the default value is 50.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListAssetModelPropertiesRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Filters the requested list of asset model properties. You can choose one of
   * the following options:</p> <ul> <li> <p> <code>ALL</code> – The list includes
   * all asset model properties for a given asset model ID. </p> </li> <li> <p>
   * <code>BASE</code> – The list includes only base asset model properties for a
   * given asset model ID. </p> </li> </ul> <p>Default: <code>BASE</code> </p>
   */
  inline ListAssetModelPropertiesFilter GetFilter() const { return m_filter; }
  inline bool FilterHasBeenSet() const { return m_filterHasBeenSet; }
  inline void SetFilter(ListAssetModelPropertiesFilter value) {
    m_filterHasBeenSet = true;
    m_filter = value;
  }
  inline ListAssetModelPropertiesRequest& WithFilter(ListAssetModelPropertiesFilter value) {
    SetFilter(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version alias that specifies the latest or active version of the asset
   * model. The details are returned in the response. The default value is
   * <code>LATEST</code>. See <a
   * href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/model-active-version.html">
   * Asset model versions</a> in the <i>IoT SiteWise User Guide</i>.</p>
   */
  inline const Aws::String& GetAssetModelVersion() const { return m_assetModelVersion; }
  inline bool AssetModelVersionHasBeenSet() const { return m_assetModelVersionHasBeenSet; }
  template <typename AssetModelVersionT = Aws::String>
  void SetAssetModelVersion(AssetModelVersionT&& value) {
    m_assetModelVersionHasBeenSet = true;
    m_assetModelVersion = std::forward<AssetModelVersionT>(value);
  }
  template <typename AssetModelVersionT = Aws::String>
  ListAssetModelPropertiesRequest& WithAssetModelVersion(AssetModelVersionT&& value) {
    SetAssetModelVersion(std::forward<AssetModelVersionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_assetModelId;
  bool m_assetModelIdHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  ListAssetModelPropertiesFilter m_filter{ListAssetModelPropertiesFilter::NOT_SET};
  bool m_filterHasBeenSet = false;

  Aws::String m_assetModelVersion;
  bool m_assetModelVersionHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoTSiteWise
}  // namespace Aws
