﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/mailmanager/MailManagerRequest.h>
#include <aws/mailmanager/MailManager_EXPORTS.h>
#include <aws/mailmanager/model/ImportDataFormat.h>

#include <utility>

namespace Aws {
namespace MailManager {
namespace Model {

/**
 */
class CreateAddressListImportJobRequest : public MailManagerRequest {
 public:
  AWS_MAILMANAGER_API CreateAddressListImportJobRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateAddressListImportJob"; }

  AWS_MAILMANAGER_API Aws::String SerializePayload() const override;

  AWS_MAILMANAGER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>A unique token that Amazon SES uses to recognize subsequent retries of the
   * same request.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateAddressListImportJobRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier of the address list for importing addresses to.</p>
   */
  inline const Aws::String& GetAddressListId() const { return m_addressListId; }
  inline bool AddressListIdHasBeenSet() const { return m_addressListIdHasBeenSet; }
  template <typename AddressListIdT = Aws::String>
  void SetAddressListId(AddressListIdT&& value) {
    m_addressListIdHasBeenSet = true;
    m_addressListId = std::forward<AddressListIdT>(value);
  }
  template <typename AddressListIdT = Aws::String>
  CreateAddressListImportJobRequest& WithAddressListId(AddressListIdT&& value) {
    SetAddressListId(std::forward<AddressListIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A user-friendly name for the import job.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateAddressListImportJobRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The format of the input for an import job.</p>
   */
  inline const ImportDataFormat& GetImportDataFormat() const { return m_importDataFormat; }
  inline bool ImportDataFormatHasBeenSet() const { return m_importDataFormatHasBeenSet; }
  template <typename ImportDataFormatT = ImportDataFormat>
  void SetImportDataFormat(ImportDataFormatT&& value) {
    m_importDataFormatHasBeenSet = true;
    m_importDataFormat = std::forward<ImportDataFormatT>(value);
  }
  template <typename ImportDataFormatT = ImportDataFormat>
  CreateAddressListImportJobRequest& WithImportDataFormat(ImportDataFormatT&& value) {
    SetImportDataFormat(std::forward<ImportDataFormatT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;

  Aws::String m_addressListId;
  bool m_addressListIdHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  ImportDataFormat m_importDataFormat;
  bool m_importDataFormatHasBeenSet = false;
};

}  // namespace Model
}  // namespace MailManager
}  // namespace Aws
