## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
    collapse = TRUE,
    comment = "#>"
)

## ----install bioc, eval=FALSE-------------------------------------------------
# if (!requireNamespace("BiocManager")) {
#     install.packages("BiocManager")
# }
# BiocManager::install("BatchSVG")

## ----install github, eval = FALSE---------------------------------------------
# remotes::install("christinehou11/BatchSVG")

## ----library, message=FALSE---------------------------------------------------
library(BatchSVG)
library(spatialLIBD)
library(cowplot)

## ----load data, comment=NA, warning=FALSE, message=FALSE----------------------
spatialLIBD_spe <- fetch_data(type = "spe")
spatialLIBD_spe

## ----load nnsvg, comment=NA, warning=FALSE, message=FALSE---------------------
libd_nnsvgs <- read.csv(
    system.file("extdata","libd-all_nnSVG_p-05-features-df.csv",
        package = "BatchSVG"),
    row.names = 1, check.names = FALSE)

## ----feature select, comment = NA, warning=FALSE------------------------------
list_batch_df <- featureSelect(input = spatialLIBD_spe, 
    batch_effect = "subject", VGs = libd_nnsvgs$gene_id)

## ----eval=FALSE---------------------------------------------------------------
# list_batch_df <- featureSelect(input = spatialLIBD_spe,
#     batch_effect = "subject", VGs = libd_nnsvgs$gene_id, verbose = FALSE)

## ----feature select class, comment = NA, warning=FALSE------------------------
class(list_batch_df)

## ----feature select print, comment = NA, warning=FALSE------------------------
head(list_batch_df$subject)

## ----svg, comment=NA, warning=FALSE, message=FALSE----------------------------
plots <- svg_nSD(list_batch_df = list_batch_df, 
            sd_interval_dev = 3, sd_interval_rank = 3)

## ----figure 1, warning=FALSE, message=FALSE, fig.width=10, fig.height=8-------
plots$subject

## ----bias detect dev, comment = NA, message=FALSE, warning=FALSE--------------
bias_dev <- biasDetect(list_batch_df = list_batch_df, 
    threshold = "dev", nSD_dev = 3)

## ----table 1, comment = NA, message=FALSE, warning=FALSE----------------------
head(bias_dev$subject$Table)

## ----size change, message=FALSE, warning=FALSE--------------------------------
# size default = 3
bias_dev_size <- biasDetect(list_batch_df = list_batch_df, 
    threshold = "dev", nSD_dev = 3, plot_point_size = 4)

## ----figure 2, warning=FALSE, message=FALSE, fig.width= 10, fig.height=4------
plot_grid(bias_dev$subject$Plot, bias_dev_size$subject$Plot)

## ----bias detect rank, comment = NA, message=FALSE, warning=FALSE-------------
bias_rank <- biasDetect(list_batch_df = list_batch_df, 
    threshold = "rank", nSD_rank = 3)

## ----table 2, comment = NA, message=FALSE, warning=FALSE----------------------
head(bias_rank$subject$Table)

## ----figure 3, message=FALSE, warning=FALSE, fig.width= 10, fig.height=4------
# shape default = 16
bias_rank_shape <- biasDetect(list_batch_df = list_batch_df, 
    threshold = "rank", nSD_rank = 3, plot_point_shape = 2)

plot_grid(bias_rank$subject$Plot, bias_rank_shape$subject$Plot)

## ----both, comment = NA, message=FALSE, warning=FALSE-------------------------
bias_both <- biasDetect(list_batch_df = list_batch_df, threshold = "both",
    nSD_dev = 3, nSD_rank = 3)

## ----table 3, comment = NA, message=FALSE, warning=FALSE----------------------
head(bias_both$subject$Table)

## ----figure 4, message=FALSE, warning=FALSE, fig.width= 10, fig.height=8------
# color default = "YlOrRd"
bias_both_color <- biasDetect(list_batch_df = list_batch_df, 
    threshold = "both", nSD_dev = 3, nSD_rank = 3, plot_palette = "Greens")

plot_grid(bias_both$subject$Plot, bias_both_color$subject$Plot,nrow = 2)

## ----figure 5, message=FALSE, warning=FALSE, fig.width= 10, fig.height=8------
# text size default = 3
bias_both_color_text <- biasDetect(list_batch_df = list_batch_df, 
    threshold = "both", nSD_dev = 3, nSD_rank = 3, 
    plot_palette = c("Blues"), plot_text_size = 4)

plot_grid(bias_both$subject$Plot, bias_both_color_text$subject$Plot,nrow = 2)

## ----new svgs, comment = NA, message=FALSE, warning=FALSE---------------------
bias_both_df <- bias_both$subject$Table
svgs_filt <- setdiff(libd_nnsvgs$gene_id, bias_both_df$gene_id)
svgs_filt_spe <- libd_nnsvgs[libd_nnsvgs$gene_id %in% svgs_filt, ]
nrow(svgs_filt_spe)

## ----session info-------------------------------------------------------------
## Session info
sessionInfo()

