/*
 * LaTeD Version 1.1
 * (c) Gene Ressler 1993, 94, 97
 *   de8827@trotter.usma.edu
 *
 * LaTeD is a graphical editor for drawings in the LaTeX "picture" 
 * environment.  It runs under MSDOS or in a Windows DOS box.  The
 * distribution includes full sources, including LaTeX source for 
 * its documentation.
 *
 * No warranty of this software is expressed or implied by the author.
 *
 * Copy and use this program freely for any purpose except for sale
 * of the program (including the source code) itself.  That is, 
 * no one can copy this program for the purpose of providing it to 
 * another person in exchange for money or other compensation, even 
 * if this program is only part of the exchange.
 *
 * All copies of computer source code in this distribution, whether
 * copies in whole or in part, must have this notice attached.
 */

/* KEY.C -- Manage extended keycodes. */
#include <stdio.h>
#include <ctype.h>
#include "window.h"

/* Change character to it's alt form or -32768 if there is none. */
LOCAL(int) altify(int ch)
{

# include "key.h"

  static char alpha_alt_hi8[] = {
#   define AKEY(U,L,UC,LC) ((unsigned)(ALT_ ## U) >> 8),
#   define NKEY(U,L,UC,LC)
#   include "key.h"
#   undef AKEY
#   undef NKEY
  };

  static char num_alt_hi8[] = {
#   define NKEY(U,L,UC,LC)  ((unsigned)(ALT_ ## U) >> 8),
#   define AKEY(U,L,UC,LC)
#   include "key.h"
#   undef NKEY
#   undef AKEY
  };

  if (ch == ' ')
    return ALT_SPACE;

  return ((ch >= 'A' && ch <= 'Z') ? alpha_alt_hi8[ch - 'A'] :
	  (ch >= 'a' && ch <= 'z') ? alpha_alt_hi8[ch - 'a'] :
	  (ch >= '0' && ch <= '9') ? num_alt_hi8[ch - '0']   : '\x80') << 8;
}

/* Values for Category field of key feature table. */
enum { 
  FK,  CTRL_FK,  SHIFT_FK,  ALT_FK,  SPECIAL, CTRL_SPECIAL, ALT_CH, ALT_NUM 
};

/* Strings for above categories. */
static char near *cat_str_tbl[] = {
  "F", "Ctrl-F", "Shift-F", "Alt-F", "",      "Ctrl-",      "Alt-", "Alt-" 
};

/* Features of scan codes.:
     Hi 3 bits are category.
     Lo 5 bits are index number interpreted
	  differently for each category. */
static unsigned char key_feature_tbl[] = {
  0xd0  /*ALT_Q*/,
  0xd6  /*ALT_W*/,
  0xc4  /*ALT_E*/,
  0xd1  /*ALT_R*/,
  0xd3  /*ALT_T*/,
  0xd8  /*ALT_Y*/,
  0xd4  /*ALT_U*/,
  0xc8  /*ALT_I*/,
  0xce  /*ALT_O*/,
  0xcf  /*ALT_P*/,
  0,0,0,0,
  0xc0  /*ALT_A*/,
  0xd2  /*ALT_S*/,
  0xc3  /*ALT_D*/,
  0xc5  /*ALT_F*/,
  0xc6  /*ALT_G*/,
  0xc7  /*ALT_H*/,
  0xc9  /*ALT_J*/,
  0xca  /*ALT_K*/,
  0xcb  /*ALT_L*/,
  0,0,0,0,0,
  0xd9  /*ALT_Z*/,
  0xd7  /*ALT_X*/,
  0xc2  /*ALT_C*/,
  0xd5  /*ALT_V*/,
  0xc1  /*ALT_B*/,
  0xcd  /*ALT_N*/,
  0xcc  /*ALT_M*/,
  0,0,0,0,0,0,0,0,
  0x01  /*F1*/,
  0x02  /*F2*/,
  0x03  /*F3*/,
  0x04  /*F4*/,
  0x05  /*F5*/,
  0x06  /*F6*/,
  0x07  /*F7*/,
  0x08  /*F8*/,
  0x09  /*F9*/,
  0x0a  /*F10*/,
  0,0,
  0x84  /*HOME*/,
  0x80  /*UP_ARROW*/,
  0x86  /*PGUP*/,
  0,
  0x82  /*LEFT_ARROW*/,
  0,
  0x83  /*RIGHT_ARROW*/,
  0,
  0x85  /*END*/,
  0x81  /*DOWN_ARROW*/,
  0x87  /*PGDN*/,
  0x88  /*INS*/,
  0x89  /*DEL*/,
  0x41  /*SHIFT_F1*/,
  0x42  /*SHIFT_F2*/,
  0x43  /*SHIFT_F3*/,
  0x44  /*SHIFT_F4*/,
  0x45  /*SHIFT_F5*/,
  0x46  /*SHIFT_F6*/,
  0x47  /*SHIFT_F7*/,
  0x48  /*SHIFT_F8*/,
  0x49  /*SHIFT_F9*/,
  0x4a  /*SHIFT_F10*/,
  0x21  /*CTRL_F1*/,
  0x22  /*CTRL_F2*/,
  0x23  /*CTRL_F3*/,
  0x24  /*CTRL_F4*/,
  0x25  /*CTRL_F5*/,
  0x26  /*CTRL_F6*/,
  0x27  /*CTRL_F7*/,
  0x28  /*CTRL_F8*/,
  0x29  /*CTRL_F9*/,
  0x2a  /*CTRL_F10*/,
  0x61  /*ALT_F1*/,
  0x62  /*ALT_F2*/,
  0x63  /*ALT_F3*/,
  0x64  /*ALT_F4*/,
  0x65  /*ALT_F5*/,
  0x66  /*ALT_F6*/,
  0x67  /*ALT_F7*/,
  0x68  /*ALT_F8*/,
  0x69  /*ALT_F9*/,
  0x6a  /*ALT_F10*/,
  0,
  0xa2  /*CTRL_LEFT_ARROW*/,
  0xa3  /*CTRL_RIGHT_ARROW*/,
  0xa5  /*CTRL_END*/,
  0xa7  /*CTRL_PGDN*/,
  0xa4  /*CTRL_HOME*/,
  0xe1  /*ALT_1*/,
  0xe2  /*ALT_2*/,
  0xe3  /*ALT_3*/,
  0xe4  /*ALT_4*/,
  0xe5  /*ALT_5*/,
  0xe6  /*ALT_6*/,
  0xe7  /*ALT_7*/,
  0xe8  /*ALT_8*/,
  0xe9  /*ALT_9*/,
  0xe0  /*ALT_0*/,
  0,0,
  0xa6  /*CTRL_PGUP*/,
};

/* Indices of specials index this table. */
static char near *special_tbl[] = {
  "\x18","\x19","\x1b","\x1a","Home","End","PgUp","PgDn","Ins","Del",
};

/* Convert extended key code to meaningful string. */
char *key2str(int key, char *str)
{
  unsigned hi8, code, cat, num;
  char *cat_str;

  if (key == SUBMENU_MARK) {
    str[0] = '\x10';
    str[1] = '\0';
  }
  else if (key & 0xff00) {

    hi8 = (unsigned)key >> 8;
    code = key_feature_tbl[hi8 - 0x10];
    cat  = code >> 5;
    cat_str = cat_str_tbl[cat];
    num  = code &  31;

    if (cat <= ALT_FK)
      /* All manner of function keys with meta-keys. */
      sprintf(str, "%s%d", cat_str, num);
    else if (cat <= CTRL_SPECIAL)
      /* Special and control-special. */
      sprintf(str, "%s%s", cat_str, (char*)special_tbl[num]);
    else 
      /* Alt char or alt num */
      sprintf(str, "%s%c", cat_str, num + (cat == ALT_CH ? 'A' : '0'));
  }
  else if (key == 0)
    str[0] = '\0';
  else if (key == 27)
    sprintf(str, "Esc");
  else if (iscntrl(key))
    sprintf(str, "Ctrl-%c", key + 'A' - 1);
  else if (key == 0xff)
    sprintf(str, "Alt-Space");
  else {
    str[0] = key;
    str[1] = '\0';
  }
  return str;
}

/* tolower() masks hi bits! */
LOCAL(int) lower_case(int ch)
{
  return ('A' <= ch && ch <= 'Z') ? ch + ('a' - 'A') : ch;
}

/* Return non-0 iff key is a case-insensitive 
   match for ch or else key is the ALT form of ch. */
int hot_p(int key, int ch)
{
  return lower_case(key) == lower_case(ch) || key == altify(ch);
}

/* Return non-0 iff key is the hot char for a hot text. 
   A hot char is either the underscored text or it's ALT equivalent. */
int key_hot_p(int key, HOT_TEXT text)
{
  int c = text->hot_char;
  return c != NO_HOT_CHAR && hot_p(key, c == HOT_SPACE ? ' ' : text->str[c]);
}

/* Return non-0 iff key is a case-insensitive match 
   for the underscored character of a hot text. */
int key_match_p(int key, HOT_TEXT text)
{
  int c = text->hot_char;
  switch (c) {
    case NO_HOT_CHAR:	return 0;
    case HOT_SPACE:	return (key == ' ');
    default:		return tolower(key) == tolower(text->str[c]);
  }
}

#ifdef TEST

int main(void)
{
  int ch;
  char buf[16];

  do {
    ch = bioskey(0);
    if (ch & 0xff) ch &= 0xff;
    cprintf("%x:%s\r\n", ch, key2str(ch, buf));
  } while (ch != 27);
}

#endif
