# frozen_string_literal: true

require 'spec_helper'

RSpec.describe Gitlab::BackgroundMigration::BackfillHasMergeRequestOfVulnerabilityReads, schema: 20230907155247, feature_category: :database do # rubocop:disable Layout/LineLength
  let(:namespaces) { table(:namespaces) }
  let(:projects) { table(:projects) }
  let(:users) { table(:users) }
  let(:scanners) { table(:vulnerability_scanners) }
  let(:vulnerabilities) { table(:vulnerabilities) }
  let(:vulnerability_reads) { table(:vulnerability_reads) }
  let(:merge_requests) { table(:merge_requests) }
  let(:merge_request_links) { table(:vulnerability_merge_request_links) }

  let(:namespace) { namespaces.create!(name: 'user', path: 'user') }
  let(:project) { projects.create!(namespace_id: namespace.id, project_namespace_id: namespace.id) }
  let(:user) { users.create!(username: 'john_doe', email: 'johndoe@gitlab.com', projects_limit: 10) }
  let(:scanner) { scanners.create!(project_id: project.id, external_id: 'external_id', name: 'Test Scanner') }

  let(:vulnerability) do
    vulnerabilities.create!(
      project_id: project.id,
      author_id: user.id,
      title: 'test',
      severity: 1,
      confidence: 1,
      report_type: 1
    )
  end

  let(:merge_request) do
    merge_requests.create!(
      target_project_id: project.id,
      source_branch: "other",
      target_branch: "main",
      author_id: user.id,
      title: 'Feedback Merge Request'
    )
  end

  let!(:vulnerability_read) do
    vulnerability_reads.create!(
      project_id: project.id,
      vulnerability_id: vulnerability.id,
      scanner_id: scanner.id,
      severity: 1,
      report_type: 1,
      state: 1,
      uuid: SecureRandom.uuid
    )
  end

  let!(:merge_request_link) do
    merge_request_links.create!(
      vulnerability_id: vulnerability.id, merge_request_id: merge_request.id)
  end

  subject(:perform_migration) do
    described_class.new(
      start_id: vulnerability_reads.first.vulnerability_id,
      end_id: vulnerability_reads.last.vulnerability_id,
      batch_table: :vulnerability_reads,
      batch_column: :vulnerability_id,
      sub_batch_size: vulnerability_reads.count,
      pause_ms: 0,
      connection: ActiveRecord::Base.connection
    ).perform
  end

  before do
    # Unset since the trigger already sets during merge_request_link creation.
    vulnerability_reads.update_all(has_merge_request: false)
  end

  it 'sets the has_merge_request of existing record' do
    expect { perform_migration }.to change { vulnerability_read.reload.has_merge_request }.from(false).to(true)
  end

  it 'does not modify has_merge_request of other vulnerabilities which do not have merge request' do
    vulnerability_2 = vulnerabilities.create!(
      project_id: project.id,
      author_id: user.id,
      title: 'test 2',
      severity: 1,
      confidence: 1,
      report_type: 1
    )

    vulnerability_read_2 = vulnerability_reads.create!(
      project_id: project.id,
      vulnerability_id: vulnerability_2.id,
      scanner_id: scanner.id,
      severity: 1,
      report_type: 1,
      state: 1,
      uuid: SecureRandom.uuid
    )

    expect { perform_migration }.not_to change { vulnerability_read_2.reload.has_merge_request }.from(false)
  end
end
