# frozen_string_literal: true

require 'spec_helper'

require_migration!

RSpec.describe AddHasMergeRequestOnVulnerabilityReadsTrigger, feature_category: :vulnerability_management do
  let(:migration) { described_class.new }
  let(:vulnerability_reads) { table(:vulnerability_reads) }
  let(:merge_requests) { table(:merge_requests) }
  let(:merge_request_links) { table(:vulnerability_merge_request_links) }
  let(:vulnerabilities) { table(:vulnerabilities) }
  let(:vulnerabilities_findings) { table(:vulnerability_occurrences) }

  let(:namespace) { table(:namespaces).create!(name: 'user', path: 'user') }
  let(:user) { table(:users).create!(id: 13, email: 'author@example.com', username: 'author', projects_limit: 10) }
  let(:project) { table(:projects).create!(id: 123, namespace_id: namespace.id, project_namespace_id: namespace.id) }

  let(:scanner) do
    table(:vulnerability_scanners).create!(
      project_id: project.id, external_id: 'test 1', name: 'test scanner 1')
  end

  let(:merge_request) do
    create_merge_request!(
      target_project_id: project.id,
      source_branch: "other",
      target_branch: "main",
      author_id: user.id,
      title: 'Feedback Merge Request 1'
    )
  end

  let(:vulnerability) do
    create_vulnerability!(
      project_id: project.id,
      author_id: user.id
    )
  end

  let(:identifier) do
    table(:vulnerability_identifiers).create!(
      project_id: project.id,
      external_type: 'uuid-v5',
      external_id: 'uuid-v5',
      fingerprint: '7e394d1b1eb461a7406d7b1e08f057a1cf11287a',
      name: 'Identifier for UUIDv5')
  end

  let(:vulnerability_read) { vulnerability_reads.first }

  before do
    create_finding!(
      vulnerability_id: vulnerability.id,
      project_id: project.id,
      scanner_id: scanner.id,
      primary_identifier_id: identifier.id
    )

    vulnerability_read.reload
  end

  describe '#up' do
    before do
      migrate!
    end

    describe 'INSERT trigger' do
      it 'updates has_merge_request in vulnerability_reads' do
        expect do
          merge_request_links.create!(
            vulnerability_id: vulnerability.id, merge_request_id: merge_request.id)
        end.to change { vulnerability_read.reload.has_merge_request }.from(false).to(true)
      end
    end

    describe 'DELETE trigger' do
      let(:merge_request2) do
        create_merge_request!(
          target_project_id: project.id,
          source_branch: "other_2",
          target_branch: "main",
          author_id: user.id,
          title: 'Feedback Merge Request 2'
        )
      end

      it 'does not change has_merge_request when there exists another merge_request' do
        merge_request_link1 = merge_request_links.create!(
          vulnerability_id: vulnerability.id, merge_request_id: merge_request.id)

        merge_request_links.create!(
          vulnerability_id: vulnerability.id, merge_request_id: merge_request2.id)

        expect do
          merge_request_link1.delete
        end.not_to change { vulnerability_read.reload.has_merge_request }
      end

      it 'unsets has_merge_request when all merge_requests are deleted' do
        merge_request_link1 = merge_request_links.create!(
          vulnerability_id: vulnerability.id, merge_request_id: merge_request.id)

        merge_request_link2 = merge_request_links.create!(
          vulnerability_id: vulnerability.id, merge_request_id: merge_request2.id)

        expect do
          merge_request_link1.delete
          merge_request_link2.delete
        end.to change { vulnerability_read.reload.has_merge_request }.from(true).to(false)
      end
    end
  end

  describe '#down' do
    before do
      migration.up
      migration.down
    end

    it 'drops the trigger' do
      expect do
        merge_request_links.create!(
          vulnerability_id: vulnerability.id, merge_request_id: merge_request.id)
      end.not_to change { vulnerability_read.reload.has_merge_request }
    end
  end

  private

  def create_vulnerability!(project_id:, author_id:, title: 'test', severity: 7, confidence: 7, report_type: 0)
    vulnerabilities.create!(
      project_id: project_id,
      author_id: author_id,
      title: title,
      severity: severity,
      confidence: confidence,
      report_type: report_type
    )
  end

  def create_merge_request!(overrides = {})
    attrs = {
      target_project_id: project.id,
      source_branch: "other",
      target_branch: "main",
      author_id: user.id,
      title: 'Feedback Merge Request'
    }.merge(overrides)

    merge_requests.create!(attrs)
  end

  # rubocop:disable Metrics/ParameterLists
  def create_finding!(
    vulnerability_id:, project_id:, scanner_id:, primary_identifier_id:,
    name: "test", severity: 7, confidence: 7, report_type: 0,
    project_fingerprint: '123qweasdzxc', location: { "image" => "alpine:3.4" }, location_fingerprint: 'test',
    metadata_version: 'test', raw_metadata: 'test', uuid: SecureRandom.uuid)
    vulnerabilities_findings.create!(
      vulnerability_id: vulnerability_id,
      project_id: project_id,
      name: name,
      severity: severity,
      confidence: confidence,
      report_type: report_type,
      project_fingerprint: project_fingerprint,
      scanner_id: scanner_id,
      primary_identifier_id: primary_identifier_id,
      location: location,
      location_fingerprint: location_fingerprint,
      metadata_version: metadata_version,
      raw_metadata: raw_metadata,
      uuid: uuid
    )
  end
  # rubocop:enable Metrics/ParameterLists
end
