/*****************************************************************************
 * Copyright (c) 2015 CEA LIST.
 *
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  Benoit Maggi (CEA LIST) benoit.maggi@cea.fr - Initial API and implementation
 *
 *****************************************************************************/
package org.eclipse.papyrus.sysml14.deprecatedelements;

import org.eclipse.emf.common.util.EList;
import org.eclipse.papyrus.sysml14.deprecatedelements.internal.impl.FlowPortImpl;
import org.eclipse.papyrus.sysml14.portsandflows.FlowDirection;
import org.eclipse.uml2.uml.Image;
import org.eclipse.uml2.uml.Interface;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.util.UMLUtil;

/**
 * this class has been implemented in order to calculate derived properties of "FlowPort" stereotype see Requirement DerivedProperties Req0010
 *
 */
public class FlowPortCustomImpl extends FlowPortImpl implements FlowPort {

	
	
	/**
	 * Icons for displaying port orientation, 
	 */
	private static EList<Image> icons;


	/**
	 * @return the icons for port orientation
	 */
	public static EList<Image> getIcons(FlowPort flowPort) {
		if (icons == null){// for PERFORMANCE we call UMLUtil.getStereotype only once
			icons = UMLUtil.getStereotype(flowPort).getIcons();
		}
		return icons;
	}


	/**
	 * C.3.2.2 FlowPort : /isAtomic : Boolean (derived) This is a derived attribute (derived from the flow port’s type).
	 * For a flow port typed by a flow specification the value of this attribute is False, otherwise the value is True.
	 * 
	 * @see org.eclipse.papyrus.sysml14.deprecatedelements.internal.impl.FlowPortImpl#isAtomic()
	 *
	 * @return true if atomic
	 */
	@Override
	public boolean isAtomic() {
		// The FlowPort is Atomic if it is not typed by a FlowSpecification
		boolean isAtomic = true;
		if (getBase_Port() != null) {
			// Find FlowPort type
			Type type = getBase_Port().getType();
			if ((type != null) && (type instanceof Interface)) {
				FlowSpecification flowSpec = UMLUtil.getStereotypeApplication(type, FlowSpecification.class);
				if (flowSpec != null) {
					isAtomic = false;
				}
			}
		}
		return isAtomic;
	}


	/**
	 * Images registered in Profile are :
	 * 0 - FlowPort IN
	 * 1 - FlowPort OUT
	 * 2 - FlowPort INOUT
	 * 3 - FlowPort NA (Non Atomic)
	 * 4 (Other) - FlowPort (default, similar to INOUT)
	 * @see org.eclipse.papyrus.sysml14.deprecatedelements.internal.impl.FlowPortImpl#getIcon()
	 *
	 * @return the icon depending on the orientation of the FlowPort
	 */
	@Override
	public Image getIcon() {
		Image image = null;
		if (getBase_Port() != null) {
			EList<Image> iconList = getIcons(this);
			switch (getDirection()) {
			case IN:
				image = iconList.get(FlowDirection.IN.getValue());
				break;
			case OUT:
				image = iconList.get(FlowDirection.OUT.getValue());
				break;
			case INOUT:
				image = iconList.get(FlowDirection.INOUT.getValue());
				break;
			default:
				image = iconList.get(FlowDirection.INOUT.getValue());
				break;
			}
		}
		return image;
	}

}
