/**
 * <copyright>
 * 
 * Copyright (c) 2016 itemis and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     itemis - Initial API and implementation
 *     itemis - [503063] Provide launching support for Sphinx Workflows
 * 
 * </copyright>
 */
package org.eclipse.sphinx.emf.util;

import com.google.common.base.Objects;
import org.eclipse.core.runtime.Assert;
import org.eclipse.emf.common.util.URI;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * Useful API extensions for {@link URI}.
 */
@SuppressWarnings("all")
public class URIExtensions {
  public static final String URI_SEGMENT_SEPARATOR = "/";
  
  public static final String URI_AUTHORITY_SEPARATOR = (URIExtensions.URI_SEGMENT_SEPARATOR + URIExtensions.URI_SEGMENT_SEPARATOR);
  
  public static final String URI_QUERY_SEPARATOR = "?";
  
  public static final String URI_FRAGMENT_SEPARATOR = "#";
  
  /**
   * Indicates whether the given URI is an actual URI. An actual URI is a URI that "looks like" a URI and can be clearly distinguished from a simple string.
   * This is generally the case for all URIs that have at least one of the following components: a {@link URI#scheme() scheme}, a {@link URI#hasDevice() device},
   * an {@link URI#hasAuthority() authority}, a {@link URI#hasPath() path} with more than 1 {@link URI#segments() segment},
   * a {@link URI#hasTrailingPathSeparator() trailing separator}, a {@link URI#hasQuery() query}, or a {@link URI#hasFragment() fragment}.
   * 
   * @param uri The URI to be examined.
   * @return <code>true</code> if the given URI is an actual URI, <code>false</code> otherwise.
   */
  public static boolean isActual(final URI uri) {
    Assert.isNotNull(uri);
    return (((((((!StringExtensions.isNullOrEmpty(uri.scheme())) || 
      uri.hasDevice()) || 
      uri.hasAuthority()) || (uri.hasPath() && (uri.segmentCount() > 1))) || 
      uri.hasTrailingPathSeparator()) || 
      uri.hasQuery()) || 
      uri.hasFragment());
  }
  
  /**
   * Retrieves the {@link URI#fragment() fragment} of the given {@link URI} and converted into a URI of its own.
   * 
   * @param uri The URI whose fragment is to be returned as URI.
   * @return The given URI's fragment as URI, or <code>null</code> if given URI has no fragment.
   */
  public static URI getFragment(final URI uri) {
    Assert.isNotNull(uri);
    boolean _hasFragment = uri.hasFragment();
    if (_hasFragment) {
      String fragment = uri.fragment();
      boolean _startsWith = fragment.startsWith(((URIExtensions.URI_SEGMENT_SEPARATOR + "") + URIExtensions.URI_SEGMENT_SEPARATOR));
      if (_startsWith) {
        final int queryIdx = fragment.indexOf(URIExtensions.URI_QUERY_SEPARATOR);
        String _xifexpression = null;
        if ((queryIdx != (-1))) {
          _xifexpression = fragment.substring(0, queryIdx);
        } else {
          _xifexpression = fragment;
        }
        final String path = _xifexpression;
        String _xifexpression_1 = null;
        if (((queryIdx != (-1)) && (fragment.length() > (queryIdx + 1)))) {
          _xifexpression_1 = fragment.substring((queryIdx + 1), fragment.length());
        } else {
          _xifexpression_1 = null;
        }
        final String query = _xifexpression_1;
        return URI.createHierarchicalURI(path.split(URIExtensions.URI_SEGMENT_SEPARATOR), query, null);
      } else {
        return URI.createURI(fragment, true);
      }
    }
    return null;
  }
  
  /**
   * Returns a copy of the given {@link URI} where the {@link URI#fragment() fragment} has been substituted
   * with the string value of provided fragment URI. The returned URI will have no fragment
   * if the provided fragment URI is <code>null</code>.
   * 
   * @param uri The URI to be manipulated.
   * @param fragment A URI specifying the fragment that the given URI's fragment is to be substituted with; may be <code>null</code>.
   * @return A URI formed by replacing the fragment of the given URI with the string value of provided fragment URI
   *         in case that the latter is not <code>null</code>; a copy of the given URI without its fragment otherwise.
   */
  public static URI substituteFragment(final URI uri, final URI fragment) {
    Assert.isNotNull(uri);
    URI result = uri.trimFragment();
    boolean _notEquals = (!Objects.equal(fragment, null));
    if (_notEquals) {
      result = result.appendFragment(fragment.toString());
    }
    return result;
  }
  
  /**
   * Returns a copy of the given {@link URI} that has been converted into a {@link URI#isPrefix() prefix}
   * and can be used in {@link URI#replacePrefix(URI, URI)} operations.
   * 
   * @param uri The URI to be returned as prefix URI.
   * @return The given URI as prefix URI.
   * @see URI#isPrefix()
   * @see URI#replacePrefix(URI, URI)
   */
  public static URI asPrefix(final URI uri) {
    Assert.isNotNull(uri);
    URI result = uri.trimQuery();
    result = result.trimFragment();
    boolean _hasTrailingPathSeparator = result.hasTrailingPathSeparator();
    boolean _not = (!_hasTrailingPathSeparator);
    if (_not) {
      result = result.appendSegment("");
    }
    return result;
  }
  
  /**
   * If given old last segment matches the {@link URI#lastSegment() last segment} of given {@link URI},
   * this returns a copy of the given URI where the last segment has been replaced by the provided new last segment.
   * 
   * @param uri The URI to be manipulated.
   * @param oldLastSegment The expected current last segment of the URI's path.
   * @param newLastSegment The intended future last segment of the URI's path.
   * @return A URI formed by replacing the last segment of the given URI with the provided new last segment
   *         in case that the given URI's existing last segment matches the provided old last segment,
   *         or <code>null</code> otherwise.
   */
  public static URI replaceLastSegment(final URI uri, final String oldLastSegment, final String newLastSegment) {
    Assert.isNotNull(uri);
    int _segmentCount = uri.segmentCount();
    boolean _greaterThan = (_segmentCount > 0);
    Assert.isLegal(_greaterThan);
    boolean _equals = uri.lastSegment().equals(oldLastSegment);
    if (_equals) {
      return uri.trimSegments(1).appendSegment(newLastSegment);
    }
    return null;
  }
  
  /**
   * If given old base URI matches the given {@link URI}, this returns a copy of the given URI
   * where all components included in the old base URI have been replaced by the provided new base URI.
   * The old and new base URI may not only include a {@link URI#path() path} but also
   * a {@link URI#query() query} and/or a {@link URI#fragment() fragment}.
   * 
   * @param uri The URI to be manipulated.
   * @param oldBaseURI The URI's expected current base URI.
   * @param newBaseURI The URI's intended future base URI.
   * @return A URI formed by replacing the provided old base URI in the given URI with the provided new base URI
   *         in case that the old base URI matches the given URI, or <code>null</code> otherwise.
   */
  public static URI replaceBaseURI(final URI uri, final URI oldBaseURI, final URI newBaseURI) {
    Assert.isNotNull(uri);
    Assert.isNotNull(oldBaseURI);
    Assert.isNotNull(newBaseURI);
    if (((uri.hasFragment() && oldBaseURI.hasFragment()) && newBaseURI.hasFragment())) {
      boolean _equals = uri.trimFragment().equals(oldBaseURI.trimFragment());
      if (_equals) {
        URI fragment = URIExtensions.getFragment(uri).replacePrefix(URIExtensions.asPrefix(URIExtensions.getFragment(oldBaseURI)), 
          URIExtensions.asPrefix(URIExtensions.getFragment(newBaseURI)));
        boolean _notEquals = (!Objects.equal(fragment, null));
        if (_notEquals) {
          return URIExtensions.substituteFragment(newBaseURI, fragment);
        }
      }
    } else {
      return uri.replacePrefix(URIExtensions.asPrefix(oldBaseURI), URIExtensions.asPrefix(newBaseURI));
    }
    return null;
  }
  
  /**
   * If given {@link URI} has a {@link URI#isHierarchical() hierarchical} {@link URI#fragment() fragment}
   * and given old last fragment segment matches the {@link URI#lastSegment() last segment} of that fragment,
   * this returns a copy of the given URI where the last segment of its fragment
   * has been replaced by the provided new last fragment segment.
   * 
   * @param uri The URI to be manipulated.
   * @param oldLastSegment The expected current last segment of the URI's fragment.
   * @param newLastSegment The intended future last segment of the URI's fragment.
   * @return A URI formed by replacing the last segment in the fragment of the given URI
   *         with the provided new last fragment segment in case that the existing last segment
   *         in the fragment of the given URI matches the provided old last fragment segment,
   *         or <code>null</code> otherwise.
   */
  public static URI replaceLastFragmentSegment(final URI uri, final String oldLastFragmentSegment, final String newLastFragmentSegment) {
    Assert.isNotNull(uri);
    URI fragment = URIExtensions.getFragment(uri);
    boolean _notEquals = (!Objects.equal(fragment, null));
    if (_notEquals) {
      fragment = URIExtensions.replaceLastSegment(fragment, oldLastFragmentSegment, newLastFragmentSegment);
      boolean _notEquals_1 = (!Objects.equal(fragment, null));
      if (_notEquals_1) {
        return URIExtensions.substituteFragment(uri, fragment);
      }
    }
    return null;
  }
}
