# Commands covered:  scan
#
# This file contains a collection of tests for one or more of the Tcl
# built-in commands.  Sourcing this file into Tcl runs the tests and
# generates output for errors.  No output means no errors were found.
#
# Copyright (c) 1991-1994 The Regents of the University of California.
# Copyright (c) 1994-1997 Sun Microsystems, Inc.
# Copyright (c) 1998-1999 by Scriptics Corporation.
#
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#
# RCS: @(#) $Id: scan.tcl,v 1.1 2006/12/29 09:02:53 fourdman Exp $

if {[lsearch [namespace children] ::tcltest] == -1} {
    package require tcltest 2
    namespace import -force ::tcltest::*
}

::tcltest::testConstraint 64bitInts [expr {0x80000000 > 0}]

test scan-1.1 {BuildCharSet, CharInSet} {
    list [scan foo {%[^o]} x] $x
} {1 f}
test scan-1.2 {BuildCharSet, CharInSet} {
    list [scan \]foo {%[]f]} x] $x
} {1 \]f}
test scan-1.3 {BuildCharSet, CharInSet} {
    list [scan abc-def {%[a-c]} x] $x
} {1 abc}
test scan-1.4 {BuildCharSet, CharInSet} {
    list [scan abc-def {%[a-c]} x] $x
} {1 abc}
test scan-1.5 {BuildCharSet, CharInSet} {
    list [scan -abc-def {%[-ac]} x] $x
} {1 -a}
test scan-1.6 {BuildCharSet, CharInSet} {
    list [scan -abc-def {%[ac-]} x] $x
} {1 -a}
test scan-1.7 {BuildCharSet, CharInSet} {
    list [scan abc-def {%[c-a]} x] $x
} {1 abc}
test scan-1.8 {BuildCharSet, CharInSet} {
    list [scan def-abc {%[^c-a]} x] $x
} {1 def-}
test scan-1.9 {BuildCharSet, CharInSet no match} {
    catch {unset x}
    list [scan {= f} {= %[TF]} x] [info exists x]
} {0 0}

test scan-2.1 {ReleaseCharSet} {
    list [scan abcde {%[abc]} x] $x
} {1 abc}
test scan-2.2 {ReleaseCharSet} {
    list [scan abcde {%[a-c]} x] $x
} {1 abc}

test scan-3.1 {ValidateFormat} {
    list [catch {scan {} {%d%1$d} x} msg] $msg
} {1 {cannot mix "%" and "%n$" conversion specifiers}}
test scan-3.2 {ValidateFormat} {
    list [catch {scan {} {%d%1$d} x} msg] $msg
} {1 {cannot mix "%" and "%n$" conversion specifiers}}
test scan-3.3 {ValidateFormat} {
    list [catch {scan {} {%2$d%d} x} msg] $msg
} {1 {"%n$" argument index out of range}}
test scan-3.4 {ValidateFormat} {
    # degenerate case, before changed from 8.2 to 8.3
    list [catch {scan {} %d} msg] $msg
} {0 {}}
test scan-3.5 {ValidateFormat} {
    list [catch {scan {} {%10c} a} msg] $msg
} {1 {field width may not be specified in %c conversion}}
test scan-3.6 {ValidateFormat} {
    list [catch {scan {} {%*1$d} a} msg] $msg
} {1 {bad scan conversion character "$"}}
test scan-3.7 {ValidateFormat} {
    list [catch {scan {} {%1$d%1$d} a} msg] $msg
} {1 {variable is assigned by multiple "%n$" conversion specifiers}}
test scan-3.8 {ValidateFormat} {
    list [catch {scan {} a x} msg] $msg
} {1 {variable is not assigned by any conversion specifiers}}
test scan-3.9 {ValidateFormat} {
    list [catch {scan {} {%2$s} x y} msg] $msg
} {1 {variable is not assigned by any conversion specifiers}}
test scan-3.10 {ValidateFormat} {
    list [catch {scan {} {%[a} x} msg] $msg
} {1 {unmatched [ in format string}}
test scan-3.11 {ValidateFormat} {
    list [catch {scan {} {%[^a} x} msg] $msg
} {1 {unmatched [ in format string}}
test scan-3.12 {ValidateFormat} {
    list [catch {scan {} {%[]a} x} msg] $msg
} {1 {unmatched [ in format string}}
test scan-3.13 {ValidateFormat} {
    list [catch {scan {} {%[^]a} x} msg] $msg
} {1 {unmatched [ in format string}}

test scan-4.1 {Tcl_ScanObjCmd, argument checks} {
    list [catch {scan} msg] $msg
} {1 {wrong # args: should be "scan string format ?varName varName ...?"}}
test scan-4.2 {Tcl_ScanObjCmd, argument checks} {
    list [catch {scan string} msg] $msg
} {1 {wrong # args: should be "scan string format ?varName varName ...?"}}
test scan-4.3 {Tcl_ScanObjCmd, argument checks} {
    # degenerate case, before changed from 8.2 to 8.3
    list [catch {scan string format} msg] $msg
} {0 {}}
test scan-4.4 {Tcl_ScanObjCmd, whitespace} {
    list [scan {   abc   def   } {%s%s} x y] $x $y
} {2 abc def}
test scan-4.5 {Tcl_ScanObjCmd, whitespace} {
    list [scan {   abc   def   } { %s %s } x y] $x $y
} {2 abc def}
test scan-4.6 {Tcl_ScanObjCmd, whitespace} {
    list [scan {   abc   def   } { %s %s } x y] $x $y
} {2 abc def}
test scan-4.7 {Tcl_ScanObjCmd, literals} {
    # degenerate case, before changed from 8.2 to 8.3
    scan {   abc   def   } { abc def }
} {}
test scan-4.8 {Tcl_ScanObjCmd, literals} {
    set x {}
    list [scan {   abcg} { abc def %1s} x] $x
} {0 {}}
test scan-4.9 {Tcl_ScanObjCmd, literals} {
    list [scan {   abc%defghi} { abc %% def%n } x] $x
} {1 10}
test scan-4.10 {Tcl_ScanObjCmd, assignment suppression} {
    list [scan {   abc   def   } { %*c%s def } x] $x
} {1 bc}
test scan-4.11 {Tcl_ScanObjCmd, XPG3-style} {
    list [scan {   abc   def   } {%2$s %1$s} x y] $x $y
} {2 def abc}
test scan-4.12 {Tcl_ScanObjCmd, width specifiers} {
    list [scan {abc123456789012} {%3s%3d%3f%3[0-9]%s} a b c d e] $a $b $c $d $e
} {5 abc 123 456.0 789 012}
test scan-4.13 {Tcl_ScanObjCmd, width specifiers} {
    list [scan {abc123456789012} {%3s%3d%3f%3[0-9]%s} a b c d e] $a $b $c $d $e
} {5 abc 123 456.0 789 012}
test scan-4.14 {Tcl_ScanObjCmd, underflow} {
    set x {}
    list [scan {a} {a%d} x] $x
} {-1 {}}
test scan-4.15 {Tcl_ScanObjCmd, underflow} {
    set x {}
    list [scan {} {a%d} x] $x
} {-1 {}}
test scan-4.16 {Tcl_ScanObjCmd, underflow} {
    set x {}
    list [scan {ab} {a%d} x] $x
} {0 {}}
test scan-4.17 {Tcl_ScanObjCmd, underflow} {
    set x {}
    list [scan {a   } {a%d} x] $x
} {-1 {}}
test scan-4.18 {Tcl_ScanObjCmd, skipping whitespace} {
    list [scan {  b} {%c%s} x y] $x $y
} {2 32 b}
test scan-4.19 {Tcl_ScanObjCmd, skipping whitespace} {
    list [scan {  b} {%[^b]%s} x y] $x $y
} {2 {  } b}
test scan-4.20 {Tcl_ScanObjCmd, string scanning} {
    list [scan {abc def} {%s} x] $x
} {1 abc}
test scan-4.21 {Tcl_ScanObjCmd, string scanning} {
    list [scan {abc def} {%0s} x] $x
} {1 abc}
test scan-4.22 {Tcl_ScanObjCmd, string scanning} {
    list [scan {abc def} {%2s} x] $x
} {1 ab}
test scan-4.23 {Tcl_ScanObjCmd, string scanning} {
    list [scan {abc def} {%*s%n} x] $x
} {1 3}
test scan-4.24 {Tcl_ScanObjCmd, charset scanning} {
    list [scan {abcdef} {%[a-c]} x] $x
} {1 abc}
test scan-4.25 {Tcl_ScanObjCmd, charset scanning} {
    list [scan {abcdef} {%0[a-c]} x] $x
} {1 abc}
test scan-4.26 {Tcl_ScanObjCmd, charset scanning} {
    list [scan {abcdef} {%2[a-c]} x] $x
} {1 ab}
test scan-4.27 {Tcl_ScanObjCmd, charset scanning} {
    list [scan {abcdef} {%*[a-c]%n} x] $x
} {1 3}
test scan-4.28 {Tcl_ScanObjCmd, character scanning} {
    list [scan {abcdef} {%c} x] $x
} {1 97}
test scan-4.29 {Tcl_ScanObjCmd, character scanning} {
    list [scan {abcdef} {%*c%n} x] $x
} {1 1}

test scan-4.30 {Tcl_ScanObjCmd, base-10 integer scanning} {
    set x {}
    list [scan {1234567890a} {%3d} x] $x
} {1 123}
test scan-4.31 {Tcl_ScanObjCmd, base-10 integer scanning} {
    set x {}
    list [scan {1234567890a} {%d} x] $x
} {1 1234567890}
test scan-4.32 {Tcl_ScanObjCmd, base-10 integer scanning} {
    set x {}
    list [scan {01234567890a} {%d} x] $x
} {1 1234567890}
test scan-4.33 {Tcl_ScanObjCmd, base-10 integer scanning} {
    set x {}
    list [scan {+01234} {%d} x] $x
} {1 1234}
test scan-4.34 {Tcl_ScanObjCmd, base-10 integer scanning} {
    set x {}
    list [scan {-01234} {%d} x] $x
} {1 -1234}
test scan-4.35 {Tcl_ScanObjCmd, base-10 integer scanning} {
    set x {}
    list [scan {a01234} {%d} x] $x
} {0 {}}
test scan-4.36 {Tcl_ScanObjCmd, base-10 integer scanning} {
    set x {}
    list [scan {0x10} {%d} x] $x
} {1 0}
test scan-4.37 {Tcl_ScanObjCmd, base-8 integer scanning} {
    set x {}
    list [scan {012345678} {%o} x] $x
} {1 342391}
test scan-4.38 {Tcl_ScanObjCmd, base-8 integer scanning} {
    set x {}
    list [scan {+1238 -1239 123a} {%o%*s%o%*s%o} x y z] $x $y $z
} {3 83 -83 83}
test scan-4.39 {Tcl_ScanObjCmd, base-16 integer scanning} {
    set x {}
    list [scan {+1238 -123a 0123} {%x%x%x} x y z] $x $y $z
} {3 4664 -4666 291}
test scan-4.40 {Tcl_ScanObjCmd, base-16 integer scanning} {
    # The behavior changed in 8.4a4/8.3.4cvs (6 Feb) to correctly
    # return '1' for 0x1 scanned via %x, to comply with 8.0 and C scanf.
    # Bug #495213
    set x {}
    list [scan {aBcDeF AbCdEf 0x1} {%x%x%x} x y z] $x $y $z
} {3 11259375 11259375 1}
test scan-4.40.1 {Tcl_ScanObjCmd, base-16 integer scanning} {
    set x {}
    list [scan {0xF 0x00A0B 0X0XF} {%x %x %x} x y z] $x $y $z
} {3 15 2571 0}
test scan-4.40.2 {Tcl_ScanObjCmd, base-16 integer scanning} {
    catch {unset x}
    list [scan {xF} {%x} x] [info exists x]
} {0 0}
test scan-4.41 {Tcl_ScanObjCmd, base-unknown integer scanning} {
    set x {}
    list [scan {10 010 0x10} {%i%i%i} x y z] $x $y $z
} {3 10 8 16}
test scan-4.42 {Tcl_ScanObjCmd, base-unknown integer scanning} {
    set x {}
    list [scan {10 010 0X10} {%i%i%i} x y z] $x $y $z
} {3 10 8 16}
test scan-4.43 {Tcl_ScanObjCmd, integer scanning, odd cases} {
    set x {}
    list [scan {+ } {%i} x] $x
} {0 {}}
test scan-4.44 {Tcl_ScanObjCmd, integer scanning, odd cases} {
    set x {}
    list [scan {+} {%i} x] $x
} {-1 {}}
test scan-4.45 {Tcl_ScanObjCmd, integer scanning, odd cases} {
    set x {}
    list [scan {0x} {%i%s} x y] $x $y
} {2 0 x}
test scan-4.46 {Tcl_ScanObjCmd, integer scanning, odd cases} {
    set x {}
    list [scan {0X} {%i%s} x y] $x $y
} {2 0 X}
test scan-4.47 {Tcl_ScanObjCmd, integer scanning, suppressed} {
    set x {}
    list [scan {123def} {%*i%s} x] $x
} {1 def}
test scan-4.48 {Tcl_ScanObjCmd, float scanning} {
    list [scan {1 2 3} {%e %f %g} x y z] $x $y $z
} {3 1.0 2.0 3.0}
test scan-4.49 {Tcl_ScanObjCmd, float scanning} {
    list [scan {.1 0.2 3.} {%e %f %g} x y z] $x $y $z
} {3 0.1 0.2 3.0}
test scan-4.50 {Tcl_ScanObjCmd, float scanning} {
    list [scan {1234567890a} %f x] $x
} {1 1234567890.0}
test scan-4.51 {Tcl_ScanObjCmd, float scanning} {
    list [scan {+123+45} %f x] $x
} {1 123.0}
test scan-4.52 {Tcl_ScanObjCmd, float scanning} {
    list [scan {-123+45} %f x] $x
} {1 -123.0}
test scan-4.53 {Tcl_ScanObjCmd, float scanning} {
    list [scan {1.0e1} %f x] $x
} {1 10.0}
test scan-4.54 {Tcl_ScanObjCmd, float scanning} {
    list [scan {1.0e-1} %f x] $x
} {1 0.1}
test scan-4.55 {Tcl_ScanObjCmd, odd cases} {
    set x {}
    list [scan {+} %f x] $x
} {-1 {}}
test scan-4.56 {Tcl_ScanObjCmd, odd cases} {
    set x {}
    list [scan {1.0e} %f%s x y] $x $y
} {2 1.0 e}
test scan-4.57 {Tcl_ScanObjCmd, odd cases} {
    set x {}
    list [scan {1.0e+} %f%s x y] $x $y
} {2 1.0 e+}
test scan-4.58 {Tcl_ScanObjCmd, odd cases} {
    set x {}
    set y {}
    list [scan {e1} %f%s x y] $x $y
} {0 {} {}}
test scan-4.59 {Tcl_ScanObjCmd, float scanning} {
    list [scan {1.0e-1x} %*f%n x] $x
} {1 6}

test scan-4.60 {Tcl_ScanObjCmd, set errors} {
    set x {}
    set y {}
    catch {unset z}; array set z {}
    set result [list [catch {scan {abc def ghi} {%s%s%s} x z y} msg] \
	    $msg $x $y]
    unset z
    set result
} {1 {couldn't set variable "z"} abc ghi}
test scan-4.61 {Tcl_ScanObjCmd, set errors} {
    set x {}
    catch {unset y}; array set y {}
    catch {unset z}; array set z {}
    set result [list [catch {scan {abc def ghi} {%s%s%s} x z y} msg] \
	    $msg $x]
    unset y
    unset z
    set result
} {1 {couldn't set variable "z"couldn't set variable "y"} abc}

# procedure that returns the range of integers

proc int_range {} {
    for { set MIN_INT 1 } { $MIN_INT > 0 } {} {
	set MIN_INT [expr { $MIN_INT << 1 }]
    }
    set MAX_INT [expr { ~ $MIN_INT }]
    return [list $MIN_INT $MAX_INT]
}

test scan-4.62 {scanning of large and negative octal integers} {
    foreach { MIN_INT MAX_INT } [int_range] {}
    set scanstring [format {%o %o %o} -1 $MIN_INT $MAX_INT]
    list [scan $scanstring {%o %o %o} a b c] \
	[expr { $a == -1 }] [expr { $b == $MIN_INT }] [expr { $c == $MAX_INT }]
} {3 1 1 1}
test scan-4.63 {scanning of large and negative hex integers} {
    foreach { MIN_INT MAX_INT } [int_range] {}
    set scanstring [format {%x %x %x} -1 $MIN_INT $MAX_INT]
    list [scan $scanstring {%x %x %x} a b c] \
	[expr { $a == -1 }] [expr { $b == $MIN_INT }] [expr { $c == $MAX_INT }]
} {3 1 1 1}

# clean up from last two tests

catch {
    rename int_range {}
}

test scan-5.1 {integer scanning} {
    set a {}; set b {}; set c {}; set d {}
    list [scan "-20 1476 \n33 0" "%d %d %d %d" a b c d] $a $b $c $d
} {4 -20 1476 33 0}
test scan-5.2 {integer scanning} {
    set a {}; set b {}; set c {}
    list [scan "-45 16 7890 +10" "%2d %*d %10d %d" a b c] $a $b $c
} {3 -4 16 7890}
test scan-5.3 {integer scanning} {
    set a {}; set b {}; set c {}; set d {}
    list [scan "-45 16 +10 987" "%ld %d %ld %d" a b c d] $a $b $c $d
} {4 -45 16 10 987}
test scan-5.4 {integer scanning} {
    set a {}; set b {}; set c {}; set d {}
    list [scan "14 1ab 62 10" "%d %x %lo %x" a b c d] $a $b $c $d
} {4 14 427 50 16}
test scan-5.5 {integer scanning} {
    set a {}; set b {}; set c {}; set d {}
    list [scan "12345670 1234567890ab cdefg" "%o	 %o %x %lx" a b c d] \
	    $a $b $c $d
} {4 2739128 342391 561323 52719}
test scan-5.6 {integer scanning} {
    set a {}; set b {}; set c {}; set d {}
    list [scan "ab123-24642" "%2x %3x %3o %2o" a b c d] $a $b $c $d
} {4 171 291 -20 52}
test scan-5.7 {integer scanning} {
    set a {}; set b {}
    list [scan "1234567 234 567  " "%*3x %x %*o %4o" a b] $a $b
} {2 17767 375}
test scan-5.8 {integer scanning} {
    set a {}; set b {}
    list [scan "a	1234" "%d %d" a b] $a $b
} {0 {} {}}
test scan-5.9 {integer scanning} {
    set a {}; set b {}; set c {}; set d {};
    list [scan "12345678" "%2d %2d %2ld %2d" a b c d] $a $b $c $d
} {4 12 34 56 78}
test scan-5.10 {integer scanning} {
    set a {}; set b {}; set c {}; set d {}
    list [scan "1 2 " "%hd %d %d %d" a b c d] $a $b $c $d
} {2 1 2 {} {}}
#
# The behavior for scaning intergers larger than MAX_INT is
# not defined by the ANSI spec.  Some implementations wrap the
# input (-16) some return MAX_INT.
#
test scan-5.11 {integer scanning} {nonPortable} {
    set a {}; set b {};
    list [scan "4294967280 4294967280" "%u %d" a b] $a \
	    [expr {$b == -16 || $b == 0x7fffffff}]
} {2 4294967280 1}
test scan-5.12 {integer scanning} {64bitInts} {
    set a {}; set b {}; set c {}
    list [scan "7810179016327718216,6c63546f6c6c6548,661432506755433062510" \
	    %ld,%lx,%lo a b c] $a $b $c
} {3 7810179016327718216 7810179016327718216 7810179016327718216}
test scan-5.13 {integer scanning and overflow} {
    # This test used to fail on some 64-bit systems. [Bug 1011860]
    scan {300000000 3000000000 30000000000} {%ld %ld %ld}
} {300000000 3000000000 30000000000}

test scan-6.1 {floating-point scanning} {
    set a {}; set b {}; set c {}; set d {}
    list [scan "2.1 -3.0e8 .99962 a" "%f%g%e%f" a b c d] $a $b $c $d
} {3 2.1 -300000000.0 0.99962 {}}
test scan-6.2 {floating-point scanning} {
    set a {}; set b {}; set c {}; set d {}
    list [scan "-1.2345 +8.2 9" "%3e %3lf %f %f" a b c d] $a $b $c $d
} {4 -1.0 234.0 5.0 8.2}
test scan-6.3 {floating-point scanning} {
    set a {}; set b {}; set c {}
    list [scan "1e00004 332E-4 3e+4" "%Lf %*2e %f %f" a b c] $a $c
} {3 10000.0 30000.0}
#
# Some libc implementations consider 3.e- bad input.  The ANSI
# spec states that digits must follow the - sign.
#
test scan-6.4 {floating-point scanning} {
    set a {}; set b {}; set c {}
    list [scan "1. 47.6 2.e2 3.e-" "%f %*f %f %f" a b c] $a $b $c
} {3 1.0 200.0 3.0}
test scan-6.5 {floating-point scanning} {
    set a {}; set b {}; set c {}; set d {}
    list [scan "4.6 99999.7 876.43e-1 118" "%f %f %f %e" a b c d] $a $b $c $d
} {4 4.6 99999.7 87.643 118.0}
test scan-6.6 {floating-point scanning} {eformat} {
    set a {}; set b {}; set c {}; set d {}
    list [scan "1.2345 697.0e-3 124 .00005" "%f %e %f %e" a b c d] $a $b $c $d
} {4 1.2345 0.697 124.0 5e-05}
test scan-6.7 {floating-point scanning} {
    set a {}; set b {}; set c {}; set d {}
    list [scan "4.6abc" "%f %f %f %f" a b c d] $a $b $c $d
} {1 4.6 {} {} {}}
test scan-6.8 {floating-point scanning} {
    set a {}; set b {}; set c {}; set d {}
    list [scan "4.6 5.2" "%f %f %f %f" a b c d] $a $b $c $d
} {2 4.6 5.2 {} {}}

test scan-7.1 {string and character scanning} {
    set a {}; set b {}; set c {}; set d {}
    list [scan "abc defghijk dum " "%s %3s %20s %s" a b c d] $a $b $c $d
} {4 abc def ghijk dum}
test scan-7.2 {string and character scanning} {
    set a {}; set b {}; set c {}; set d {}
    list [scan "a       bcdef" "%c%c%1s %s" a b c d] $a $b $c $d
} {4 97 32 b cdef}
test scan-7.3 {string and character scanning} {
    set a {}; set b {}; set c {}
    list [scan "123456 test " "%*c%*s %s %s %s" a b c] $a $b $c
} {1 test {} {}}
test scan-7.4 {string and character scanning} {
    set a {}; set b {}; set c {}; set d
    list [scan "ababcd01234  f 123450" {%4[abcd] %4[abcd] %[^abcdef] %[^0]} a b c d] $a $b $c $d
} {4 abab cd {01234  } {f 12345}}
test scan-7.5 {string and character scanning} {
    set a {}; set b {}; set c {}
    list [scan "aaaaaabc aaabcdefg  + +  XYZQR" {%*4[a] %s %*4[a]%s%*4[ +]%c} a b c] $a $b $c
} {3 aabc bcdefg 43}
test scan-7.6 {string and character scanning, unicode} {
    set a {}; set b {}; set c {}; set d {}
    list [scan "abc d\u00c7fghijk dum " "%s %3s %20s %s" a b c d] $a $b $c $d
} "4 abc d\u00c7f ghijk dum"
test scan-7.7 {string and character scanning, unicode} {
    set a {}; set b {}
    list [scan "ab\u00c7cdef" "ab%c%c" a b] $a $b
} "2 199 99"
test scan-7.8 {string and character scanning, unicode} {
    set a {}; set b {}
    list [scan "ab\ufeffdef" "%\[ab\ufeff\]" a] $a
} "1 ab\ufeff"

test scan-8.1 {error conditions} {
    catch {scan a}
} 1
test scan-8.2 {error conditions} {
    catch {scan a} msg
    set msg
} {wrong # args: should be "scan string format ?varName varName ...?"}
test scan-8.3 {error conditions} {
    list [catch {scan a %D x} msg] $msg
} {1 {bad scan conversion character "D"}}
test scan-8.4 {error conditions} {
    list [catch {scan a %O x} msg] $msg
} {1 {bad scan conversion character "O"}}
test scan-8.5 {error conditions} {
    list [catch {scan a %X x} msg] $msg
} {1 {bad scan conversion character "X"}}
test scan-8.6 {error conditions} {
    list [catch {scan a %F x} msg] $msg
} {1 {bad scan conversion character "F"}}
test scan-8.7 {error conditions} {
    list [catch {scan a %E x} msg] $msg
} {1 {bad scan conversion character "E"}}
test scan-8.8 {error conditions} {
    list [catch {scan a "%d %d" a} msg] $msg
} {1 {different numbers of variable names and field specifiers}}
test scan-8.9 {error conditions} {
    list [catch {scan a "%d %d" a b c} msg] $msg
} {1 {variable is not assigned by any conversion specifiers}}
test scan-8.10 {error conditions} {
    set a {}; set b {}; set c {}; set d {}
    list [expr {[scan "  a" " a %d %d %d %d" a b c d] <= 0}] $a $b $c $d
} {1 {} {} {} {}}
test scan-8.11 {error conditions} {
    set a {}; set b {}; set c {}; set d {}
    list [scan "1 2" "%d %d %d %d" a b c d] $a $b $c $d
} {2 1 2 {} {}}
test scan-8.12 {error conditions} {
    catch {unset a}
    set a(0) 44
    list [catch {scan 44 %d a} msg] $msg
} {1 {couldn't set variable "a"}}
test scan-8.13 {error conditions} {
    catch {unset a}
    set a(0) 44
    list [catch {scan 44 %c a} msg] $msg
} {1 {couldn't set variable "a"}}
test scan-8.14 {error conditions} {
    catch {unset a}
    set a(0) 44
    list [catch {scan 44 %s a} msg] $msg
} {1 {couldn't set variable "a"}}
test scan-8.15 {error conditions} {
    catch {unset a}
    set a(0) 44
    list [catch {scan 44 %f a} msg] $msg
} {1 {couldn't set variable "a"}}
test scan-8.16 {error conditions} {
    catch {unset a}
    set a(0) 44
    list [catch {scan 44 %f a} msg] $msg
} {1 {couldn't set variable "a"}}
catch {unset a}
test scan-8.17 {error conditions} {
    list [catch {scan 44 %2c a} msg] $msg
} {1 {field width may not be specified in %c conversion}}
test scan-8.18 {error conditions} {
    list [catch {scan abc {%[} x} msg] $msg
} {1 {unmatched [ in format string}}
test scan-8.19 {error conditions} {
    list [catch {scan abc {%[^a} x} msg] $msg
} {1 {unmatched [ in format string}}
test scan-8.20 {error conditions} {
    list [catch {scan abc {%[^]a} x} msg] $msg
} {1 {unmatched [ in format string}}
test scan-8.21 {error conditions} {
    list [catch {scan abc {%[]a} x} msg] $msg
} {1 {unmatched [ in format string}}

test scan-9.1 {lots of arguments} {
    scan "10 20 30 40 50 60 70 80 90 100 110 120 130 140 150 160 170 180 190 200" "%d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d" a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a16 a17 a18 a19 a20
} 20
test scan-9.2 {lots of arguments} {
    scan "10 20 30 40 50 60 70 80 90 100 110 120 130 140 150 160 170 180 190 200" "%d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d" a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a16 a17 a18 a19 a20
    set a20
} 200

test scan-10.1 {miscellaneous tests} {
    set a {}
    list [scan ab16c ab%dc a] $a
} {1 16}
test scan-10.2 {miscellaneous tests} {
    set a {}
    list [scan ax16c ab%dc a] $a
} {0 {}}
test scan-10.3 {miscellaneous tests} {
    set a {}
    list [catch {scan ab%c114 ab%%c%d a} msg] $msg $a
} {0 1 114}
test scan-10.4 {miscellaneous tests} {
    set a {}
    list [catch {scan ab%c14 ab%%c%d a} msg] $msg $a
} {0 1 14}
test scan-10.5 {miscellaneous tests} {
    catch {unset arr}
    set arr(2) {}
    list [catch {scan ab%c14 ab%%c%d arr(2)} msg] $msg $arr(2)
} {0 1 14}

test scan-11.1 {alignment in results array (TCL_ALIGN)} {
    scan "123 13.6" "%s %f" a b
    set b
} 13.6
test scan-11.2 {alignment in results array (TCL_ALIGN)} {
    scan "1234567 13.6" "%s %f" a b
    set b
} 13.6
test scan-11.3 {alignment in results array (TCL_ALIGN)} {
    scan "12345678901 13.6" "%s %f" a b
    set b
} 13.6
test scan-11.4 {alignment in results array (TCL_ALIGN)} {
    scan "123456789012345 13.6" "%s %f" a b
    set b
} 13.6
test scan-11.5 {alignment in results array (TCL_ALIGN)} {
    scan "1234567890123456789 13.6" "%s %f" a b
    set b
} 13.6

test scan-12.1 {Tcl_ScanObjCmd, inline case} {
    scan a %c
} 97
test scan-12.2 {Tcl_ScanObjCmd, inline case} {
    scan abc %c%c%c%c
} {97 98 99 {}}
test scan-12.3 {Tcl_ScanObjCmd, inline case} {
    scan abc %s%c
} {abc {}}
test scan-12.4 {Tcl_ScanObjCmd, inline case, underflow} {
    scan abc abc%c
} {}
test scan-12.5 {Tcl_ScanObjCmd, inline case} {
    scan abc bogus%c%c%c
} {{} {} {}}
test scan-12.6 {Tcl_ScanObjCmd, inline case} {
    # degenerate case, behavior changed from 8.2 to 8.3
    list [catch {scan foo foobar} msg] $msg
} {0 {}}
test scan-12.7 {Tcl_ScanObjCmd, inline case lots of arguments} {
    scan "10 20 30 40 50 60 70 80 90 100 110 120 130 140\
	    150 160 170 180 190 200" \
	    "%d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d"
} {10 20 30 40 50 60 70 80 90 100 110 120 130 140 150 160 170 180 190 200 {}}

test scan-13.1 {Tcl_ScanObjCmd, inline XPG case} {
    scan a {%1$c}
} 97
test scan-13.2 {Tcl_ScanObjCmd, inline XPG case} {
    scan abc {%1$c%2$c%3$c%4$c}
} {97 98 99 {}}
test scan-13.3 {Tcl_ScanObjCmd, inline XPG case} {
    list [catch {scan abc {%1$c%1$c}} msg] $msg
} {1 {variable is assigned by multiple "%n$" conversion specifiers}}
test scan-13.4 {Tcl_ScanObjCmd, inline XPG case} {
    scan abc {%2$s%1$c}
} {{} abc}
test scan-13.5 {Tcl_ScanObjCmd, inline XPG case, underflow} {
    scan abc {abc%5$c}
} {}
test scan-13.6 {Tcl_ScanObjCmd, inline XPG case} {
    catch {scan abc {bogus%1$c%5$c%10$c}} msg
    list [llength $msg] $msg
} {10 {{} {} {} {} {} {} {} {} {} {}}}
test scan-13.7 {Tcl_ScanObjCmd, inline XPG case lots of arguments} {
    scan "10 20 30 40 50 60 70 80 90 100 110 120 130 140 150 160 170 180 190 200" {%20$d %18$d %17$d %16$d %15$d %14$d %13$d %12$d %11$d %10$d %9$d %8$d %7$d %6$d %5$d %4$d %3$d %2$d %1$d}
} {190 180 170 160 150 140 130 120 110 100 90 80 70 60 50 40 30 20 {} 10}
test scan-13.8 {Tcl_ScanObjCmd, inline XPG case lots of arguments} {
    set msg [scan "10 20 30" {%100$d %5$d %200$d}]
    list [llength $msg] [lindex $msg 99] [lindex $msg 4] [lindex $msg 199]
} {200 10 20 30}

# cleanup
::tcltest::cleanupTests
return
