/*
 * Copyright (c) 2015 Goldman Sachs.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * and Eclipse Distribution License v. 1.0 which accompany this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 */

package org.eclipse.collections.impl.stack.immutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.io.Serializable;

import org.eclipse.collections.api.LazyBooleanIterable;
import org.eclipse.collections.api.BooleanIterable;
import org.eclipse.collections.api.bag.primitive.MutableBooleanBag;
import org.eclipse.collections.api.block.function.primitive.BooleanToObjectFunction;
import org.eclipse.collections.api.block.function.primitive.ObjectBooleanToObjectFunction;
import org.eclipse.collections.api.block.predicate.primitive.BooleanPredicate;
import org.eclipse.collections.api.block.procedure.primitive.BooleanProcedure;
import org.eclipse.collections.api.iterator.BooleanIterator;
import org.eclipse.collections.api.list.primitive.BooleanList;
import org.eclipse.collections.api.list.primitive.MutableBooleanList;
import org.eclipse.collections.api.set.primitive.MutableBooleanSet;
import org.eclipse.collections.api.stack.ImmutableStack;
import org.eclipse.collections.api.stack.primitive.BooleanStack;
import org.eclipse.collections.api.stack.primitive.ImmutableBooleanStack;
import org.eclipse.collections.impl.bag.mutable.primitive.BooleanHashBag;
import org.eclipse.collections.impl.block.procedure.checked.primitive.CheckedBooleanProcedure;
import org.eclipse.collections.impl.factory.Stacks;
import org.eclipse.collections.impl.factory.primitive.BooleanStacks;
import org.eclipse.collections.impl.iterator.UnmodifiableBooleanIterator;
import org.eclipse.collections.impl.lazy.primitive.LazyBooleanIterableAdapter;
import org.eclipse.collections.impl.list.mutable.primitive.BooleanArrayList;
import org.eclipse.collections.impl.set.mutable.primitive.BooleanHashSet;
import org.eclipse.collections.impl.stack.mutable.primitive.BooleanArrayStack;
import net.jcip.annotations.Immutable;

/**
 * ImmutableBooleanArrayStack is the non-modifiable equivalent of {@link BooleanArrayStack}.
 * This file was automatically generated from template file immutablePrimitiveArrayStack.stg.
 *
 * @since 4.0.
 */
@Immutable
final class ImmutableBooleanArrayStack
        implements ImmutableBooleanStack, Serializable
{
    private static final long serialVersionUID = 1L;
    private final BooleanArrayList delegate;

    private ImmutableBooleanArrayStack(boolean[] newElements)
    {
        this.checkOptimizedSize(newElements.length);
        this.delegate = new BooleanArrayList(newElements);
    }

    private ImmutableBooleanArrayStack(BooleanArrayList newElements)
    {
        this.checkOptimizedSize(newElements.size());
        this.delegate = newElements;
    }

    private void checkOptimizedSize(int length)
    {
        if (length <= 1)
        {
            throw new IllegalArgumentException("Use BooleanStacks.immutable.with() to instantiate an optimized collection");
        }
    }

    public static ImmutableBooleanArrayStack newStack(BooleanIterable iterable)
    {
        return new ImmutableBooleanArrayStack(iterable.toArray());
    }

    public static ImmutableBooleanArrayStack newStackWith(boolean... elements)
    {
        boolean[] newArray = new boolean[elements.length];
        System.arraycopy(elements, 0, newArray, 0, elements.length);
        return new ImmutableBooleanArrayStack(newArray);
    }

    public static ImmutableBooleanArrayStack newStackFromTopToBottom(boolean... items)
    {
        return new ImmutableBooleanArrayStack(BooleanArrayList.newListWith(items).reverseThis());
    }

    public static ImmutableBooleanArrayStack newStackFromTopToBottom(BooleanIterable items)
    {
        return new ImmutableBooleanArrayStack(BooleanArrayList.newList(items).reverseThis());
    }

    public ImmutableBooleanStack push(boolean item)
    {
        BooleanArrayList newDelegate = BooleanArrayList.newList(this.delegate);
        newDelegate.add(item);
        return new ImmutableBooleanArrayStack(newDelegate);
    }

    public ImmutableBooleanStack pop()
    {
        BooleanArrayList newDelegate = BooleanArrayList.newList(this.delegate);
        newDelegate.removeAtIndex(this.delegate.size() - 1);
        return BooleanStacks.immutable.with(newDelegate.toArray());
    }

    public ImmutableBooleanStack pop(int count)
    {
        this.checkNegativeCount(count);
        if (count == 0)
        {
            return this;
        }
        this.checkSizeLessThanCount(count);
        BooleanArrayList newDelegate = BooleanArrayList.newList(this.delegate);
        while (count > 0)
        {
            newDelegate.removeAtIndex(newDelegate.size() - 1);
            count--;
        }
        return BooleanStacks.immutable.with(newDelegate.toArray());
    }

    private void checkNegativeCount(int count)
    {
        if (count < 0)
        {
            throw new IllegalArgumentException("Count must be positive but was " + count);
        }
    }

    public boolean peek()
    {
        return this.delegate.getLast();
    }

    public BooleanList peek(int count)
    {
        this.checkNegativeCount(count);
        this.checkSizeLessThanCount(count);
        if (count == 0)
        {
            return new BooleanArrayList();
        }
        MutableBooleanList subList = new BooleanArrayList(count);
        int index = this.delegate.size() - 1;
        for (int i = 0; i < count; i++)
        {
            subList.add(this.delegate.get(index - i));
        }
        return subList;
    }

    public boolean peekAt(int index)
    {
        this.rangeCheck(index);
        return this.delegate.get(this.delegate.size() - 1 - index);
    }

    private void rangeCheck(int index)
    {
        if (index < 0 || index > this.delegate.size() - 1)
        {
            throw new IllegalArgumentException("Index " + index + " out of range.Should be between 0 and " + (this.delegate.size() - 1));
        }
    }

    public BooleanIterator booleanIterator()
    {
        return new UnmodifiableBooleanIterator(this.delegate.asReversed().booleanIterator());
    }

    public void forEach(BooleanProcedure procedure)
    {
        this.each(procedure);
    }

    /**
     * @since 7.0.
     */
    public void each(BooleanProcedure procedure)
    {
        this.delegate.asReversed().forEach(procedure);
    }

    public int count(BooleanPredicate predicate)
    {
        return this.delegate.asReversed().count(predicate);
    }

    public boolean anySatisfy(BooleanPredicate predicate)
    {
        return this.delegate.asReversed().anySatisfy(predicate);
    }

    public boolean allSatisfy(BooleanPredicate predicate)
    {
        return this.delegate.asReversed().allSatisfy(predicate);
    }

    public boolean noneSatisfy(BooleanPredicate predicate)
    {
        return this.delegate.asReversed().noneSatisfy(predicate);
    }

    public ImmutableBooleanStack select(BooleanPredicate predicate)
    {
        return BooleanStacks.immutable.withAllReversed(this.delegate.asReversed().select(predicate));
    }

    public ImmutableBooleanStack reject(BooleanPredicate predicate)
    {
        return BooleanStacks.immutable.withAllReversed(this.delegate.asReversed().reject(predicate));
    }

    public boolean detectIfNone(BooleanPredicate predicate, boolean ifNone)
    {
        return this.delegate.asReversed().detectIfNone(predicate, ifNone);
    }

    public <V> ImmutableStack<V> collect(BooleanToObjectFunction<? extends V> function)
    {
        return Stacks.immutable.withAllReversed(this.delegate.asReversed().collect(function));
    }


    public boolean[] toArray()
    {
        return this.delegate.asReversed().toArray();
    }

    public boolean contains(boolean value)
    {
        return this.delegate.asReversed().contains(value);
    }

    public boolean containsAll(boolean... source)
    {
        return this.delegate.asReversed().containsAll(source);
    }

    public boolean containsAll(BooleanIterable source)
    {
        return this.delegate.asReversed().containsAll(source);
    }

    public MutableBooleanList toList()
    {
        return BooleanArrayList.newList(this);
    }

    public MutableBooleanSet toSet()
    {
        return BooleanHashSet.newSet(this);
    }

    public MutableBooleanBag toBag()
    {
        return BooleanHashBag.newBag(this);
    }

    public <V> V injectInto(V injectedValue, ObjectBooleanToObjectFunction<? super V, ? extends V> function)
    {
        return this.delegate.toReversed().injectInto(injectedValue, function);
    }

    public LazyBooleanIterable asLazy()
    {
        return new LazyBooleanIterableAdapter(this);
    }

    public ImmutableBooleanStack toImmutable()
    {
        return this;
    }

    public int size()
    {
        return this.delegate.size();
    }

    public boolean isEmpty()
    {
        return false;
    }

    public boolean notEmpty()
    {
        return true;
    }

    @Override
    public boolean equals(Object otherStack)
    {
        if (otherStack == this)
        {
            return true;
        }
        if (!(otherStack instanceof BooleanStack))
        {
            return false;
        }
        BooleanStack stack = (BooleanStack) otherStack;
        if (this.size() != stack.size())
        {
            return false;
        }
        for (int i = 0; i < this.size(); i++)
        {
            if (this.peekAt(i) != stack.peekAt(i))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int hashCode = 1;
        BooleanIterable iterable = this.delegate.asReversed();
        BooleanIterator iterator = iterable.booleanIterator();
        while (iterator.hasNext())
        {
            boolean item = iterator.next();
            hashCode = 31 * hashCode + (item ? 1231 : 1237);
        }
        return hashCode;
    }

    @Override
    public String toString()
    {
        return this.delegate.asReversed().toString();
    }

    public String makeString()
    {
        return this.delegate.asReversed().makeString();
    }

    public String makeString(String separator)
    {
        return this.delegate.asReversed().makeString(separator);
    }

    public String makeString(String start, String separator, String end)
    {
        return this.delegate.asReversed().makeString(start, separator, end);
    }

    public void appendString(Appendable appendable)
    {
        this.delegate.asReversed().appendString(appendable);
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.delegate.asReversed().appendString(appendable, separator);
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        this.delegate.asReversed().appendString(appendable, start, separator, end);
    }

    private void checkSizeLessThanCount(int count)
    {
        if (this.delegate.size() < count)
        {
            throw new IllegalArgumentException("Count must be less than size: Count = " + count + " Size = " + this.delegate.size());
        }
    }

    private Object writeReplace()
    {
        return new ImmutableBooleanStackSerializationProxy(this);
    }

    private static class ImmutableBooleanStackSerializationProxy implements Externalizable
    {
        private static final long serialVersionUID = 1L;

        private BooleanStack stack;

        @SuppressWarnings("UnusedDeclaration")
        public ImmutableBooleanStackSerializationProxy()
        {
            // Empty constructor for Externalizable class
        }

        protected ImmutableBooleanStackSerializationProxy(BooleanStack stack)
        {
            this.stack = stack;
        }

        public void writeExternal(final ObjectOutput out) throws IOException
        {
            out.writeInt(this.stack.size());
            try
            {
                this.stack.forEach(new CheckedBooleanProcedure()
                {
                    @Override
                    public void safeValue(boolean item) throws IOException
                    {
                        out.writeBoolean(item);
                    }
                });
            }
            catch (RuntimeException e)
            {
                if (e.getCause() instanceof IOException)
                {
                    throw (IOException) e.getCause();
                }
                throw e;
            }
        }

        public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
        {
            int size = in.readInt();
            BooleanArrayList deserializedDelegate = new BooleanArrayList(size);

            for (int i = 0; i < size; i++)
            {
                deserializedDelegate.add(in.readBoolean());
            }

            this.stack = ImmutableBooleanArrayStack.newStackFromTopToBottom(deserializedDelegate);
        }

        protected Object readResolve()
        {
            return this.stack;
        }
    }
}
