/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.generic.internal.ui.config;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.xmlbeans.XmlException;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ICellModifier;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.window.Window;
import org.eclipse.jsr220Orm.generic.xml.VendorDefinitionDocument;
import org.eclipse.jsr220Orm.generic.xml.VendorDefinitionDocument.VendorDefinition;
import org.eclipse.jsr220Orm.generic.xml.VendorDefinitionDocument.VendorDefinition.JdbcTypeMap;
import org.eclipse.jsr220Orm.generic.xml.VendorDefinitionDocument.VendorDefinition.PersistenceXml;
import org.eclipse.jsr220Orm.generic.xml.VendorDefinitionDocument.VendorDefinition.PersistenceXml.PropertyMapping;
import org.eclipse.jsr220Orm.xml.EntityManagerDocument;
import org.eclipse.jsr220Orm.xml.EntityManagerDocument.EntityManager;
import org.eclipse.jsr220Orm.xml.EntityManagerDocument.EntityManager.Properties;
import org.eclipse.jsr220Orm.xml.EntityManagerDocument.EntityManager.Properties.Property;
import org.eclipse.jsr220orm.core.internal.product.OrmProduct;
import org.eclipse.jsr220orm.core.internal.product.OrmProductRegistry;
import org.eclipse.jsr220orm.core.nature.PersistenceProperties;
import org.eclipse.jsr220orm.core.util.RdbUtils;
import org.eclipse.jsr220orm.generic.GenericPlugin;
import org.eclipse.jsr220orm.generic.Utils;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;
import org.eclipse.ui.views.navigator.ResourceSorter;
import org.eclipse.wst.rdb.internal.core.connection.ConnectionInfo;
import org.eclipse.wst.rdb.internal.core.definition.DatabaseDefinition;

public class GenericPropertyComposite extends Composite {

    private Label lPersistence = null;
    private Text tPersistence = null;
    private Button bPersistence = null;
    private Label lParFile = null;
    private Text tParFile = null;
    private Button bParFile = null;
    private Group cProperties = null;
    private Table tProperties = null;
    private Button bAdd = null;
    private Button bRemove = null;
    private IProject project;
    private Button bNew = null;
    private TableViewer tableViewer;
    private Properties properties;
    private EntityManagerDocument emd;
    private Label lContextName;
    private Text tContextName;

    public GenericPropertyComposite(Composite parent, int style) {
        super(parent, style);
        initialize();
    }

    private void initialize() {
        GridData gridData2 = new GridData();
        gridData2.horizontalAlignment = org.eclipse.swt.layout.GridData.END;
        GridData gridData11 = new GridData();
        gridData11.horizontalAlignment = org.eclipse.swt.layout.GridData.END;
        GridData gridData1 = new GridData();
        gridData1.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
        GridData gridData = new GridData();
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
        GridData gridData3 = new GridData();
        gridData3.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
        gridData3.horizontalSpan = 2;
        GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 3;
        lPersistence = new Label(this, SWT.NONE);
        lPersistence.setText("persistence.xml location");
        lPersistence.setLayoutData(gridData2);
        tPersistence = new Text(this, SWT.BORDER);
        tPersistence.setLayoutData(gridData);
        tPersistence
                .addModifyListener(new org.eclipse.swt.events.ModifyListener() {
                    public void modifyText(org.eclipse.swt.events.ModifyEvent e) {
                        loadProperties();
                    }
                });
        bPersistence = new Button(this, SWT.NONE);
        bPersistence.setText("Browse...");
        bPersistence
                .addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
                    public void widgetSelected(
                            org.eclipse.swt.events.SelectionEvent e) {
                        findFile(tPersistence, "Choose persistence.xml",
                                "Choose your persistence.xml file.", "xml");
                    }
                });
        lContextName = new Label(this, SWT.NONE);
        lContextName.setText("Persistence context name");
        lContextName.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_END));
        tContextName = new Text(this, SWT.BORDER);
        tContextName.setLayoutData(gridData3);
        lParFile = new Label(this, SWT.NONE);
        lParFile.setText(".par file location");
        lParFile.setLayoutData(gridData11);
        tParFile = new Text(this, SWT.BORDER);
        tParFile.setLayoutData(gridData1);
        bParFile = new Button(this, SWT.NONE);
        bParFile.setText("Browse...");
        bParFile
                .addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
                    public void widgetSelected(
                            org.eclipse.swt.events.SelectionEvent e) {
                        findFile(tParFile, "Select par",
                                "Select where to create your par file.", "par",
                                "jar");
                    }
                });
        this.setLayout(gridLayout);
        createGroup();
        setSize(new Point(300, 200));
    }

    protected void findFile(Text text, String title, String message,
            String... exts) {
        FileSelectionDialog dialog = new FileSelectionDialog(getShell());
        dialog.setTitle(title);
        dialog.setMessage(message);
        dialog.setInput(project);
        dialog.setAllowMultiple(false);
        dialog.setValidExtentions(exts);
        dialog.setSorter(new ResourceSorter(ResourceSorter.TYPE));
        try {
            dialog.setInitialSelection(project.getFile(text.getText()));
        } catch (Exception e) {
            // Do Nothing
        }
        if (dialog.open() == Window.OK) {
            IFile file = (IFile) dialog.getFirstResult();
            file.getFullPath().removeFirstSegments(1);
            text.setText(file.getFullPath().removeFirstSegments(1)
                    .toPortableString());
        }
    }

    /**
     * This method initializes group
     * 
     */
    private void createGroup() {
        GridData gridData6 = new GridData();
        gridData6.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
        GridData gridData5 = new GridData();
        gridData5.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
        GridLayout gridLayout1 = new GridLayout();
        gridLayout1.numColumns = 2;
        GridData gridData3 = new GridData();
        gridData3.horizontalSpan = 3;
        gridData3.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
        gridData3.grabExcessHorizontalSpace = true;
        cProperties = new Group(this, SWT.NONE);
        cProperties.setText("Properties");
        createTable();
        cProperties.setLayout(gridLayout1);
        cProperties.setLayoutData(gridData3);
        bAdd = new Button(cProperties, SWT.NONE);
        bNew = new Button(cProperties, SWT.NONE);
        bNew.setText("New");
        bNew.setLayoutData(gridData6);
        bNew
                .addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
                    public void widgetSelected(
                            org.eclipse.swt.events.SelectionEvent e) {
                        newProperty();
                    }
                });
        bAdd.setText("Add");
        bAdd.setLayoutData(gridData5);
        bAdd
                .addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
                    public void widgetSelected(
                            org.eclipse.swt.events.SelectionEvent e) {
                        addProperty();
                    }
                });
        bRemove = new Button(cProperties, SWT.NONE);
        bRemove.setText("Remove");
        bRemove
                .addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
                    public void widgetSelected(
                            org.eclipse.swt.events.SelectionEvent e) {
                        removeProperty();
                    }
                });
    }

    /**
     * This method initializes table
     * 
     */
    private void createTable() {
        GridData gridData4 = new GridData();
        gridData4.verticalSpan = 6;
        gridData4.verticalAlignment = org.eclipse.swt.layout.GridData.FILL;
        gridData4.grabExcessVerticalSpace = true;
        gridData4.grabExcessHorizontalSpace = true;
        gridData4.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
        tProperties = new Table(cProperties, SWT.NONE);
        tProperties.setEnabled(true);
        tProperties.setHeaderVisible(true);
        tProperties.setLayoutData(gridData4);
        tProperties.setLinesVisible(true);
        TableColumn tcKey = new TableColumn(tProperties, SWT.NONE);
        tcKey.setWidth(220);
        tcKey.setText("Name");
        TableColumn tcValue = new TableColumn(tProperties, SWT.NONE);
        tcValue.setWidth(220);
        tcValue.setText("Value");
        tableViewer = new TableViewer(tProperties);
        tableViewer.setUseHashlookup(true);
        tableViewer.setColumnProperties(new String[] { "Name", "Value" });

        // Create the cell editors
        CellEditor[] editors = new CellEditor[2];
        editors[0] = new TextCellEditor(tProperties);
        editors[1] = new TextCellEditor(tProperties);

        // Assign the cell editors to the viewer
        tableViewer.setCellEditors(editors);
        tableViewer.setCellModifier(new ICellModifier() {

            public void modify(Object element, String property, Object value) {
                TableItem item = (TableItem) element;
                Property property2 = (Property) item.getData();
                if ("Name".equals(property)) {
                    property2.getName().setStringValue(value.toString());
                    item.setText(0, value.toString());
                } else if ("Value".equals(property)) {
                    property2.getValue().setStringValue(value.toString());
                    item.setText(1, value.toString());
                }
            }

            public Object getValue(Object element, String property) {
                Property property2 = (Property) element;
                if ("Name".equals(property)) {
                    return property2.getName().getStringValue();
                } else if ("Value".equals(property)) {
                    return property2.getValue().getStringValue();
                }
                return null;
            }

            public boolean canModify(Object element, String property) {
                return true;
            }

        });
    }

    public void setIProject(IProject project) {
        this.project = project;
        try {
            PersistenceProperties properties = new PersistenceProperties(
                    project);
            String persistenceFileName = properties.getPersistenceFileName();
            tPersistence
                    .setText(persistenceFileName != null ? persistenceFileName
                            : "persistence.xml");
            tParFile.setText(properties.getParFileName());
        } catch (Exception e) {
            GenericPlugin.logException(e, "Error loading poperties.",
                    "Could not load persistence properties.");
        }
        bAdd.setEnabled(!getVendorProperties().isEmpty());
   }

    public boolean saveChanges() {
        try {
            PersistenceProperties properties = new PersistenceProperties(
                    project);
            String persistenceFileName = properties.getPersistenceFileName();
            properties.setParFileName(tParFile.getText());
            properties.setPersistenceFileName(tPersistence.getText());
            String contextName = tContextName.getText();
            if(contextName == null || contextName.trim().length() == 0){
                contextName = null;
            }
            emd.getEntityManager().setName(contextName);
            properties.save();
            Utils.savePersistenceXml(project, emd);
            return true;
        } catch (Exception e) {
            GenericPlugin.logException(e, "Error loading poperties.",
                    "Could not load persistence properties.");
            return false;
        }
    }

    private void loadProperties() {
        try {
            emd = Utils.loadPersistenceXml(project, tPersistence.getText());
            EntityManager em = emd.getEntityManager();
            String contextName = em.getName();
            tContextName.setText(contextName != null ? contextName : "");
            properties = em.getProperties();
            if(properties == null){
                properties = em.addNewProperties();
            }
            Property[] propertyArray = properties.getPropertyArray();
            tProperties.removeAll();
            for (int x = 0; x < propertyArray.length; x++) {
                Property property = propertyArray[x];
                TableItem item = new TableItem(tProperties, SWT.NONE);
                item.setText(0, property.getName().getStringValue());
                item.setText(1, property.getValue().getStringValue());
                item.setData(property);
            }
            setKnownInfo();
        } catch (Exception e) {
            GenericPlugin.log(e);
        }
    }

    private void setKnownInfo() throws CoreException, IOException {
        HashMap<String, String> keyValue = new HashMap<String, String>();
        PersistenceProperties properties = new PersistenceProperties(
                project);
        VendorDefinitionDocument vdd;
        VendorDefinition vendorDef = null;
        try {
            vdd = createVendorDefDoc();
            vendorDef = vdd.getVendorDefinition();
        } catch (Exception e) {
            GenericPlugin.logException(e);
        }
        ConnectionInfo info = getActiveConnectionInfo(properties);
        DatabaseDefinition dbdef = null;
        if (info != null) {
            dbdef = info.getDatabaseDefinition();
            if(dbdef != null && vendorDef != null){
                // find the best type table for our database
                String dbProduct = dbdef.getProduct();
                String dbVersion = dbdef.getVersion();
                JdbcTypeMap match = null;
                for (int i = 0; i < vendorDef.sizeOfJdbcTypeMapArray(); i++) {
                    JdbcTypeMap map = vendorDef.getJdbcTypeMapArray(i);
                    String database = map.getDatabase();
                    if (database.equalsIgnoreCase(dbProduct)) {
                        if (match == null) {
                            match = map;
                        }
                        if (dbVersion.equals(map.getVersion())) {
                            match = map;
                            break;
                        }
                    }
                }
                if (match == null) {
                    match = vendorDef.getJdbcTypeMapArray(0);
                }
                String dialect = match.getDialect();
                setPropertyValue(keyValue, "dialect", dialect);
                setPropertyValue(keyValue, "connection.driver", match.getDefaultDriverClass());
            }
            setPropertyValue(keyValue, "connection.driver", info.getDriverClassName());
            setPropertyValue(keyValue, "connection.url", info.getURL());
            setPropertyValue(keyValue, "connection.username", info.getUserName());
            setPropertyValue(keyValue, "connection.password", info.getPassword());
        }
        setPropertyValue(keyValue, "connection.username", properties.getConnectionUserName());
        setPropertyValue(keyValue, "connection.password", properties.getConnectionPassword());
        if(vendorDef != null){
            PersistenceXml persistenceXml = vendorDef
                    .getPersistenceXmlArray(0);
            PropertyMapping[] propertyMappings = persistenceXml.getPropertyMappingArray();
            for (int x = 0; x < propertyMappings.length; x++) {
                PropertyMapping mapping = propertyMappings[x];
                String propertyName = mapping.getPropertyName();
                String key = mapping.getKey();
                changeKey(keyValue, key, propertyName);
            }
        }
        TableItem[] items = tProperties.getItems();
        for (int x = 0; x < items.length; x++) {
            TableItem item = items[x];
            String key = item.getText(0);
            String value = keyValue.get(key);
            if(value != null){
                item.setText(1, value);
                Property prop = (Property) item.getData();
                prop.getValue().setStringValue(value);
                keyValue.remove(key);
            }
        }
        for(Iterator it = keyValue.entrySet().iterator(); it.hasNext();){
            Map.Entry entry = (Entry) it.next();
            String propName = (String) entry.getKey();
            String propValue = (String) entry.getValue();
            Property property = this.properties.addNewProperty();
            TableItem item = new TableItem(tProperties, SWT.NONE);
            item.setText(0, propName);
            item.setText(1, propValue);
            item.setData(property);
            property.addNewName().setStringValue(propName);
            property.addNewValue().setStringValue(propValue);
        }
    }

    private void setPropertyValue(HashMap<String, String> keyValue, 
            String key, String value) {
        if(key != null && value != null){
            keyValue.put(key, value);
        }
    }
    
    private void changeKey(HashMap<String, String> keyValue, 
            String oldKey, String newKey) {
        if(oldKey != null && newKey != null){
            String value = keyValue.get(oldKey);
            if (value != null) {
                keyValue.put(newKey, value);
                keyValue.remove(oldKey);
            }
        }
    }
    
    private void addProperty() {
        List<String> props = getVendorProperties();
        if (props.isEmpty()) {
            MessageDialog.openInformation(getShell(),
                    "No properties in vendor definition file.",
                    "No more properties in vendor definition file to add.");
            return;
        }
        ElementListSelectionDialog dialog = new ElementListSelectionDialog(
                getShell(), new LabelProvider());
        dialog.setTitle("Select a property.");
        dialog.setMessage("Select a property to add.");
        dialog.setElements(props.toArray());
        dialog.setAllowDuplicates(false);
        dialog.setMultipleSelection(true);
        dialog
                .setEmptyListMessage("No more properties in vendor definition file to add.");
        dialog.setEmptySelectionMessage("You must select a property to add.");
        if (dialog.open() == Window.OK) {
            VendorDefinitionDocument vdd;
            org.eclipse.jsr220Orm.generic.xml.VendorDefinitionDocument.VendorDefinition.PersistenceXml.Property[] properties = null;
            try {
                vdd = createVendorDefDoc();
                VendorDefinition vendorDef = vdd.getVendorDefinition();
                PersistenceXml persistenceXml = vendorDef
                        .getPersistenceXmlArray(0);
                properties = persistenceXml.getPropertyArray();
            } catch (Exception e) {
                GenericPlugin.logException(e);
            }
            Object[] results = dialog.getResult();
            for (int x = 0; x < results.length; x++) {
                String propName = (String) results[x];
                Property property = this.properties.addNewProperty();
                TableItem item = new TableItem(tProperties, SWT.NONE);
                item.setText(0, propName);
                item.setData(property);
                property.addNewName().setStringValue(propName);
                property.addNewValue();
                if (properties != null) {
                    for (int y = 0; y < properties.length; y++) {
                        String propertyName = properties[x].getName();
                        if (propName.equals(propertyName)) {
                            String value = properties[x].getValue();
                            item.setText(1, value);
                            property.getValue().setStringValue(value);
                            break;
                        }
                    }
                }
            }
        }
        bAdd.setEnabled(!getVendorProperties().isEmpty());
  }

    private List<String> getVendorProperties() {
        ArrayList<String> props = new ArrayList<String>();
        VendorDefinitionDocument vdd;
        try {
            vdd = createVendorDefDoc();
        } catch (Exception e) {
            GenericPlugin.logException(e);
            return props;
        }
        VendorDefinition vendorDef = vdd.getVendorDefinition();
        PersistenceXml persistenceXml = vendorDef.getPersistenceXmlArray(0);
        PropertyMapping[] propertyMappings = persistenceXml
                .getPropertyMappingArray();
        for (int x = 0; x < propertyMappings.length; x++) {
            PropertyMapping mapping = propertyMappings[x];
            String propertyName = mapping.getPropertyName();
            if (!props.contains(propertyName)) {
                props.add(propertyName);
            }
        }
        org.eclipse.jsr220Orm.generic.xml.VendorDefinitionDocument.VendorDefinition.PersistenceXml.Property[] properties = persistenceXml
                .getPropertyArray();
        for (int x = 0; x < properties.length; x++) {
            String propertyName = properties[x].getName();
            if (!props.contains(propertyName)) {
                props.add(propertyName);
            }
        }
        TableItem[] items = tProperties.getItems();
        for (int x = 0; x < items.length; x++) {
            TableItem item = items[x];
            props.remove(item.getText(0));
        }
        return props;
    }

    protected VendorDefinitionDocument createVendorDefDoc()
            throws XmlException, IOException, CoreException {
        PersistenceProperties properties = new PersistenceProperties(
                project);
        String vendorDefRes = properties.getProperty("ejb3.vendor.definition");
        if (vendorDefRes == null) {
            vendorDefRes = "vendor-generic.xml";
        }
        String namespace = GenericPlugin.getPluginId();
        String productId = properties.getPersistenceProduct();
        if (productId != null) {
            OrmProduct ormProduct = OrmProductRegistry.INSTANCE
                    .getOrmProductById(productId);
            if (ormProduct != null) {
                namespace = ormProduct.getNamespace();
            }
        }
        VendorDefinitionDocument vdd = Utils.loadVendorDefinition(namespace, vendorDefRes);
        return vdd;
    }

    private void newProperty() {
        TableItem item = new TableItem(tProperties, SWT.NONE);
        Property property = this.properties.addNewProperty();
        property.addNewName();
        property.addNewValue();
        item.setData(property);
    }

    private void removeProperty() {
        int[] selectionIndices = tProperties.getSelectionIndices();
        for (int x = 0; x < selectionIndices.length; x++) {
            TableItem item = tProperties.getItem(selectionIndices[x]);
            Property property = (Property) item.getData();
            if (property != null) {
                Property[] propertyArray = this.properties.getPropertyArray();
                for (int y = 0; y < propertyArray.length; y++) {
                    Property temp = propertyArray[y];
                    if (property.equals(temp)) {
                        this.properties.removeProperty(y);
                    }
                }
            }
        }
        if (selectionIndices != null && selectionIndices.length > 0) {
            tProperties.remove(selectionIndices);
        }
        bAdd.setEnabled(!getVendorProperties().isEmpty());
  }

    public ConnectionInfo getActiveConnectionInfo(PersistenceProperties props) {
        String rdbConnName = props.getConnectionName();
        if (rdbConnName != null) {
            ConnectionInfo[] rdbConnections = RdbUtils.getRdbConnections();
            for (int x = rdbConnections.length - 1; x >= 0; x--) {
                if (rdbConnName.equals(rdbConnections[x].getName())) {
                    return rdbConnections[x];
                }
            }
        }
        return null;
    }
}
