/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.generic.io;

import java.io.IOException;
import java.io.InputStream;
import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.apache.xmlbeans.XmlException;
import org.eclipse.jsr220Orm.generic.xml.AnnotationsDocument;
import org.eclipse.jsr220Orm.generic.xml.AnnotationsDocument.Annotations.Ann;
import org.eclipse.jsr220Orm.generic.xml.AnnotationsDocument.Annotations.Ann.Value;
import org.eclipse.jsr220orm.core.OrmPlugin;
import org.eclipse.jsr220orm.core.util.JdbcUtils;
import org.eclipse.jsr220orm.generic.Utils;

/**
 * Registry of known annotations. Methods are provided to translate between
 * String names and int codes for use in switch statements and so on.
 */
public class AnnotationRegistry {
	
	protected Map<String, AnnInfo> nameInfoMap = new HashMap();
	protected Map<Class, AnnInfo> classInfoMap = new HashMap();
	protected Set<Class> enumSet = new HashSet(); // enums found so far
	protected Map<String, Object> namedValueMap = new HashMap(); 
		// qualified name -> instance	
	
    protected static final String ANNOTATION_REGISTRY_XML_NAMESPACE = 
        "http://jsr220orm.eclipse.org/generic/xml";
	
	/**
	 * Information we keep about an annotation.
	 */
	protected static class AnnInfo {
		public Class annotationType;
		public boolean marker;
		public boolean singleValue;
		public Object defaultProxy;
		public Map defaultValueMap = new HashMap();
		public int index;
		public Map<String, Integer> valueIndexMap;
	}
	
	public AnnotationRegistry() {
		registerJavaSqlTypes();
	}

	/**
	 * Load our info.
	 */
	public void load(String namespace) throws IOException, XmlException {
    	InputStream ins = Utils.openBundleResource(namespace, 
    		"annotations.xml");
    	try {
    		AnnotationsDocument doc = AnnotationsDocument.Factory.parse(ins, 
    				Utils.getXmlOptions(ANNOTATION_REGISTRY_XML_NAMESPACE));
    		int c = 0;
    		for (Ann ann : doc.getAnnotations().getAnnArray()) {
				String name = ann.getName();
    			try {
					Class cls = Class.forName(name);
					AnnInfo info = add(cls);
					info.index = c++;
					info.marker = ann.getMarker();
					Value[] values = ann.getValueArray();
					if (values.length == 0) {
						info.valueIndexMap = Collections.EMPTY_MAP;
					} else {
						info.valueIndexMap = new HashMap();
						for (int i = 0; i < values.length; i++) {
							info.valueIndexMap.put(values[i].getName(), i);
						}
					}
				} catch (Exception e) {
					OrmPlugin.log(name + ": " + e, e);
				}
			}
    	} finally {
    		Utils.close(ins);
    	}
	}
	
	protected AnnInfo add(Class cls) {
		AnnInfo info = new AnnInfo();
		info.annotationType = cls;
		info.defaultValueMap = new HashMap();
		String fullName = cls.getName();
		String shortName = cls.getSimpleName();
		nameInfoMap.put(shortName, info);
		nameInfoMap.put(fullName, info);
		classInfoMap.put(info.annotationType, info);
		registerEnumsAndDefaults(info);
		return info;
	}
	
	/**
	 * Find all enum values of annotationCls and register them so we can
	 * quickly resolve their values from String's. Also registers the
	 * defaults for all values.
	 */
	protected void registerEnumsAndDefaults(AnnInfo info) {
		Method[] a = info.annotationType.getDeclaredMethods();
		for (int i = a.length - 1; i >= 0; i--) {
			Method m = a[i];
			Class rt = m.getReturnType();
			if (rt.isArray()) {
				rt = rt.getComponentType();
			}
			if (rt.isEnum() && !enumSet.contains(rt)) {
				enumSet.add(rt);
				String fullName = rt.getName();
				String shortName = rt.getSimpleName();
				Object[] values = rt.getEnumConstants();
				for (int j = values.length - 1; j >= 0; j--) {
					Enum e = (Enum)values[j];
					namedValueMap.put(shortName + "." + e.name(), e);
					namedValueMap.put(fullName + "." + e.name(), e);
				}
			}
			Object defaultValue = m.getDefaultValue();
			if (defaultValue != null) {
				info.defaultValueMap.put(m.getName(), defaultValue);
			}
		}
		info.singleValue = a.length == 1 && a[0].getName().equals("value");
	}
	
	/**
	 * Find and register all of the constants from java.sql.Types. 
	 */
	protected void registerJavaSqlTypes() {
		Map map = JdbcUtils.getJdbcNameIntMap();
		for (Iterator i = map.entrySet().iterator(); i.hasNext(); ) {
			Map.Entry e = (Map.Entry)i.next();
			namedValueMap.put((String)e.getKey(), e.getValue());
		}
	}
	
	protected AnnInfo getInfo(String name) {
		return (AnnInfo)nameInfoMap.get(name);
	}
	
	protected AnnInfo getInfo(Class annotationType) {
		return (AnnInfo)classInfoMap.get(annotationType);
	}
	
	/**
	 * Lookup the Class of the annotation for name (short or full).
	 */
	public Class getClass(String name) {
		AnnInfo i = getInfo(name);
		return i == null ? null : i.annotationType;
	}
	
	/**
	 * Convert an named value (e.g. AccessType.FIELD or Types.VARCHAR) into an 
	 * instance of the enum or some other object or null if not found.
	 */
	public Object getNamedValue(String name) {
		return namedValueMap.get(name);
	}
	
	/**
	 * Return an implementation of the annotation that provides the default
	 * values.
	 */
	public <T extends Annotation> T getDefaultProxy(Class<T> annotationType) {
		AnnInfo i = getInfo(annotationType);
		if (i.defaultProxy == null) {
			i.defaultProxy = Proxy.newProxyInstance(annotationType.getClassLoader(),
				new Class[]{annotationType}, new DefaultHandler(
						"Defaults:" + annotationType.getName()));
		}
		return (T)i.defaultProxy;
	}
	
	/**
	 * Return an implementation of the annotation that provides the default
	 * values and also implements parts of AnnotationEx. Only 
	 * {@link AnnotationEx#setDefault(String, Object)} is supported.
	 */
	public <T extends Annotation> T getDefaultProxyEx(Class<T> annotationType) {
		return (T)Proxy.newProxyInstance(annotationType.getClassLoader(),
					new Class[]{annotationType, AnnotationEx.class},
			new DefaultHandlerEx(annotationType));
	}
	
	/**
	 * Get the ordering index for the annotation class name. 
	 */
	public int getOrderingIndex(String annotationTypeName) {
		AnnInfo i = getInfo(annotationTypeName);
		return i == null ? 10000 : i.index;		
	}
	
	/**
	 * Get the ordering map for the values of annotationType.
	 */
	public Map<String, Integer> getOrderingMap(Class annotationType) {
		AnnInfo i = getInfo(annotationType);
		return i == null ? Collections.EMPTY_MAP : i.valueIndexMap;				
	}
	
	/**
	 * Can this annotation be a marker annotation? If this returns false
	 * and the annotation ends up empty it should be removed from the
	 * source. 
	 */
	public boolean isMarker(Class annotationType) {
		AnnInfo i = getInfo(annotationType);
		return i != null && i.marker;
	}
	
	/**
	 * Can this annotation be a single value annotation?
	 */
	public boolean isSingleValue(Class annotationType) {
		AnnInfo i = getInfo(annotationType);
		return i != null && i.singleValue;
	}
	
	/**
	 * Get the default for name for the annotationType.
	 */
	public Object getDefault(Class annotationType, String name) {
		return getInfo(annotationType).defaultValueMap.get(name);
	}
	
	/**
	 * This just returns the default value for each method and is used to
	 * construct an instance of an annotation that returns the defaults
	 * only.
	 */
	private static class DefaultHandler implements InvocationHandler {
		
		final String name;
		
		DefaultHandler(String name) {
			this.name = name;
		}
		
		public Object invoke(Object proxy, Method method, Object[] args) 
				throws Throwable {
			if ("toString".equals(method.getName())) {
				return name;
			}
			return method.getDefaultValue();
		}

	}
		
	/**
	 * This just returns the default value for each method and is used to
	 * construct an instance of an annotation that returns the defaults
	 * only. It also supports {@link AnnotationEx#setDefault(String, Object)}
	 * and {@link AnnotationEx#get(String)}.
	 */
	protected static class DefaultHandlerEx implements InvocationHandler {
		
		protected final Class annotationType;
		protected Map defaultMap;
		
		public DefaultHandlerEx(Class annotationType) {
			this.annotationType = annotationType;
		}
		
		public Object invoke(Object proxy, Method method, Object[] args) 
				throws Throwable {
			String methodName = method.getName();
			if (args != null && args.length == 1 && "get".equals(methodName)) {
				method = annotationType.getDeclaredMethod(
						methodName = (String)args[0], (Class[])null);
				args = null;
			}
			if (args == null) {
				if ("toString".equals(methodName)) {
					return "DefaultsEx:" + annotationType.getName() + "@" + 
						Integer.toHexString(System.identityHashCode(this));
				}
				if (defaultMap != null && defaultMap.containsKey(methodName)) {
					return defaultMap.get(methodName);
				}
			} else if (args.length == 2 && "setDefault".equals(methodName)) {
				if (defaultMap == null) {
					defaultMap = new HashMap();
				}
				defaultMap.put(args[0], args[1]);
				return null;
			}
			return method.getDefaultValue();
		}
		
	}
	
}
