package org.eclipse.jsr220orm.generic.io;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.persistence.JoinColumn;
import javax.persistence.PrimaryKeyJoinColumn;

import org.eclipse.jsr220orm.generic.GenericEntityModelManager;
import org.eclipse.jsr220orm.metadata.Join;
import org.eclipse.jsr220orm.metadata.JoinPair;
import org.eclipse.jsr220orm.metadata.OrmColumn;
import org.eclipse.jsr220orm.metadata.OrmTable;

/**
 * Handles Join, JoinColumn, JoinColumns, PrimaryKeyJoinColumn etc. 
 */
public class JoinIO {
	
	protected final EntityIO entityIO;

	public JoinIO(EntityIO entityIO) {
		this.entityIO = entityIO;
	}
	
	/**
	 * Update a Join in the model from annotations. If the join is null if is 
	 * created and returned. If there are problems the join (if any) is
	 * deleted and null is returned.
	 * 
	 * @param metaDataChanged The meta data under the annotations is known
	 * 			to have changed
	 * @param srcTable The referencing table
	 * @param destTable The referenced table
	 * @param columnNameStrategy Strategy for generating default name of a 
	 * 			src column from a corresponding dest column
	 */
	protected Join updateModelJoin(Join join, AnnotationEx joinColumn, 
			Object[] joinColumns, Map joinColumnsLocation, 
			boolean metaDataChanged, 
			OrmTable srcTable, OrmTable destTable, 
			ColumnNameStrategy columnNameStrategy,
			String comment, boolean nullable, int relativePositionInTable) {
		
		GenericEntityModelManager mm = entityIO.getModelManager();
		
		String prefix = "";
		if (joinColumn instanceof PrimaryKeyJoinColumn
				|| joinColumns instanceof PrimaryKeyJoinColumn[]) {
			prefix = "PrimaryKey";
		}
		
		List targetPk = destTable.getPrimaryKeyList();
		
		join = ensureJoin(join);
		join.setComment(comment);
		
		boolean destTableChanged = join.getPairList().size() > 0 
			&& join.getDestTable() != destTable;
		
		boolean refToNonPk = false;
		boolean problems = false;
		List<OrmColumn> destCols = new ArrayList();
		List<AnnotationEx> jcAnnotations = new ArrayList();
		
		if (joinColumns != null) {
			for (int i = 0; i < joinColumns.length; i++) {
				AnnotationEx jc = (AnnotationEx)joinColumns[i];
				OrmColumn dest;
				String refColName = (String)jc.get("referencedColumnName");
				if (refColName.length() == 0) {
					if (joinColumns.length > 1 || targetPk.size() != 1) {
						problems = true;
						entityIO.addProblem("The referencedColumnName value " +
								"is required if there is more than one column " +
								"in the join", jc);
						continue;
					} else {
						dest = (OrmColumn)targetPk.get(0);
					}
				} else {
					dest = destTable.findColumn(refColName);
					if (dest == null && !(metaDataChanged ||destTableChanged)) {
						dest = getDest(join, (String)jc.get("name"), i);
						if (dest != null) {
							entityIO.registerForMetaDataUpdate();
						}
					}
					if (dest == null) {
						problems = true;
						entityIO.addProblem("Column '" + refColName + 
								"' not found on table '" + destTable.getName() + 
								"'", jc, "referencedColumnName");
						continue;
					}
				}
				if (!dest.isPrimaryKey()) {
					refToNonPk = true;
				}
				destCols.add(dest);
				jcAnnotations.add(jc);
			}
			if (!refToNonPk && !problems && destCols.size() != targetPk.size()) {
				problems = true;
				entityIO.addProblem(
						"Referenced table has " + targetPk.size() + " primary " +
						"key column(s) but there are " + destCols.size() + 
						" " + prefix + "JoinColumn(s)", joinColumnsLocation);
			}
			if (joinColumn != null) {
				entityIO.addProblem("Unexpected JoinColumn", joinColumn);
			}
		} else {
			if (joinColumn == null) {
				for (int i = 0; i < targetPk.size(); i++) {
					joinColumn = (AnnotationEx)mm.getAnnotationRegistry().
						getDefaultProxyEx(JoinColumn.class);
					jcAnnotations.add(joinColumn);
					destCols.add((OrmColumn)targetPk.get(i));					
				}
			} else {
				jcAnnotations.add(joinColumn);					
				String refColName = (String)joinColumn.get("referencedColumnName");
				if (refColName.length() == 0) {
					if (targetPk.size() != 1) {
						problems = true;
						entityIO.addProblem(
								"Referenced table has " + targetPk.size() + 
								" primary key columns so " + prefix + 
								"JoinColumns is " +
								"required for a reference to its PK", joinColumn);
					} else {
						destCols.add((OrmColumn)targetPk.get(0));	
					}
				} else {
					OrmColumn dest = destTable.findColumn(refColName);
					if (dest == null && !(metaDataChanged || destTableChanged)) {
						dest = getDest(join, (String)joinColumn.get("name"), 0);
						if (dest != null) {
							entityIO.registerForMetaDataUpdate();
						}
					}					
					if (dest == null) {
						problems = true;
						entityIO.addProblem("Column '" + refColName + 
								"' not found on table '" + destTable.getName() + 
								"'", joinColumn, "referencedColumnName");
					} else {
						destCols.add(dest);
						if (!dest.isPrimaryKey()) {
							refToNonPk = true;
						}						
					}
				}
			}
		}

		if (problems) {
			List srcColumns = join.getSrcColumns();
			join.delete();
			for (Object c : srcColumns) {
				((OrmColumn)c).delete();
			}
			join = null;
		} else {
			Set pairsNotFound = new HashSet();
			pairsNotFound.addAll(join.getPairList());
			
			int n = destCols.size();
			for (int i = 0; i < n; i++) {
				OrmColumn dest = destCols.get(i);
				JoinPair p = join.findPairForDest(dest);
				if (p == null) {
					p = mm.getFactory().createJoinPair();
					p.setDest(dest);
					join.getPairList().add(p);
				} else {
					pairsNotFound.remove(p);
				}
				OrmColumn src = p.getSrc();
				if (src == null) {
					p.setSrc(src = entityIO.createOrmColumn());
				}
				src.setRelativePositionInTable(relativePositionInTable + i);
							
				AnnotationEx jc = jcAnnotations.get(i);
				((AnnotationEx)jc).setDefault("name", 
						columnNameStrategy.getColumnName(dest));
				((AnnotationEx)jc).setDefault("nullable", nullable);	
				
				src.setName((String)jc.get("name"));
				if (jc instanceof JoinColumn) {
					src.setNullable((Boolean)jc.get("nullable"));
					src.setUpdatable((Boolean)jc.get("updatable"));
					src.setInsertable((Boolean)jc.get("insertable"));
				}
				src.setComment(join.getComment());
				src.setJdbcType(dest.getJdbcType());
				src.setLength(dest.getLength());
				src.setScale(dest.getScale());
				src.setTable(srcTable);
				String cd = (String)jc.get("columnDefinition");
				if (cd.length() == 0) {
					src.setColumnDefinitionSpecified(false);
					cd = mm.getColumnDefinition(src);
				} else {
					src.setColumnDefinitionSpecified(true);
				}
				src.setColumnDefinition(cd);
				src.setOriginalColumnDefinition(cd);
				mm.updateDatabaseType(src);
			}
			
			// get rid of all the JoinPair's we did not find and their src cols
			if (!pairsNotFound.isEmpty()) {
				for (Iterator i = pairsNotFound.iterator(); i.hasNext(); ) {
					OrmColumn src = ((JoinPair)i.next()).getSrc();
					if (src != null) {
						src.delete(); // this will delete the JoinPair as well
					}
				}
			}
		}
		return join;
	}
	
	/**
	 * If srcColName is not null then return the dest for the pair with
	 * matching src or null if none. Otherwise get the dest column for the 
	 * JoinPair at index or null if index is out of range.
	 */
	protected OrmColumn getDest(Join join, String srcColName, int index) {
		List pairList = join.getPairList();
		if (index < pairList.size()) {
			return ((JoinPair)pairList.get(index)).getDest();
		} else {
			return null;
		}
	}

	/**
	 * If j is null then create a new Join and add our entityIO to its
	 * adapter list so that it receives model events. Return j or the newly
	 * created Join. 
	 */
	public Join ensureJoin(Join j) {
		if (j == null) {
			j = entityIO.getModelManager().getFactory().createJoin();
			j.eAdapters().add(entityIO);			
		}
		return j;
	}
	
	/**
	 * Update the JoinColumn or PrimaryKeyJoinColumn annotation jc to match p.
	 */
	public void updateMetaDataJoinColumn(AnnotationEx jc, Join join, JoinPair p,
			ColumnNameStrategy columnNameStrategy, boolean defaultNullable) {
		
		GenericEntityModelManager mm = entityIO.getModelManager();		
		
		OrmColumn dest = p.getDest();
		OrmColumn src = p.getSrc();
		
		jc.setDefault("name", columnNameStrategy.getColumnName(dest));
		jc.setDefault("nullable", defaultNullable);
		
		jc.set("name", src.getName(), true);
		src.setName((String)jc.get("name"));
		
		// dont try to set these for a PrimaryKeyJoinColumn
		if (jc instanceof JoinColumn) {
			jc.set("insertable", src.isInsertable());
			jc.set("nullable", src.isNullable());
			jc.set("updatable", src.isUpdatable());
		}
		
		// referencedColumnName is required if there is more than one col
		if (!dest.isPrimaryKey() || join.getPairList().size() > 1) {
			jc.set("referencedColumnName", dest.getName());
		} else {
			jc.set("referencedColumnName", null);
		}

		// if the columnDefinition has changed then apply the change to the
		// annotation - this may remove it if it now matches the default
		boolean colDefChanged = !src.getColumnDefinition().equals(
				src.getOriginalColumnDefinition());
		String stdColDef =  mm.getColumnDefinition(src);
		if (colDefChanged) {
			jc.setDefault("columnDefinition", stdColDef);			
			jc.set("columnDefinition", src.getColumnDefinition(), true);
		}
		
		// if the annotation now has no columnDefinition value then put
		// the default value into the model
		if (!jc.hasValue("columnDefinition")) {
			src.setColumnDefinition(stdColDef);
		}
		
		// reset the originalColumnDefinition attribute to detect more changes
		src.setOriginalColumnDefinition(src.getColumnDefinition());		

		mm.updateDatabaseType(src);
	}		

}
