/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.ui.internal.views;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IProject;
import org.eclipse.emf.edit.provider.ComposedAdapterFactory;
import org.eclipse.emf.edit.provider.ReflectiveItemProviderAdapterFactory;
import org.eclipse.emf.edit.provider.resource.ResourceItemProviderAdapterFactory;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.internal.ui.javaeditor.EditorUtility;
import org.eclipse.jdt.ui.JavaUI;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jsr220orm.metadata.AttributeMetaData;
import org.eclipse.jsr220orm.metadata.EntityMetaData;
import org.eclipse.jsr220orm.metadata.MetadataElement;
import org.eclipse.jsr220orm.metadata.provider.MetadataItemProviderAdapterFactory;
import org.eclipse.jsr220orm.ui.OrmUiPlugin;
import org.eclipse.jsr220orm.ui.internal.views.provider.OutlineContentProvider;
import org.eclipse.jsr220orm.ui.internal.views.provider.OutlineLabelProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.views.properties.PropertySheetPage;

public class PersistenceOutline extends AbstractMetadataView implements
        ISelectionProvider {

    public static final String ID = "org.eclipse.jsr220orm.ui.internal.views.PersistenceOutline";

    private TreeViewer treeViewer;

    /**
     * This viewer's control.
     */
    private Tree tree;

    private MetadataElement rootElement;

    private PropertySheetPage propertySheetPage;

    private ComposedAdapterFactory adapterFactory;

    private boolean busy;

    public void createPartControl(Composite parent) {
        GridLayout parentLayout = new GridLayout();
        parent.setLayout(parentLayout);
        GridData treeViewerLData = new GridData();
        treeViewerLData.grabExcessHorizontalSpace = true;
        treeViewerLData.grabExcessVerticalSpace = true;
        treeViewerLData.horizontalAlignment = GridData.FILL;
        treeViewerLData.verticalAlignment = GridData.FILL;
        treeViewer = new TreeViewer(parent, SWT.NONE);
        tree = (Tree) treeViewer.getControl();
        tree.setLayoutData(treeViewerLData);
        List factories = new ArrayList();
        factories.add(new ResourceItemProviderAdapterFactory());
        factories.add(new MetadataItemProviderAdapterFactory());
        factories.add(new ReflectiveItemProviderAdapterFactory());
        adapterFactory = new ComposedAdapterFactory(factories);

        treeViewer
                .setContentProvider(new OutlineContentProvider(adapterFactory));
        treeViewer.setLabelProvider(new OutlineLabelProvider(adapterFactory));
        treeViewer.addSelectionChangedListener(new ISelectionChangedListener() {

            public void selectionChanged(SelectionChangedEvent event) {
                if (getSite().getPage().getActivePart() == PersistenceOutline.this) {
                    try {
                        navigateToSource(event.getSelection());
                    } catch (Exception e) {
                        OrmUiPlugin.logException(e, getSite().getShell());
                    }
                }
            }

        });
    }

    protected void navigateToSource(ISelection selection)
            throws JavaModelException, PartInitException {
        IStructuredSelection sSel = (IStructuredSelection) selection;
        Object obj = sSel.getFirstElement();
        if (obj instanceof MetadataElement) {
            MetadataElement element = (MetadataElement) obj;
            IProject project = (IProject) element.adapt(IProject.class);
            EntityMetaData entity = (EntityMetaData) element
                    .getAncestor(EntityMetaData.class);
            if (project == null || entity == null) {
                return;
            }
            IJavaProject javaProject = JavaCore.create(project);
            if (javaProject == null) {
                return;
            }
            IType iType = javaProject.findType(entity.getClassName());
            if (iType == null) {
                return;
            }
            IJavaElement javaElement = iType;
            AttributeMetaData att = (AttributeMetaData) element
                    .getAncestor(AttributeMetaData.class);
            if (att != null) {
                if (att.isField()) {
                    javaElement = iType.getField(att.getName());
                } else {
                    javaElement = iType.getMethod(att.getGetterMethodName(),
                            new String[0]);
                }
            }
            if (javaElement != null) {
                treeViewer.getControl().setRedraw(false);
                busy = true;
                try {
                	IEditorPart openInEditor = 
                		EditorUtility.openInEditor(javaElement, false);
//                    IEditorPart openInEditor = JavaUI.openInEditor(javaElement);
                    JavaUI.revealInEditor(openInEditor, javaElement);
                } finally {
                    busy = false;
                    treeViewer.getControl().setRedraw(true);
                }
            }
        }

    }

    public void setFocus() {
        treeViewer.getControl().setFocus();
    }

    private void setTreeRoot(MetadataElement element) {
        if (treeViewer == null) {
            return;
        }
        rootElement = element;
        Object[] roots = element == null 
        	? new Object[0] 
            : new Object[] { element };
        treeViewer.setInput(roots);
        treeViewer.setExpandedElements(roots);
    }

    protected void doUpdate(MetadataElement metadataElement) {
        if (busy) {
            return;
        }
        MetadataElement element;
        if (metadataElement == null) {
        	element = null;
        } else {
	        element = metadataElement.getAncestor(
	        		EntityMetaData.class);
	        if (element == null) {
	            element = metadataElement;
	        }
        }
        if (element == null || !element.equals(rootElement)) {
            setTreeRoot(element);
        }
        if (treeViewer != null && element != null) {
            IStructuredSelection sSel = 
            	(IStructuredSelection)treeViewer.getSelection();
            if (sSel != null) {
                if (sSel.getFirstElement() == metadataElement) {
                    return;
                }
            }
            treeViewer.setSelection(new StructuredSelection(metadataElement));
        }
    }

    // /**
    // * This is how the framework determines which interfaces we implement.
    // */
    // public Object getAdapter(Class key) {
    // if (key.equals(IPropertySheetPage.class)) {
    // return getPropertySheetPage();
    // }
    // return super.getAdapter(key);
    // }
    //
    // /**
    // * This accesses a cached version of the property sheet.
    // */
    // public IPropertySheetPage getPropertySheetPage() {
    // if (propertySheetPage == null) {
    // propertySheetPage = new PropertySheetPage();
    // propertySheetPage.setPropertySourceProvider(new
    // AdapterFactoryContentProvider(adapterFactory));
    // treeViewer.addSelectionChangedListener(new ISelectionChangedListener() {
    //            
    // public void selectionChanged(SelectionChangedEvent event) {
    // propertySheetPage.selectionChanged(PersistenceOutline.this,
    // event.getSelection());
    // }
    //            
    // });
    // }
    //
    // return propertySheetPage;
    // }

    public void addSelectionChangedListener(ISelectionChangedListener listener) {
        treeViewer.addSelectionChangedListener(listener);
    }

    public void removeSelectionChangedListener(
            ISelectionChangedListener listener) {
        treeViewer.removeSelectionChangedListener(listener);
    }

    public ISelection getSelection() {
        return treeViewer.getSelection();
    }

    public void setSelection(ISelection selection) {
        treeViewer.setSelection(selection);
    }

    public void dispose() {
        adapterFactory.dispose();
        if (propertySheetPage != null) {
            propertySheetPage.dispose();
        }
        super.dispose();
    }

}
