/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.ui.internal.wizards;

import java.lang.reflect.Constructor;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.ui.JavaElementLabelProvider;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.jsr220orm.core.internal.product.OrmProduct;
import org.eclipse.jsr220orm.ui.IProductActivationWizard;
import org.eclipse.jsr220orm.ui.OrmUiPlugin;
import org.eclipse.jsr220orm.ui.internal.properties.SelectProductComposite;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;
import org.osgi.framework.Bundle;

public class AddPersistencePage extends WizardPage {

    private IProject project;
    private IJavaProject javaProject;
    private SelectProductComposite prodSelect;
    private Text fProjectField;
    private IWorkspaceRoot fWorkspaceRoot = ResourcesPlugin.getWorkspace().getRoot();
    private IProductActivationWizard selectedWizard;

    public AddPersistencePage() {
       super("addPersistence", 
               "Add Persistence to project.", 
               OrmUiPlugin.getImageDescriptor("icons/orm16.png"));
    }

    public void setIProject(IProject project) {
       this.project = project;
    }

    public void createControl(Composite parent) {
        Composite composite = new Composite(parent, SWT.NONE);
        GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 3;
        composite.setLayout(gridLayout);
        setControl(composite);
        Label label = new Label(composite, SWT.NONE);
        fProjectField = new Text (composite, SWT.BORDER);
        GridData gridData = new GridData();
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalAlignment = GridData.FILL;
        fProjectField.setLayoutData(gridData);
        Button button = new Button (composite, SWT.NONE);
        this.prodSelect = new SelectProductComposite(composite, true);
        gridData = new GridData();
        gridData.horizontalSpan = 3;
        this.prodSelect.setLayoutData(gridData);
        label.setText("Project");
        button.setText("Browse");
        button.addSelectionListener(new SelectionListener() {
            public void widgetDefaultSelected(SelectionEvent e) {
            }
            public void widgetSelected(SelectionEvent e) {
                IJavaProject jproject= chooseProject();
                if (jproject != null) {
                    IPath path= jproject.getProject().getFullPath().makeRelative();
                    fProjectField.setText(path.toString());
                }
            }
        });
        fProjectField.addModifyListener(new ModifyListener() {
            public void modifyText(ModifyEvent e) {
                updateProjectStatus();
            }
        });
        if(project != null){
            IPath path= project.getProject().getFullPath().makeRelative();
            fProjectField.setText(path.toString());
        }
    }
    
    private void updateProjectStatus() {
        IProject project = getIProjectFromText();
        if(prodSelect != null){
            try {
                prodSelect.setProject(project);
            } catch (Exception e) {
                OrmUiPlugin.logException(e, "Problem loading presistence settings.", getShell());
            }
        }
    }
    
    private IProject getIProjectFromText() {
        if(fProjectField == null){
            return null;
        }
        String str= fProjectField.getText();
        if (str.length() == 0) {
            setErrorMessage("Please select a project."); 
            return null;
        }
        IPath path= new Path(str);
        if (path.segmentCount() != 1) {
            setErrorMessage("Please select a valid project path."); 
            return null;
        }
        IProject project= fWorkspaceRoot.getProject(path.toString());
        if (!project.exists()) {
            setErrorMessage("Project does not exists."); 
            return null;
        }
        setErrorMessage(null);
        return project;
    }

    private IJavaProject chooseProject() {
        IJavaProject[] projects;
        try {
            projects= JavaCore.create(fWorkspaceRoot ).getJavaProjects();
        } catch (JavaModelException e) {
            OrmUiPlugin.logException(e, "Could not find Java Projects", getShell());
            projects= new IJavaProject[0];
        }
        
        ILabelProvider labelProvider= new JavaElementLabelProvider(JavaElementLabelProvider.SHOW_DEFAULT);
        ElementListSelectionDialog dialog= new ElementListSelectionDialog(getShell(), labelProvider);
        dialog.setTitle("Select a Project"); 
        dialog.setMessage("Select a Project"); 
        dialog.setElements(projects);
        dialog.setInitialSelections(new Object[] { javaProject });
        if (dialog.open() == Window.OK) {           
            return (IJavaProject) dialog.getFirstResult();
        }           
        return null;        
    }

    public boolean canFlipToNextPage() {
        OrmProduct selectedProduct = getSelectedProduct();
        return selectedProduct != null || true;
    }

    public OrmProduct getSelectedProduct() {
        return prodSelect.getSelectedProduct();
    }

    public boolean performFinish() {
        if(selectedWizard != null){
            if(!selectedWizard.performFinish()){
                return false;
            }
        }
       return prodSelect.setProductOnProject();
    }

    public void flushProps() {
        prodSelect.flushProps();
    }

    public IWizardPage getNextPage() {
        flushProps();
        selectedWizard = getNextIWizard();
        if(selectedWizard != null){
            return selectedWizard.getStartingPage();
        }
        return super.getNextPage();
    }
    
    private IProductActivationWizard getNextIWizard() {
        flushProps();
        OrmProduct ormProduct = getSelectedProduct();
        if(ormProduct == null){
            return null;
        }
        IExtensionRegistry pluginRegistry = Platform.getExtensionRegistry();
        IExtensionPoint extensionPoint = pluginRegistry.getExtensionPoint(
                "org.eclipse.jsr220orm.ui", "productActivationWizard");
        if (extensionPoint != null) {
            IExtension[] extensions = extensionPoint.getExtensions();
            for (int i = 0; i < extensions.length; ++i) {
                IExtension extension = extensions[i];
                if(extension.getNamespace().equals(ormProduct.getNamespace())){
                    IConfigurationElement[] configElements = extension
                    .getConfigurationElements();
                    for (int j = 0; j < configElements.length; ++j) {
                        IConfigurationElement configurationElement = configElements[j];
                        if (configurationElement.getName().equals("wizard")) {
                            String className = configurationElement.getAttribute("class");
                            Class clazz;
                            try {
                                Bundle bundle = Platform.getBundle(ormProduct.getNamespace());
                                clazz = bundle.loadClass(className);
                                IProductActivationWizard activationPage;
                                try {
                                    activationPage = (IProductActivationWizard) clazz.newInstance();
                                } catch (Exception e) {
                                    try {
                                        Constructor constructor = clazz
                                                .getConstructor(new Class[] { String.class });
                                        activationPage = (IProductActivationWizard) constructor
                                                .newInstance(new Object[] { ormProduct
                                                        .getName() });
                                    } catch (Exception e2) {
                                        throw e;
                                    }
                                }
                                activationPage.setIProject(project);
                                activationPage.addPages();
                                return activationPage;
                            } catch (Exception e) {
                                String message = className + " could not be created.";
                                OrmUiPlugin.log(new Status(IStatus.ERROR, OrmUiPlugin.getPluginId(), 
                                        0, message, e));
                                break;
                            }
                        }
                    }
                }
            }
        }
        return null;
    }

    /** 
     * The <code>WizardSelectionPage</code> implementation of an <code>IDialogPage</code>
     * method disposes of all nested wizards. Subclasses may extend.
     */
    public void dispose() {
        super.dispose();
        // notify nested wizard
        if(selectedWizard != null){
            selectedWizard.dispose();
        }
    }

    /**
     * Returns the currently selected wizard node within this page.
     * 
     * @return the wizard node, or <code>null</code> if no node is selected
     */
    public IProductActivationWizard getSelectedWizard() {
        return selectedWizard;
    }

    public boolean canFinish() {
        return selectedWizard == null || selectedWizard.canFinish();
    }
}
