/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* Copyright (C) 2000-2001 Ximian, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors: Hans Petter Jansson <hpj@ximian.com>
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <gnome.h>
#include "global.h"
#include <gal/e-table/e-table-scrolled.h>
#include <gal/e-table/e-table-memory.h>
#include <gal/e-table/e-table-memory-callbacks.h>
#include <gal/e-table/e-cell-text.h>
#include <gal/e-paned/e-hpaned.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include "tables.h"
#include "xml.h"
#include "validate.h"
#include "import-settings.h"
#include "export-settings.h"
#include "export-clients.h"

#define IMPORTS_COL_LAST        3
#define EXPORTS_COL_LAST        3
#define EXPORT_CLIENTS_COL_LAST 3

typedef gboolean  (XstValidatorFunc)(gpointer val);
typedef gchar    *(XstFilterFunc)   (gchar *val);
typedef void      (XstChangedFunc)  (gpointer val);

typedef enum
{
	XST_TABLE_COL_STRING,
	XST_TABLE_COL_STRING_BLANK,
	XST_TABLE_COL_STATE
}
XstTableColType;

typedef struct
{
	gchar *xml_tag;
	XstTableColType type;
	XstFilterFunc *filter;
	XstValidatorFunc *validator;
	XstChangedFunc *changed;
}
XstTableColDef;

static void set_mount_button_label_neg    (gboolean unmount);
static void mount_button_from_import_node (xmlNodePtr node);
static void export_set_writable           (gboolean writable);

extern XstTool *tool;

GtkWidget *imports_table_scrolled;
GtkWidget *exports_table_scrolled;
GtkWidget *export_clients_table_scrolled;

static const gchar *imports_spec =
"<ETableSpecification cursor-mode='line'>"

"  <ETableColumn model_col='0' _title='Host' expansion='1.0' minimum_width='40'"
"                resizable='true' cell='string' compare='string'/>"

"  <ETableColumn model_col='1' _title='Path' expansion='1.0' minimum_width='80'"
"                resizable='true' cell='string' compare='string'/>"

"  <ETableColumn model_col='2' _title='Mount Point' expansion='1.0' minimum_width='80'"
"                resizable='true' cell='string' compare='string'/>"

"  <ETableColumn model_col='3' _title='Mounted' expansion='1.0' minimum_width='16'"
"                resizable='true' cell='checkbox' compare='integer'/>"

"</ETableSpecification>";


static const gchar *imports_basic_state =
"<ETableState>"
"  <column source='0'/>"
"  <column source='1'/>"
"  <column source='2'/>"
"  <column source='3'/>"
"  <grouping>"
"    <leaf column='0' ascending='true'/>"
"  </grouping>"
"</ETableState>";


static const gchar *exports_spec =
"<ETableSpecification cursor-mode='line'>"

"  <ETableColumn model_col='0' _title='Name' expansion='1.0' minimum_width='40'"
"                resizable='true' cell='string' compare='string'/>"

"  <ETableColumn model_col='1' _title='Path' expansion='1.0' minimum_width='80'"
"                resizable='true' cell='string' compare='string'/>"

#if 0
"  <ETableColumn model_col='2' _title='Clients' expansion='1.0' minimum_width='80'"
"                resizable='true' cell='string' compare='string'/>"
#endif

"  <ETableColumn model_col='2' _title='Writable' expansion='1.0' minimum_width='16'"
"                resizable='true' cell='checkbox' compare='integer'/>"

"</ETableSpecification>";


static const gchar *exports_basic_state =
"<ETableState>"
"  <column source='0'/>"
"  <column source='1'/>"
"  <column source='2'/>"
"  <grouping>"
"    <leaf column='0' ascending='true'/>"
"  </grouping>"
"</ETableState>";


static const gchar *export_clients_spec =
"<ETableSpecification cursor-mode='line'>"

"  <ETableColumn model_col='0' _title='Pattern' expansion='1.0' minimum_width='40'"
"                resizable='true' cell='string' compare='string'/>"

"  <ETableColumn model_col='1' _title='Write' expansion='1.0' minimum_width='16'"
"                resizable='true' cell='checkbox' compare='integer'/>"

"</ETableSpecification>";

static const gchar *export_clients_basic_state =
"<ETableState>"
"  <column source='0'/>"
"  <column source='1'/>"
"  <grouping>"
"    <leaf column='0' ascending='true'/>"
"  </grouping>"
"</ETableState>";

XstTableColDef imports_table_col_def [] =
{
	{ "host",    XST_TABLE_COL_STRING, xst_util_nice_hostname_dup, validate_import_host_name,   NULL },
	{ "path",    XST_TABLE_COL_STRING, xst_util_nice_path_dup,     validate_import_remote_path, NULL },
	{ "point",   XST_TABLE_COL_STRING, xst_util_nice_path_dup,     validate_import_local_path,  NULL },
	{ "mounted", XST_TABLE_COL_STATE,  NULL,                       NULL,                        set_mount_button_label_neg },
	{ NULL }
};

XstTableColDef exports_table_col_def [] =
{
	{ "name",     XST_TABLE_COL_STRING,       NULL,                   validate_export_name,       NULL },
	{ "path",     XST_TABLE_COL_STRING,       xst_util_nice_path_dup, validate_export_local_path, NULL },
/*	{ "clients",  XST_TABLE_COL_STRING_BLANK, NULL,                   NULL,                       NULL }, */
	{ "write", XST_TABLE_COL_STATE,        NULL,                   NULL,                       export_set_writable },
	{ NULL }
};

XstWidgetUserPolicy import_selected_policies[] =
{
	{ "imports_settings_button", XST_WIDGET_MODE_SENSITIVE },
	{ "imports_delete_button",   XST_WIDGET_MODE_SENSITIVE },
	{ "imports_mount_button",    XST_WIDGET_MODE_SENSITIVE },
	{ NULL }
};

XstWidgetUserPolicy import_unselected_policies[] =
{
	{ "imports_settings_button", XST_WIDGET_MODE_INSENSITIVE },
	{ "imports_delete_button",   XST_WIDGET_MODE_INSENSITIVE },
	{ "imports_mount_button",    XST_WIDGET_MODE_INSENSITIVE },
	{ NULL }
};

XstWidgetUserPolicy export_selected_policies[] =
{
	{ "exports_settings_button", XST_WIDGET_MODE_SENSITIVE },
	{ "exports_delete_button",   XST_WIDGET_MODE_SENSITIVE },
	{ NULL }
};

XstWidgetUserPolicy export_unselected_policies[] =
{
	{ "exports_settings_button", XST_WIDGET_MODE_INSENSITIVE },
	{ "exports_delete_button",   XST_WIDGET_MODE_INSENSITIVE },
	{ NULL }
};

XstWidgetUserPolicy export_clients_selected_policies[] =
{
	{ "export_clients_update_button", XST_WIDGET_MODE_SENSITIVE },
	{ "export_clients_delete_button", XST_WIDGET_MODE_SENSITIVE },
	{ NULL }
};

XstWidgetUserPolicy export_clients_unselected_policies[] =
{
	{ "exports_clients_update_button", XST_WIDGET_MODE_INSENSITIVE },
	{ "exports_clients_delete_button", XST_WIDGET_MODE_INSENSITIVE },
	{ NULL }
};

static void
set_value_at_with_col_def (ETableModel *etm, int col, int row, const void *val, void *data)
{
	XstTableColDef *def_table;
	XstTableColDef *def;
	xmlNodePtr node;
	void *value;

	g_return_if_fail (xst_tool_get_access (tool));

	def_table = data;
	def = &def_table [col];
	value = (void *) val;

	if (!def_table)
		g_error ("No column definition for this table.");

	if (!def)
		g_error ("No column definition for this table/column.");

	node = e_table_memory_get_data (E_TABLE_MEMORY (etm), row);
	if (!node)
		return;

	if (def->filter)
		value = def->filter (value);

	if (def->validator && !def->validator (value))
		return;

	if (def->type == XST_TABLE_COL_STRING)
	{
		xst_xml_set_child_content (node, def->xml_tag, g_strdup (value));

		if (def->filter)
			free (value);
	}
	else if (def->type == XST_TABLE_COL_STRING_BLANK)
	{
		if (def->filter)
			free (value);
	}
	else if (def->type == XST_TABLE_COL_STATE)
	{
		xst_xml_element_set_state (node, def->xml_tag, (gboolean) value);
	}

	if (def->changed)
		def->changed (value);

	xst_dialog_modify (tool->main_dialog);
}

static void *
value_at_with_col_def (ETableModel *etm, int col, int row, void *data)
{
	XstTableColDef *def_table;
	XstTableColDef *def;
	xmlNodePtr node;
	void *ret;

	def_table = data;
	def = &def_table [col];

	if (!def_table)
		g_error ("No column definition for this table.");

	if (!def)
		g_error ("No column definition for this table/column.");

	node = e_table_memory_get_data (E_TABLE_MEMORY (etm), row);

	if (def->type == XST_TABLE_COL_STRING)
	{
		if (!node)
			return g_strdup ("");

		node = xst_xml_element_find_first (node, def->xml_tag);
		if (!node)
			return g_strdup ("");

		ret = xst_xml_element_get_content (node);
	}
	else if (def->type == XST_TABLE_COL_STRING_BLANK)
	{
		ret = g_strdup ("");
	}
	else if (def->type == XST_TABLE_COL_STATE)
	{
		ret = (void *) xst_xml_element_get_state (node, def->xml_tag);
	}
	else
		ret = g_strdup ("");

	return ret;
}

static int
imports_col_count (ETableModel *etm, void *data)
{
        return IMPORTS_COL_LAST;
}

static int
exports_col_count (ETableModel *etm, void *data)
{
        return EXPORTS_COL_LAST;
}

static int
export_clients_col_count (ETableModel *etm, void *data)
{
        return EXPORT_CLIENTS_COL_LAST;
}

static void *
duplicate_value (ETableModel *etm, int col, const void *value, void *data)
{
        return g_strdup (value);
}

static void
free_value (ETableModel *etm, int col, void *value, void *data)
{
	g_free (value);
}

static void *
initialize_value (ETableModel *etm, int col, void *data)
{
        return g_strdup ("");
}

static gboolean
value_is_empty (ETableModel *etm, int col, const void *value, void *data)
{
        return !(value && *(char *) value);
}

static char *
value_to_string (ETableModel *etm, int col, const void *value, void *data)
{
        return g_strdup (value);
}

static void
set_value_at (ETableModel *etm, int col, int row, const void *val, void *data)
{
}

static gboolean
is_editable (ETableModel *etm, int col, int row, void *model_data)
{
	return (xst_tool_get_access (tool) &&
		(xst_dialog_get_complexity (tool->main_dialog) == XST_DIALOG_ADVANCED));
}

static void
export_clients_set_value_at (ETableModel *etm, int col, int row, const void *val, void *data)
{
	ExportClient *client;

	g_return_if_fail (xst_tool_get_access (tool));

	client = e_table_memory_get_data (E_TABLE_MEMORY (etm), row);
	if (!client)
		return;

	if (col > 1)
		g_error ("Bad column passed to exports_clients_set_value_at().");

#warning FIXME: Might have a memory leak here, but the input is const.

	if (col == 0)
	{
		if (client->pattern)
			g_free (client->pattern);

		client->pattern = g_strdup (val);
	}
	else
		client->write = (gboolean) val;

	export_clients_from_current_client ();
}

static void *
export_clients_value_at (ETableModel *etm, int col, int row, void *model_data)
{
#warning FIXME: Does the return value get freed?

	ExportClient *client;
	void *ret;

	client = e_table_memory_get_data (E_TABLE_MEMORY (etm), row);
	if (!client)
		return g_strdup ("");

	if (col > 1)
		g_error ("Bad column passed to export_clients_value_at().");

	switch (col)
	{
	case 0:
		ret = g_strdup (client->pattern);
		break;
	case 1:
		ret = (void *) client->write;
		break;
		
	default:
		ret = NULL;
	}

	return ret;
}

static void
imports_table_cursor_change_cb (ETable *et, gint row, gpointer user_data)
{
	ETableModel *model;
	xmlNodePtr node;

	/* Prepare import settings dialog */

	model = et->model;
	node = e_table_memory_get_data (E_TABLE_MEMORY (model), row);
	if (!node)
		g_error ("Row has no model.");

	import_settings_from_node (node);
	mount_button_from_import_node (node);

	/* Activate list action buttons */

	xst_dialog_set_widget_user_modes (tool->main_dialog, import_selected_policies);
}

static void
exports_table_cursor_change_cb (ETable *et, gint row, gpointer user_data)
{
	ETableModel *model;
	xmlNodePtr node;

	/* Prepare export settings dialog */

	model = et->model;
	node = e_table_memory_get_data (E_TABLE_MEMORY (model), row);
	if (!node)
		g_error ("Row has no model.");

	export_settings_from_node (node);

	/* Activate list action buttons */

	xst_dialog_set_widget_user_modes (tool->main_dialog, export_selected_policies);
}

static void
export_clients_table_cursor_change_cb (ETable *et, gint row, gpointer user_data)
{
	ETableModel *model;
	ExportClient *client;

	/* Update export clients dialog */

	model = et->model;
	client = e_table_memory_get_data (E_TABLE_MEMORY (model), row);
	if (!client)
		g_error ("Row has no model.");

	export_clients_from_client (client);

	/* Activate list action buttons */

	xst_dialog_set_widget_user_modes (tool->main_dialog, export_clients_selected_policies);
}

static ETableExtras *
create_extras (void)
{
	ETableExtras *extras;
/*	ECell *ec; */

	extras = e_table_extras_new ();

/*	e_table_extras_add_compare (extras, "id_compare", id_compare); */

	/* Sometimes we want to center cols, sometimes we don't... */
/*	ec = e_cell_text_new (NULL, GTK_JUSTIFY_CENTER);
	gtk_object_set (GTK_OBJECT (ec), "color_column", COL_USER_COLOR, NULL);
	e_table_extras_add_cell (extras, "my_cell", ec);
*/	
	return extras;
}

static ETableScrolled *
create_generic_table (ETableMemoryCalbacksColumnCountFn col_count_fn,
		      ETableMemoryCalbacksValueAtFn value_at_fn,
		      ETableMemoryCalbacksSetValueAtFn set_value_at_fn,
		      const gchar *spec, const gchar *state, XstTableColDef *col_def)
{
	ETableScrolled *table_scrolled;
	ETableModel *model;
	ETableExtras *extras;

	extras = create_extras ();

	model = e_table_memory_callbacks_new (col_count_fn,
					      value_at_fn,
					      set_value_at_fn ? set_value_at_fn : set_value_at,
					      is_editable,
					      duplicate_value,
					      free_value,
					      initialize_value,
					      value_is_empty,
					      value_to_string,
					      col_def);

#if 0
	root_path = e_table_memory_node_insert (E_TABLE_MEMORY (model), NULL, 0, g_strdup (""));
	e_tree_model_root_node_set_visible (model, FALSE);
#endif
	table_scrolled = E_TABLE_SCROLLED (e_table_scrolled_new (E_TABLE_MODEL (model), extras, spec, state));

	return (table_scrolled);
}

static ETableScrolled *
create_imports_table (void)
{
	ETableScrolled *imports_table;
	ETable *table;

	imports_table = create_generic_table (imports_col_count, value_at_with_col_def,
					      set_value_at_with_col_def, imports_spec, imports_basic_state,
					      imports_table_col_def);

	table = e_table_scrolled_get_table (E_TABLE_SCROLLED (imports_table));

	gtk_signal_connect (GTK_OBJECT (table), "cursor_change", imports_table_cursor_change_cb, NULL);
#if 0
	gtk_signal_connect (GTK_OBJECT (table), "double_click", on_settings_clicked, NULL);
#endif

	return imports_table;
}

static ETableScrolled *
create_exports_table (void)
{
	ETableScrolled *exports_table;
	ETable *table;

	exports_table = create_generic_table (exports_col_count, value_at_with_col_def,
					      set_value_at_with_col_def, exports_spec, exports_basic_state,
					      exports_table_col_def);

	table = e_table_scrolled_get_table (E_TABLE_SCROLLED (exports_table));

	gtk_signal_connect (GTK_OBJECT (table), "cursor_change", exports_table_cursor_change_cb, NULL);
#if 0
	gtk_signal_connect (GTK_OBJECT (table), "double_click", on_settings_clicked, NULL);
#endif

	return exports_table;
}

static ETableScrolled *
create_export_clients_table (void)
{
	ETableScrolled *export_clients_table;
	ETable *table;

	export_clients_table = create_generic_table (export_clients_col_count, export_clients_value_at,
						     export_clients_set_value_at, export_clients_spec,
						     export_clients_basic_state, NULL);

	table = e_table_scrolled_get_table (E_TABLE_SCROLLED (export_clients_table));

	gtk_signal_connect (GTK_OBJECT (table), "cursor_change", export_clients_table_cursor_change_cb,
			    NULL);
#if 0
	gtk_signal_connect (GTK_OBJECT (table), "double_click", on_settings_clicked, NULL);
#endif

	return export_clients_table;
}

void
create_tables (void)
{
	GtkWidget *container;
        GtkWidget *table_scrolled;

	/* Imports table */

	table_scrolled = GTK_WIDGET (create_imports_table ());

	container = xst_dialog_get_widget (tool->main_dialog, "imports_holder");
	gtk_box_pack_end (GTK_BOX (container), table_scrolled, TRUE, TRUE, 0);
	gtk_widget_show_all (table_scrolled);

	imports_table_scrolled = table_scrolled;

	/* Exports table */

	table_scrolled = GTK_WIDGET (create_exports_table ());

	container = xst_dialog_get_widget (tool->main_dialog, "exports_holder");
	gtk_box_pack_end (GTK_BOX (container), table_scrolled, TRUE, TRUE, 0);
	gtk_widget_show_all (table_scrolled);

	exports_table_scrolled = table_scrolled;

	/* Export clients table */

	table_scrolled = GTK_WIDGET (create_export_clients_table ());

	container = xst_dialog_get_widget (tool->main_dialog, "export_clients_holder");
	gtk_box_pack_end (GTK_BOX (container), table_scrolled, TRUE, TRUE, 0);
	gtk_widget_show_all (table_scrolled);

	export_clients_table_scrolled = table_scrolled;
}

static void
populate_table (ETableModel *model, xmlNodePtr root_node, gchar *type)
{
	xmlNodePtr node;

	g_return_if_fail (model != NULL);

	if (!root_node)
		return;

	e_table_memory_freeze (E_TABLE_MEMORY (model));

	for (node = xst_xml_element_find_first (root_node, type); node;
	     node = xst_xml_element_find_next (node, type))
	{
		e_table_memory_insert (E_TABLE_MEMORY (model), -1, node);
	}

	e_table_memory_thaw (E_TABLE_MEMORY (model));
}

void
populate_tables (void)
{
	ETableModel *model;
	ETable *table [3];
	xmlNodePtr node [3];
	gchar *type [3] = { "import", "export", NULL };
	gint i;

	g_assert (imports_table_scrolled);
	g_assert (exports_table_scrolled);

	printf ("Populating tables.\n");

	table [0] = e_table_scrolled_get_table (E_TABLE_SCROLLED (imports_table_scrolled));
	table [1] = e_table_scrolled_get_table (E_TABLE_SCROLLED (exports_table_scrolled));
	table [2] = NULL;

	node [0] = shares_xml_get_imports_node ();
	node [1] = shares_xml_get_exports_node ();
	node [2] = NULL;

	for (i = 0; table [i]; i++)
	{
		model = (table [i]->model);
		populate_table (model, node [i], type [i]);
	}
}

/**
 * table_get_rows_as_gslist: Gets all row data from table, returning it in a GSList.
 **/
GSList *
table_get_rows_as_gslist (ETableMemory *model)
{
	GSList *gslist = NULL;
	gint rows;
	gint i;

	rows = e_table_model_row_count (E_TABLE_MODEL (model));

	for (i = 0; i < rows; i++)
	{
		gslist = g_slist_append (gslist, e_table_memory_get_data (E_TABLE_MEMORY (model), i));
	}

	return gslist;
}

/**
 * table_set_rows_from_gslist: Takes a GSList of row data, to replace current contents of table.
 **/
void
table_set_rows_from_gslist (ETableMemory *model, GSList *gslist)
{
	gint row;

	e_table_memory_clear (model);

	for (row = 0; gslist; gslist = gslist->next, row++)
	{
		e_table_memory_insert (model, row, gslist->data);
	}
}

static void
table_insert_row (ETable *table, void *data)
{
	ETableModel *model;

	model = table->model;

	e_table_memory_freeze (E_TABLE_MEMORY (model));
	e_table_memory_insert (E_TABLE_MEMORY (model), -1, data);
	e_table_set_cursor_row (table, e_table_model_row_count (model) - 1);
	e_table_memory_thaw (E_TABLE_MEMORY (model));
}

/* --- */

static void
set_mount_button_label (gboolean mount)
{
	GtkWidget *button;

	button = xst_dialog_get_widget (tool->main_dialog, "imports_mount_button");
	g_return_if_fail (button != NULL);

	gtk_widget_set (button, "label", mount ? "Mount" : "Unmount", NULL);
}

static void
set_mount_button_label_neg (gboolean unmount)
{
	set_mount_button_label (!unmount);
}

static void
mount_button_from_import_node (xmlNodePtr node)
{
	gboolean state;
	
	state = xst_xml_element_get_state (node, "mounted");
	set_mount_button_label (!state);
}

void
import_table_update_selected (void)
{
	ETable *table;
	xmlNodePtr node;

	table = e_table_scrolled_get_table (E_TABLE_SCROLLED (imports_table_scrolled));
	e_table_model_row_changed (table->model, e_table_get_cursor_row (table));

	node = import_table_get_selected_node ();
	g_assert (node);

	mount_button_from_import_node (node);

	xst_dialog_modify (tool->main_dialog);
}


xmlNodePtr
import_table_get_selected_node (void)
{
	ETable *table;
	ETableModel *model;
	gint row;

	table = e_table_scrolled_get_table (E_TABLE_SCROLLED (imports_table_scrolled));
	model = table->model;
	row = e_table_get_cursor_row (table);
	if (row < 0)
		return NULL;

	return e_table_memory_get_data (E_TABLE_MEMORY (model), row);
}

void
import_table_insert_node (xmlNodePtr node)
{
	ETable *table;

	table = e_table_scrolled_get_table (E_TABLE_SCROLLED (imports_table_scrolled));
	table_insert_row (table, node);

	xst_dialog_set_widget_user_modes (tool->main_dialog, import_unselected_policies);
	xst_dialog_modify (tool->main_dialog);
}

void
import_table_delete_selected_node (void)
{
	ETable *table;
	ETableModel *model;
	xmlNodePtr node;

	table = e_table_scrolled_get_table (E_TABLE_SCROLLED (imports_table_scrolled));
	model = table->model;

	/* Deactivate list action buttons */

	xst_dialog_set_widget_user_modes (tool->main_dialog, import_unselected_policies);

	/* Remove from XML and table */

	e_table_memory_freeze (E_TABLE_MEMORY (model));

	node = import_table_get_selected_node ();
	g_assert (node);
	xst_xml_element_destroy (node);

	e_table_memory_remove (E_TABLE_MEMORY (model), e_table_get_cursor_row (table));
	e_table_memory_thaw (E_TABLE_MEMORY (model));

	xst_dialog_modify (tool->main_dialog);
}

void
import_table_mount_selected_node_toggle_cb (void)
{
	ETable *table;
	ETableModel *model;
	xmlNodePtr node;
	gboolean state;

	table = e_table_scrolled_get_table (E_TABLE_SCROLLED (imports_table_scrolled));
	model = table->model;

	node = import_table_get_selected_node ();
	g_assert (node);

	state = xst_xml_element_get_state (node, "mounted");
	xst_xml_element_set_state (node, "mounted", !state);

	import_table_update_selected ();
}

/* --- */


static void
export_set_general_write (xmlNodePtr export_node, gboolean general_write)
{
	xmlNodePtr node;

	for (node = xst_xml_element_find_first (export_node, "allow"); node;
	     node = xst_xml_element_find_next (node, "allow"))
	{
		gchar *pattern;

		pattern = xst_xml_get_child_content (node, "pattern");
		if (pattern && !strcmp (pattern, "0.0.0.0/0"))
		{
			xst_xml_element_set_state (node, "write", general_write);
			break;
		}
		else if (pattern)
			g_free (pattern);
	}
}

static void
export_set_writable (gboolean writable)
{
	xmlNodePtr node;

	node = export_table_get_selected_node ();
	export_set_general_write (node, writable);
}

void
export_table_update_selected (void)
{
	ETable *table;

	table = e_table_scrolled_get_table (E_TABLE_SCROLLED (exports_table_scrolled));
	e_table_model_row_changed (table->model, e_table_get_cursor_row (table));

	xst_dialog_modify (tool->main_dialog);
}


xmlNodePtr
export_table_get_selected_node (void)
{
	ETable *table;
	ETableModel *model;
	gint row;

	table = e_table_scrolled_get_table (E_TABLE_SCROLLED (exports_table_scrolled));
	model = table->model;
	row = e_table_get_cursor_row (table);
	if (row < 0)
		return NULL;

	return e_table_memory_get_data (E_TABLE_MEMORY (model), row);
}

void
export_table_insert_node (xmlNodePtr node)
{
	ETable *table;

	table = e_table_scrolled_get_table (E_TABLE_SCROLLED (exports_table_scrolled));
	table_insert_row (table, node);

	xst_dialog_set_widget_user_modes (tool->main_dialog, export_unselected_policies);
	xst_dialog_modify (tool->main_dialog);
}

void
export_table_delete_selected_node (void)
{
	ETable *table;
	ETableModel *model;
	xmlNodePtr node;

	table = e_table_scrolled_get_table (E_TABLE_SCROLLED (exports_table_scrolled));
	model = table->model;

	/* Deactivate list action buttons */

	xst_dialog_set_widget_user_modes (tool->main_dialog, export_unselected_policies);

	/* Remove from XML and table */

	e_table_memory_freeze (E_TABLE_MEMORY (model));

	node = export_table_get_selected_node ();
	g_assert (node);
	xst_xml_element_destroy (node);

	e_table_memory_remove (E_TABLE_MEMORY (model), e_table_get_cursor_row (table));
	e_table_memory_thaw (E_TABLE_MEMORY (model));

	xst_dialog_modify (tool->main_dialog);
}

/* --- */

void
export_clients_table_update_selected (void)
{
	ETable *table;

	table = e_table_scrolled_get_table (E_TABLE_SCROLLED (export_clients_table_scrolled));
	e_table_model_row_changed (table->model, e_table_get_cursor_row (table));
}


ExportClient *
export_clients_table_get_selected (void)
{
	ETable *table;
	ETableModel *model;
	gint row;

	table = e_table_scrolled_get_table (E_TABLE_SCROLLED (export_clients_table_scrolled));
	model = table->model;
	row = e_table_get_cursor_row (table);
	if (row < 0)
		return NULL;

	return e_table_memory_get_data (E_TABLE_MEMORY (model), row);
}

void
export_clients_table_insert (ExportClient *client)
{
	ETable *table;

	table = e_table_scrolled_get_table (E_TABLE_SCROLLED (export_clients_table_scrolled));
	table_insert_row (table, client);

	xst_dialog_set_widget_user_modes (tool->main_dialog, export_clients_unselected_policies);
}

void
export_clients_table_delete_selected (void)
{
	ETable *table;
	ETableModel *model;
	ExportClient *client;

	table = e_table_scrolled_get_table (E_TABLE_SCROLLED (export_clients_table_scrolled));
	model = table->model;

	/* Deactivate list action buttons */

	xst_dialog_set_widget_user_modes (tool->main_dialog, export_clients_unselected_policies);

	/* Remove from XML and table */

	e_table_memory_freeze (E_TABLE_MEMORY (model));

	client = export_clients_table_get_selected ();
	g_assert (client);
	g_free (client->pattern);
	g_free (client);

	e_table_memory_remove (E_TABLE_MEMORY (model), e_table_get_cursor_row (table));
	e_table_memory_thaw (E_TABLE_MEMORY (model));
}

void
export_clients_table_clear (void)
{
	ETable *table;

	table = e_table_scrolled_get_table (E_TABLE_SCROLLED (export_clients_table_scrolled));
	e_table_memory_clear (E_TABLE_MEMORY (table->model));
}

void
export_clients_table_to_export_node (xmlNodePtr node)
{
	ETable *table;
	ETableModel *model;
	GSList *rowdata, *slist;

	shares_xml_remove_allow_nodes (node);

	table = e_table_scrolled_get_table (E_TABLE_SCROLLED (export_clients_table_scrolled));
	model = table->model;
	rowdata = table_get_rows_as_gslist (E_TABLE_MEMORY (model));

	for (slist = rowdata ; slist; slist = g_slist_next (slist))
	{
		ExportClient *client;

		client = slist->data;
		shares_xml_new_allow_node (node, client->pattern, client->write);
	}

	if (rowdata)
		g_slist_free (rowdata);
}

void
export_clients_table_from_export_node (xmlNodePtr export_node)
{
	ETable *table;
	ETableModel *model;
	GSList *rowdata = NULL;
	xmlNodePtr node;

	table = e_table_scrolled_get_table (E_TABLE_SCROLLED (export_clients_table_scrolled));
	model = table->model;

	export_clients_table_clear ();

	for (node = xst_xml_element_find_first (export_node, "allow"); node;
	     node = xst_xml_element_find_next (node, "allow"))
	{
		ExportClient *client;
		gchar *pattern;

		client = g_new0 (ExportClient, 1);
		pattern = xst_xml_get_child_content (node, "pattern");
		if (pattern && strcmp (pattern, "0.0.0.0/0"))
		{
			client->pattern = pattern;
			client->write = xst_xml_element_get_state (node, "write");
			rowdata = g_slist_append (rowdata, client);
		}
		else if (pattern)
			g_free (pattern);
	}

	if (rowdata)
	{
		table_set_rows_from_gslist (E_TABLE_MEMORY (model), rowdata);
		g_slist_free (rowdata);
	}
}
