/*
 * Decompiled with CFR 0.152.
 */
package org.apache.brooklyn.util.javalang;

import com.google.common.collect.Sets;
import java.lang.reflect.Member;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.Set;
import org.apache.brooklyn.util.guava.Maybe;
import org.apache.brooklyn.util.javalang.Reflections;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class MethodAccessibleReflections {
    private static final Logger LOG = LoggerFactory.getLogger(MethodAccessibleReflections.class);
    private static final Set<String> SET_ACCESSIBLE_FAILED_LOGGED_METHODS = Sets.newConcurrentHashSet();
    private static final Set<String> SET_ACCESSIBLE_SUCCEEDED_LOGGED_METHODS = Sets.newConcurrentHashSet();
    private static final Set<String> FIND_ACCESSIBLE_FAILED_LOGGED_METHODS = Sets.newConcurrentHashSet();

    MethodAccessibleReflections() {
    }

    static boolean trySetAccessible(Method method) {
        try {
            method.setAccessible(true);
            if (SET_ACCESSIBLE_SUCCEEDED_LOGGED_METHODS.add(method.toString())) {
                LOG.warn("Discouraged use of setAccessible, called for method " + method);
            } else if (LOG.isTraceEnabled()) {
                LOG.trace("Discouraged use of setAccessible, called for method " + method);
            }
            return true;
        }
        catch (SecurityException e) {
            boolean added = SET_ACCESSIBLE_FAILED_LOGGED_METHODS.add(method.toString());
            if (added) {
                LOG.warn("Problem setting accessible for method " + method, (Throwable)e);
            } else if (LOG.isTraceEnabled()) {
                LOG.trace("Problem setting accessible for method " + method, (Throwable)e);
            }
            return false;
        }
    }

    static boolean isAccessible(Member m) {
        return m != null && Modifier.isPublic(m.getModifiers());
    }

    static boolean isAccessible(Class<?> c) {
        return c != null && Modifier.isPublic(c.getModifiers());
    }

    static Maybe<Method> findAccessibleMethod(Method method) {
        if (!MethodAccessibleReflections.isAccessible(method)) {
            String err = "Method is not public, so not normally accessible for " + method;
            if (FIND_ACCESSIBLE_FAILED_LOGGED_METHODS.add(method.toString())) {
                LOG.warn(err + "; usage may subsequently fail");
            }
            return Maybe.absent(err);
        }
        boolean declaringClassAccessible = MethodAccessibleReflections.isAccessible(method.getDeclaringClass());
        if (declaringClassAccessible) {
            return Maybe.of(method);
        }
        if (Modifier.isStatic(method.getModifiers())) {
            String err = "Static method not declared on a public class, so not normally accessible for " + method;
            if (FIND_ACCESSIBLE_FAILED_LOGGED_METHODS.add(method.toString())) {
                LOG.warn(err + "; usage may subsequently fail");
            }
            return Maybe.absent(err);
        }
        Maybe<Method> altMethod = MethodAccessibleReflections.tryFindAccessibleEquivalent(method);
        if (altMethod.isPresent()) {
            LOG.debug("Switched method for publicly accessible equivalent: method={}; origMethod={}", (Object)altMethod.get(), (Object)method);
            return altMethod;
        }
        String err = "No accessible (overridden) method found in public super-types for method " + method;
        if (FIND_ACCESSIBLE_FAILED_LOGGED_METHODS.add(method.toString())) {
            LOG.warn(err);
        }
        return Maybe.absent(err);
    }

    private static Maybe<Method> tryFindAccessibleEquivalent(Method method) {
        Class<?> clazz = method.getDeclaringClass();
        for (Class<?> interf : Reflections.getAllInterfaces(clazz)) {
            Maybe<Method> altMethod = MethodAccessibleReflections.tryFindAccessibleMethod(interf, method.getName(), method.getParameterTypes());
            if (!altMethod.isPresent()) continue;
            return altMethod;
        }
        for (Class<?> superClazz = clazz.getSuperclass(); superClazz != null; superClazz = superClazz.getSuperclass()) {
            Maybe<Method> altMethod = MethodAccessibleReflections.tryFindAccessibleMethod(superClazz, method.getName(), method.getParameterTypes());
            if (!altMethod.isPresent()) continue;
            return altMethod;
        }
        return Maybe.absent();
    }

    private static Maybe<Method> tryFindAccessibleMethod(Class<?> clazz, String methodName, Class<?> ... parameterTypes) {
        if (!MethodAccessibleReflections.isAccessible(clazz)) {
            return Maybe.absent();
        }
        try {
            Method altMethod = clazz.getMethod(methodName, parameterTypes);
            if (MethodAccessibleReflections.isAccessible(altMethod) && !Modifier.isStatic(altMethod.getModifiers())) {
                return Maybe.of(altMethod);
            }
        }
        catch (NoSuchMethodException | SecurityException exception) {
            // empty catch block
        }
        return Maybe.absent();
    }
}

