# Test create_public_report_month() --------------------------------------------

# Set up test data
data_file <- system.file("sample_data/sample_epitrax_data.csv",
                         package = "epitraxr")
data <- read_epitrax_data(data_file)

diseases <- data.frame(
  EpiTrax_name = c("Influenza", "COVID-19", "Measles", "Syphilis"),
  Public_name = c("Alpha", "Bravo", "Charlie", "Delta")
)

config_file <- system.file("tinytest/test_files/configs/good_config.yaml",
                          package = "epitraxr")
config <- get_report_config(config_file)

# Test with valid input for February 2024
result <- create_public_report_month(
  data = data,
  disease = diseases,
  y = 2024,
  m = 2,
  config = config
)

# Check output structure
expect_true(is.list(result))
expect_true(all(c("name", "report") %in% names(result)))
expect_true(is.data.frame(result$report))

# Check report content
report <- result$report
expect_true(all(c("Alpha", "Bravo", "Charlie", "Delta") %in% report$Disease))
expect_true(all(c("Rate_per_100k", "Avg_5yr_Rate", "Trend") %in% colnames(report)))


# Test create_public_report_ytd() ----------------------------------------------

# Set up test data
data_file <- system.file("sample_data/sample_epitrax_data.csv",
                         package = "epitraxr")
data <- read_epitrax_data(data_file)

diseases <- data.frame(
  EpiTrax_name = c("Influenza", "COVID-19", "Measles", "Syphilis"),
  Public_name = c("Alpha", "Bravo", "Charlie", "Delta")
)

config_file <- system.file("tinytest/test_files/configs/good_config.yaml",
                          package = "epitraxr")
config <- get_report_config(config_file)

# Test with valid input
ytd_result <- create_public_report_ytd(
  data = data,
  disease = diseases,
  y = 2024,
  m = 2,
  config = config
)

# Check output structure
expect_true(is.list(ytd_result))
expect_true(all(c("name", "report") %in% names(ytd_result)))
expect_true(is.data.frame(ytd_result$report))

# Check report content
ytd_report <- ytd_result$report
expect_true(all(c("Alpha", "Bravo", "Charlie", "Delta") %in% report$Disease))
expect_true(all(c("YTD_Rate_per_100k", "Avg_5yr_Rate", "Trend") %in% colnames(ytd_report)))


# Test create_report_annual_counts() -------------------------------------------

data <- data.frame(
  disease = c("A", "A", "B", "C"),
  year = c(2020, 2021, 2020, 2021),
  counts = c(5, 7, 8, 0)
)

diseases <- c("A", "B", "C", "D")

result <- create_report_annual_counts(data, diseases)

expected_result <- data.frame(
  disease = diseases,
  `2020` = c(5,8,0,0),
  `2021` = c(7,0,0,0),
  check.names = FALSE
)

expect_true(is.data.frame(result))
expect_equivalent(result, expected_result)


# Test create_report_monthly_counts() ------------------------------------------

data <- data.frame(
  disease = c("A", "A", "B", "B", "C"),
  year = c(2024, 2024, 2024, 2024, 2024),
  month = c(1, 2, 1, 3, 1),
  counts = c(5, 7, 8, 3, 2)
)

diseases <- c("A", "B", "C", "D")

result <- create_report_monthly_counts(
  data = data,
  diseases = diseases,
  y = 2024
)

expected_result <- data.frame(
  disease = diseases,
  Jan = c(5, 8, 2, 0),
  Feb = c(7, 0, 0, 0),
  Mar = c(0, 3, 0, 0)
)

expect_true(is.data.frame(result))
expect_equivalent(result, expected_result)


# Test create_report_monthly_avgs() --------------------------------------------

data <- data.frame(
  disease = c("A", "A", "B", "B", "C"),
  year = c(2023, 2024, 2023, 2024, 2023),
  month = c(1, 1, 2, 2, 1),
  counts = c(10, 20, 15, 25, 8)
)

diseases <- c("A", "B", "C", "D")
config <- list(rounding_decimals = 1)

result <- create_report_monthly_avgs(data, diseases, config)

expected_result <- data.frame(
  disease = diseases,
  Jan = c(15.0, 0.0, 4.0, 0.0),  # (10+20)/2, 0, 8/2, 0
  Feb = c(0.0, 20.0, 0.0, 0.0)   # 0, (15+25)/2, 0, 0
)

expect_true(is.data.frame(result))
expect_equivalent(result, expected_result)


# Test create_report_monthly_medians() -----------------------------------------

data <- data.frame(
  disease = c("A", "A", "A", "B", "B", "B", "C"),
  year = c(2022, 2023, 2024, 2022, 2023, 2024, 2022),
  month = c(1, 1, 1, 2, 2, 2, 1),
  counts = c(10, 20, 30, 5, 15, 25, 8)
)

diseases <- c("A", "B", "C", "D")

result <- create_report_monthly_medians(data, diseases)

# With the fix, medians should be calculated across all years (2022-2024)
# Disease A: Jan has [10,20,30] → median = 20
# Disease B: Feb has [5,15,25] → median = 15, Jan has [0,0,0] → median = 0
# Disease C: Jan has [8,0,0] → median = 0, Feb has [0,0,0] → median = 0
# Disease D: all months have [0,0,0] → median = 0
expected_result <- data.frame(
  disease = diseases,
  Jan = c(20, 0, 0, 0),
  Feb = c(0, 15, 0, 0)
)

expect_true(is.data.frame(result))
expect_equal(result[,c("disease","Jan","Feb")], expected_result)

# Test the specific bug case: missing year/disease combinations
test_data <- data.frame(
  disease = c("A", "A", "B"),  # A has 2022,2023 but not 2024; B only has 2024
  year = c(2022, 2023, 2024),
  month = c(1, 1, 1),
  counts = c(10, 20, 5)
)

bug_result <- create_report_monthly_medians(test_data, c("A", "B"))

# A: Jan should have [10,20,0] → median = 10
# B: Jan should have [0,0,5] → median = 0
bug_expected <- data.frame(
  disease = c("A", "B"),
  Jan = c(10, 0),
  Feb = c(0, 0)
)

expect_equal(bug_result[,c("disease","Jan","Feb")], bug_expected)


# Test create_report_ytd_counts() ----------------------------------------------

data <- data.frame(
  disease = c("A", "A", "B", "B", "C"),
  year = c(2024, 2023, 2024, 2023, 2023),
  month = c(1, 1, 2, 2, 1),
  counts = c(10, 20, 15, 25, 8)
)

diseases <- c("A", "B", "C", "D")
config <- list(
  current_population = 56000,
  avg_5yr_population = 103000,
  rounding_decimals = 1
)

# Test with raw counts
result <- create_report_ytd_counts(
  data = data,
  diseases = diseases,
  y = 2024,
  m = 2,
  config = config,
  as.rates = FALSE
)

expected_result <- data.frame(
  disease = diseases,
  Current_YTD_Counts = c(10, 15, 0, 0), # 2024 counts up to month 2
  Avg_5yr_YTD_Counts = c(20, 25, 8, 0)  # 2023 counts up to month 2
)

expect_true(is.data.frame(result))
expect_equal(result, expected_result)

# Test with rates
result <- create_report_ytd_counts(
  data = data,
  diseases = diseases,
  y = 2024,
  m = 2,
  config = config,
  as.rates = TRUE
)

expected_result <- data.frame(
  disease = diseases,
  Current_YTD_Rate_per_100k = c(17.9, 26.8, 0.0, 0.0),
  Avg_5yr_YTD_Rate_per_100k = c(19.4, 24.3, 7.8, 0.0)
)

expect_true(is.data.frame(result))
expect_equal(result, expected_result)


# Test create_report_ytd_medians() ---------------------------------------------

data <- data.frame(
  disease = c("A", "A", "A", "A", "B", "B", "B", "B", "C"),
  year = c(2020, 2021, 2022, 2023, 2020, 2021, 2022, 2023, 2020),
  month = c(1, 1, 2, 1, 2, 2, 3, 2, 1),
  counts = c(10, 20, 30, 25, 5, 15, 8, 12, 7)
)

diseases <- c("A", "B", "C", "D")

# Test with m = 2 (Jan-Feb YTD)
result <- create_report_ytd_medians(data, diseases, 2)

expected_result <- data.frame(
  disease = diseases,
  median_counts = c(22.5, 8.5, 0, 0)
)

expect_true(is.data.frame(result))
expect_equal(result, expected_result)

# Test with m = 1 (January only)
result_jan <- create_report_ytd_medians(data, diseases, 1)

expected_jan <- data.frame(
  disease = diseases,
  median_counts = c(15, 0, 0, 0)
)

expect_true(is.data.frame(result_jan))
expect_equal(result_jan, expected_jan)


# Test create_report_grouped_stats() -------------------------------------------

data <- data.frame(
  disease = c("Flu", "Flu", "Flu", "Flu", "Measles", "Measles", "Measles", "Measles", "COVID"),
  year = c(2023, 2024, 2023, 2024, 2023, 2024, 2023, 2024, 2023),
  month = c(3, 3, 1, 1, 3, 3, 2, 2, 1),
  counts = c(15, 25, 10, 12, 8, 5, 6, 4, 20)
)

diseases <- data.frame(
  EpiTrax_name = c("Flu", "Measles", "COVID"),
  Group_name = c("Respiratory", "Vaccine-Preventable", "Respiratory")
)

config <- list(
  current_population = 100000,
  avg_5yr_population = 100000,
  rounding_decimals = 1,
  trend_threshold = 0.15
)

# Test with March (month 3) 2024
result <- create_report_grouped_stats(data, diseases, 2024, 3, config)

expected_result <- data.frame(
  Group = c("Respiratory", "Respiratory", "Vaccine-Preventable"),
  Disease = c("COVID", "Flu", "Measles"),
  `March 2024` = c(0, 25, 5),
  `March 2024 Rate` = c(0, 25, 5),
  `Historical March Avg` = c(0, 15, 8),
  `Historical March Median` = c(0, 15, 8),
  `2024 YTD` = c(0, 37, 9),
  `Historical 2024 YTD Avg` = c(20, 25, 14),
  `Historical 2024 YTD Median` = c(20, 25, 14),
  `YTD Trend` = compute_trend(c(0, 37, 9), c(20, 25, 14), threshold = 0.15),
  check.names = FALSE
)

expect_true(is.data.frame(result))
expect_equal(result, expected_result)

# Test with missing Group_name column (NULL case)
diseases_no_group <- data.frame(
  EpiTrax_name = c("Flu", "Measles", "COVID")
)

# Should issue a warning and set all groups to "Uncategorized"
expect_warning(
  result_no_group <- create_report_grouped_stats(data, diseases_no_group, 2024, 3, config),
  "no groups were provided"
)

expected_result$Group <- c("Uncategorized", "Uncategorized", "Uncategorized")

expect_true(is.data.frame(result_no_group))
expect_equal(result_no_group, expected_result)

# Test with NA values in Group_name column
diseases_with_na <- data.frame(
  EpiTrax_name = c("Flu", "Measles", "COVID"),
  Group_name = c("Respiratory", NA, "Respiratory")
)

result_with_na <- create_report_grouped_stats(data, diseases_with_na, 2024, 3, config)

expected_result$Group <- c("Respiratory", "Respiratory", "Uncategorized")

expect_true(is.data.frame(result_with_na))
expect_equal(result_with_na, expected_result)


# Test create_public_report_combined_month_ytd() -------------------------------

# Set up test data
data_file <- system.file("sample_data/sample_epitrax_data.csv",
                         package = "epitraxr")
data <- read_epitrax_data(data_file)

diseases <- data.frame(
  EpiTrax_name = c("Influenza", "COVID-19", "Measles", "Syphilis"),
  Public_name = c("Alpha", "Bravo", "Charlie", "Delta")
)

config_file <- system.file("tinytest/test_files/configs/good_config.yaml",
                          package = "epitraxr")
config <- get_report_config(config_file)

# Test with valid input for February 2024
result <- create_public_report_combined_month_ytd(data, diseases, 2024, 2, config)

# Check output structure
expect_true(is.list(result))
expect_true(all(c("name", "report") %in% names(result)))
expect_true(is.data.frame(result$report))

# Check report name
expect_equal(result$name, "public_report_combined_Feb2024")

# Check report content
report <- result$report
expect_true(all(c("Alpha", "Bravo", "Charlie", "Delta") %in% report$Disease))

# Check expected columns are present
expected_columns <- c(
  "Disease", "Feb Cases", "Feb Average Cases", "Trend",
  "YTD Cases", "YTD Average Cases", "YTD Rate per 100k", "YTD Average Rate per 100k"
)
expect_true(all(expected_columns %in% colnames(report)))

# Check data types
expect_true(is.character(report$Disease))
expect_true(is.numeric(report$`Feb Cases`))
expect_true(is.numeric(report$`YTD Cases`))
expect_true(is.character(report$Trend))

# Test with March 2024 (different month)
result_march <- create_public_report_combined_month_ytd(data, diseases, 2024, 3, config)

expect_equal(result_march$name, "public_report_combined_Mar2024")
march_report <- result_march$report
expect_true("Mar Cases" %in% colnames(march_report))
expect_true("Mar Average Cases" %in% colnames(march_report))

# Test data integrity - YTD should be cumulative
alpha_march <- march_report[march_report$Disease == "Alpha", ]
expect_equal(alpha_march$`YTD Cases`, 1466)

beta_march <- march_report[march_report$Disease == "Bravo", ]
expect_equal(beta_march$`YTD Cases`, 1191)
