## ----setup, include=FALSE-----------------------------------------------------
knitr::opts_chunk$set(echo = TRUE)

## ----install_packages, echo=FALSE, warning=FALSE, results='hide',message=FALSE----

###*****************************
# INITIAL COMMANDS TO RESET THE SYSTEM
seedNo=14159
set.seed(seedNo)
###*****************************

###*****************************
require("sicegar")
require("dplyr")
require("ggplot2")
require("cowplot")
###*****************************

## ----generate_data------------------------------------------------------------
noise_parameter <- 1
reps <- 5
time <- rep(seq(3, 24, 3), reps)
mean_values <- doublesigmoidalFitFormula_h0(time,
                                       finalAsymptoteIntensityRatio = .3,
                                       maximum = 10,
                                       slope1Param = 1,
                                       midPoint1Param = 7,
                                       slope2Param = 1,
                                       midPointDistanceParam = 8,
                                       h0 = 3)
intensity <- rnorm(n = length(mean_values), mean = mean_values, sd = rep(noise_parameter, length(mean_values)))

dataInput <- data.frame(time, intensity)
ggplot(dataInput, aes(time, intensity)) + 
  geom_point() + 
  scale_y_continuous(limits = c(-1, 13), expand = expansion(mult = c(0, 0))) + 
  theme_bw()

## -----------------------------------------------------------------------------
fitObj_zero <- fitAndCategorize(dataInput,
                           threshold_minimum_for_intensity_maximum = 0.3,
                           threshold_intensity_range = 0.1,
                           threshold_t0_max_int = 1E10,
                           use_h0 = FALSE)   # Default

fitObj_free <- fitAndCategorize(dataInput,
                           threshold_minimum_for_intensity_maximum = 0.3,
                           threshold_intensity_range = 0.1,
                           threshold_t0_max_int = 1E10,
                           use_h0 = TRUE)

## ----zero_free_plots, fig.height=4, fig.width=8-------------------------------
# Double-sigmoidal fit with parameter related lines
fig_a <- figureModelCurves(dataInput = fitObj_zero$normalizedInput,
                                  doubleSigmoidalFitVector = fitObj_zero$doubleSigmoidalModel,
                                  showParameterRelatedLines = TRUE,
                                  use_h0 = FALSE)   # Default

fig_b <- figureModelCurves(dataInput = fitObj_free$normalizedInput,
                                  doubleSigmoidalFitVector = fitObj_free$doubleSigmoidalModel,
                                  showParameterRelatedLines = TRUE,
                                  use_h0 = TRUE)

plot_grid(fig_a, fig_b, ncol = 2) # function from the cowplot package

## ----normalize_data-----------------------------------------------------------
normalizedInput_free <- normalizeData(dataInput = dataInput, 
                                 dataInputName = "doubleSigmoidalSample")
head(normalizedInput_free$timeIntensityData) # the normalized time and intensity data

## -----------------------------------------------------------------------------
# Fit the double-sigmoidal model
doubleSigmoidalModel_free <- multipleFitFunction_h0(dataInput=normalizedInput_free,
                                            model="doublesigmoidal")

doubleSigmoidalModel_free <- parameterCalculation_h0(doubleSigmoidalModel_free)

## ----plot_raw_fit, echo=TRUE, message=FALSE, warning=FALSE, comment=FALSE, fig.height=4, fig.width=6----
# double-sigmoidal fit
figureModelCurves(dataInput = normalizedInput_free,
                  doubleSigmoidalFitVector = doubleSigmoidalModel_free,
                  showParameterRelatedLines = TRUE,
                  use_h0 = TRUE)

## -----------------------------------------------------------------------------
fitObj_zero$doubleSigmoidalModel |>
  dplyr::select(finalAsymptoteIntensityRatio_Estimate, maximum_Estimate, slope1Param_Estimate, midPoint1Param_Estimate,
         slope2Param_Estimate, midPointDistanceParam_Estimate) |> 
  c()


fitObj_free$doubleSigmoidalModel |>
  dplyr::select(finalAsymptoteIntensityRatio_Estimate, maximum_Estimate, slope1Param_Estimate, midPoint1Param_Estimate,
         slope2Param_Estimate, midPointDistanceParam_Estimate, h0_Estimate) |> c()



