/**
 * Copyright (c) 2021 CEA LIST.
 * 
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 */
package org.eclipse.papyrus.designer.transformation.tracing.barectf.library;

import com.google.common.base.Objects;
import org.eclipse.emf.common.util.EList;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Typedef;
import org.eclipse.papyrus.designer.transformation.library.xtend.CppUtils;
import org.eclipse.papyrus.designer.transformation.tracing.barectf.library.BareCTFconfig;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Enumeration;
import org.eclipse.uml2.uml.EnumerationLiteral;
import org.eclipse.uml2.uml.PrimitiveType;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.ValueSpecification;
import org.eclipse.uml2.uml.util.UMLUtil;
import org.eclipse.xtend2.lib.StringConcatenation;

@SuppressWarnings("all")
public class CTFTypeUtils {
  /**
   * Calculate the typename, defaulting to ctf_integer
   */
  public static CharSequence ctfType(final Type type) {
    StringConcatenation _builder = new StringConcatenation();
    {
      if (((Objects.equal(type.getQualifiedName(), "CORBA::String") || Objects.equal(type.getQualifiedName(), "PrimitiveTypes::String")) || Objects.equal(type.getQualifiedName(), "UMLPrimitiveTypes::String"))) {
        _builder.append("string");
        _builder.newLine();
      } else {
        if ((((Objects.equal(type.getQualifiedName(), "CORBA::Float") || Objects.equal(type.getQualifiedName(), "CORBA::Double")) || Objects.equal(type.getQualifiedName(), "AnsiCLibrary::float")) || Objects.equal(type.getQualifiedName(), "AnsiCLibrary::double"))) {
          _builder.append("float");
          _builder.newLine();
        } else {
          if (((Objects.equal(type.getQualifiedName(), "UMLPrimitiveTypes::Boolean") || Objects.equal(type.getQualifiedName(), "PrimitiveTypes::Boolean")) || Objects.equal(type.getQualifiedName(), "CORBA::Boolean"))) {
            _builder.append("/* bool */ uint8");
            _builder.newLine();
          } else {
            if ((type instanceof PrimitiveType)) {
              String _primitiveType = CTFTypeUtils.primitiveType(((PrimitiveType)type));
              _builder.append(_primitiveType);
              _builder.newLineIfNotEmpty();
            } else {
              if ((type instanceof Enumeration)) {
                String _enumType = CTFTypeUtils.enumType(((Enumeration)type));
                _builder.append(_enumType);
                _builder.newLineIfNotEmpty();
              } else {
                if ((type instanceof DataType)) {
                  _builder.append("// should not happen due to flattening");
                  _builder.newLine();
                } else {
                  CharSequence _cppType = CppUtils.cppType(type);
                  _builder.append(_cppType);
                  _builder.newLineIfNotEmpty();
                }
              }
            }
          }
        }
      }
    }
    return _builder;
  }
  
  /**
   * Handle primitive types applying the typedef stereotype
   * Map some C/C++ types to their according CTF types
   */
  public static String primitiveType(final PrimitiveType pt) {
    final Typedef typedef = UMLUtil.<Typedef>getStereotypeApplication(pt, Typedef.class);
    String name = pt.getName();
    if ((typedef != null)) {
      name = typedef.getDefinition();
    }
    if (((Objects.equal(name, "int") || Objects.equal(name, "Integer")) || Objects.equal(name, "long"))) {
      return "int32";
    } else {
      if ((Objects.equal(name, "unsigned int") || Objects.equal(name, "unsigned long"))) {
        return "uint32";
      } else {
        boolean _equals = Objects.equal(name, "short");
        if (_equals) {
          return "int16";
        } else {
          boolean _equals_1 = Objects.equal(name, "unsigned short");
          if (_equals_1) {
            return "uint16";
          } else {
            boolean _equals_2 = Objects.equal(name, "char");
            if (_equals_2) {
              return "uint16";
            } else {
              boolean _equals_3 = Objects.equal(name, "bool");
              if (_equals_3) {
                return "uint8";
              } else {
                boolean _equals_4 = Objects.equal(name, "std::string");
                if (_equals_4) {
                  return "string";
                }
              }
            }
          }
        }
      }
    }
    return name;
  }
  
  public static String enumType(final Enumeration enumT) {
    boolean _containsKey = BareCTFconfig.types.containsKey(enumT);
    boolean _not = (!_containsKey);
    if (_not) {
      BareCTFconfig.types.put(enumT, Boolean.valueOf(true));
      String _name = enumT.getName();
      String _plus = (_name + ".yaml");
      BareCTFconfig.writeFile(_plus, CTFTypeUtils.enumTypeContents(enumT));
    }
    return enumT.getName();
  }
  
  /**
   * Create BareCTF enumeration declaration based on mappings.
   * Assumes that number of values fit in 8bit (0..255)
   */
  public static String enumTypeContents(final Enumeration enumT) {
    StringConcatenation _builder = new StringConcatenation();
    int i = 0;
    _builder.newLineIfNotEmpty();
    _builder.append("$field-type-aliases:");
    _builder.newLine();
    _builder.append("\t");
    String _name = enumT.getName();
    _builder.append(_name, "\t");
    _builder.append(":");
    _builder.newLineIfNotEmpty();
    _builder.append("\t\t");
    _builder.append("class: unsigned-enum");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("size: 8");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("alignment: 8");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("mappings:");
    _builder.newLine();
    {
      EList<EnumerationLiteral> _ownedLiterals = enumT.getOwnedLiterals();
      for(final EnumerationLiteral literal : _ownedLiterals) {
        _builder.append("\t\t\t");
        {
          ValueSpecification _specification = literal.getSpecification();
          boolean _tripleNotEquals = (_specification != null);
          if (_tripleNotEquals) {
            _builder.append(i = Integer.parseInt(literal.getSpecification().stringValue()), "\t\t\t");
          }
        }
        _builder.newLineIfNotEmpty();
        _builder.append("\t\t\t");
        String _escapeLitName = CTFTypeUtils.escapeLitName(literal);
        _builder.append(_escapeLitName, "\t\t\t");
        _builder.append(" : [");
        int _plusPlus = i++;
        _builder.append(_plusPlus, "\t\t\t");
        _builder.append("]");
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder.toString();
  }
  
  /**
   * postfix with _ to avoid problematic literals like true and false
   * Might need completition with additional keyword
   */
  public static String escapeLitName(final EnumerationLiteral literal) {
    final String name = literal.getName();
    final String lcName = literal.getName().toLowerCase();
    if ((Objects.equal(lcName, "true") || Objects.equal(lcName, "false"))) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append(name);
      _builder.append("_");
      return _builder.toString();
    }
    return name;
  }
  
  /**
   * convert tabs to space
   */
  public static String confTabs(final CharSequence seq) {
    return seq.toString().replaceAll("\t", "  ");
  }
}
