# Commands covered:  append lappend
#
# This file contains a collection of tests for one or more of the Tcl
# built-in commands.  Sourcing this file into Tcl runs the tests and
# generates output for errors.  No output means no errors were found.
#
# Copyright (c) 1991-1993 The Regents of the University of California.
# Copyright (c) 1994-1996 Sun Microsystems, Inc.
# Copyright (c) 1998-1999 by Scriptics Corporation.
#
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#
# RCS: @(#) $Id: appendComp.tcl,v 1.1 2006/12/29 09:02:53 fourdman Exp $

if {[lsearch [namespace children] ::tcltest] == -1} {
    package require tcltest
    namespace import -force ::tcltest::*
}
catch {unset x}

test appendComp-1.1 {append command} {
    catch {unset x}
    proc foo {} {append ::x 1 2 abc "long string"}
    list [foo] $x
} {{12abclong string} {12abclong string}}
test appendComp-1.2 {append command} {
    proc foo {} {
	set x ""
	list [append x first] [append x second] [append x third] $x
    }
    foo
} {first firstsecond firstsecondthird firstsecondthird}
test appendComp-1.3 {append command} {
    proc foo {} {
	set x "abcd"
	append x
    }
    foo
} abcd

test appendComp-2.1 {long appends} {
    proc foo {} {
	set x ""
	for {set i 0} {$i < 1000} {set i [expr $i+1]} {
	    append x "foobar "
	}
	set y "foobar"
	set y "$y $y $y $y $y $y $y $y $y $y"
	set y "$y $y $y $y $y $y $y $y $y $y"
	set y "$y $y $y $y $y $y $y $y $y $y "
	expr {$x == $y}
    }
    foo
} 1

test appendComp-3.1 {append errors} {
    proc foo {} {append}
    list [catch {foo} msg] $msg
} {1 {wrong # args: should be "append varName ?value value ...?"}}
test appendComp-3.2 {append errors} {
    proc foo {} {
	set x ""
	append x(0) 44
    }
    list [catch {foo} msg] $msg
} {1 {can't set "x(0)": variable isn't array}}
test appendComp-3.3 {append errors} {
    proc foo {} {
	catch {unset x}
	append x
    }
    list [catch {foo} msg] $msg
} {1 {can't read "x": no such variable}}

test appendComp-4.1 {lappend command} {
    proc foo {} {
	global x
	catch {unset x}
	lappend x 1 2 abc "long string"
    }
    list [foo] $x
} {{1 2 abc {long string}} {1 2 abc {long string}}}
test appendComp-4.2 {lappend command} {
    proc foo {} {
	set x ""
	list [lappend x first] [lappend x second] [lappend x third] $x
    }
    foo
} {first {first second} {first second third} {first second third}}
test appendComp-4.3 {lappend command} {
    proc foo {} {
	global x
	set x old
	unset x
	lappend x new
    }
    set result [foo]
    rename foo {}
    set result
} {new}
test appendComp-4.4 {lappend command} {
    proc foo {} {
	set x {}
	lappend x \{\  abc
    }
    foo
} {\{\  abc}
test appendComp-4.5 {lappend command} {
    proc foo {} {
	set x {}
	lappend x \{ abc
    }
    foo
} {\{ abc}
test appendComp-4.6 {lappend command} {
    proc foo {} {
	set x {1 2 3}
	lappend x
    }
    foo
} {1 2 3}
test appendComp-4.7 {lappend command} {
    proc foo {} {
	set x "a\{"
	lappend x abc
    }
    foo
} "a\\\{ abc"
test appendComp-4.8 {lappend command} {
    proc foo {} {
	set x "\\\{"
	lappend x abc
    }
    foo
} "\\{ abc"
test appendComp-4.9 {lappend command} {
    proc foo {} {
	set x " \{"
	list [catch {lappend x abc} msg] $msg
    }
    foo
} {1 {unmatched open brace in list}}
test appendComp-4.10 {lappend command} {
    proc foo {} {
	set x "	\{"
	list [catch {lappend x abc} msg] $msg
    }
    foo
} {1 {unmatched open brace in list}}
test appendComp-4.11 {lappend command} {
    proc foo {} {
	set x "\{\{\{"
	list [catch {lappend x abc} msg] $msg
    }
    foo
} {1 {unmatched open brace in list}}
test appendComp-4.12 {lappend command} {
    proc foo {} {
	set x "x \{\{\{"
	list [catch {lappend x abc} msg] $msg
    }
    foo
} {1 {unmatched open brace in list}}
test appendComp-4.13 {lappend command} {
    proc foo {} {
	set x "x\{\{\{"
	lappend x abc
    }
    foo
} "x\\\{\\\{\\\{ abc"
test appendComp-4.14 {lappend command} {
    proc foo {} {
	set x " "
	lappend x abc
    }
    foo
} "abc"
test appendComp-4.15 {lappend command} {
    proc foo {} {
	set x "\\ "
	lappend x abc
    }
    foo
} "{ } abc"
test appendComp-4.16 {lappend command} {
    proc foo {} {
	set x "x "
	lappend x abc
    }
    foo
} "x abc"
test appendComp-4.17 {lappend command} {
    proc foo {} { lappend x }
    foo
} {}
test appendComp-4.18 {lappend command} {
    proc foo {} { lappend x {} }
    foo
} {{}}
test appendComp-4.19 {lappend command} {
    proc foo {} { lappend x(0) }
    foo
} {}
test appendComp-4.20 {lappend command} {
    proc foo {} { lappend x(0) abc }
    foo
} {abc}

proc check {var size} {
    set l [llength $var]
    if {$l != $size} {
	return "length mismatch: should have been $size, was $l"
    }
    for {set i 0} {$i < $size} {set i [expr $i+1]} {
	set j [lindex $var $i]
	if {$j != "item $i"} {
	    return "element $i should have been \"item $i\", was \"$j\""
	}
    }
    return ok
}
test appendComp-5.1 {long lappends} {
    catch {unset x}
    set x ""
    for {set i 0} {$i < 300} {set i [expr $i+1]} {
	lappend x "item $i"
    }
    check $x 300
} ok

test appendComp-6.1 {lappend errors} {
    proc foo {} {lappend}
    list [catch {foo} msg] $msg
} {1 {wrong # args: should be "lappend varName ?value value ...?"}}
test appendComp-6.2 {lappend errors} {
    proc foo {} {
	set x ""
	lappend x(0) 44
    }
    list [catch {foo} msg] $msg
} {1 {can't set "x(0)": variable isn't array}}

test appendComp-7.1 {lappendComp-created var and error in trace on that var} {
    proc bar {} {
	global x
	catch {rename foo ""}
	catch {unset x}
	trace variable x w foo
	proc foo {} {global x; unset x}
	catch {lappend x 1}
	proc foo {args} {global x; unset x}
	info exists x
	set x
	lappend x 1
	list [info exists x] [catch {set x} msg] $msg
    }
    bar
} {0 1 {can't read "x": no such variable}}
test appendComp-7.2 {lappend var triggers read trace, index var} {
    proc bar {} {
	catch {unset myvar}
	catch {unset ::result}
	trace variable myvar r foo
	proc foo {args} {append ::result $args}
	lappend myvar a
	list [catch {set ::result} msg] $msg
    }
    bar
} {0 {myvar {} r}}
test appendComp-7.3 {lappend var triggers read trace, stack var} {
    proc bar {} {
	catch {unset ::myvar}
	catch {unset ::result}
	trace variable ::myvar r foo
	proc foo {args} {append ::result $args}
	lappend ::myvar a
	list [catch {set ::result} msg] $msg
    }
    bar
} {0 {::myvar {} r}}
test appendComp-7.4 {lappend var triggers read trace, array var} {
    # The behavior of read triggers on lappend changed in 8.0 to
    # not trigger them.  Maybe not correct, but been there a while.
    proc bar {} {
	catch {unset myvar}
	catch {unset ::result}
	trace variable myvar r foo
	proc foo {args} {append ::result $args}
	lappend myvar(b) a
	list [catch {set ::result} msg] $msg
    }
    bar
} {0 {myvar b r}}
test appendComp-7.5 {lappend var triggers read trace, array var} {
    # The behavior of read triggers on lappend changed in 8.0 to
    # not trigger them.  Maybe not correct, but been there a while.
    proc bar {} {
	catch {unset myvar}
	catch {unset ::result}
	trace variable myvar r foo
	proc foo {args} {append ::result $args}
	lappend myvar(b) a b
	list [catch {set ::result} msg] $msg
    }
    bar
} {0 {myvar b r}}
test appendComp-7.6 {lappend var triggers read trace, array var exists} {
    proc bar {} {
	catch {unset myvar}
	catch {unset ::result}
	set myvar(0) 1
	trace variable myvar r foo
	proc foo {args} {append ::result $args}
	lappend myvar(b) a
	list [catch {set ::result} msg] $msg
    }
    bar
} {0 {myvar b r}}
test appendComp-7.7 {lappend var triggers read trace, array stack var} {
    proc bar {} {
	catch {unset ::myvar}
	catch {unset ::result}
	trace variable ::myvar r foo
	proc foo {args} {append ::result $args}
	lappend ::myvar(b) a
	list [catch {set ::result} msg] $msg
    }
    bar
} {0 {::myvar b r}}
test appendComp-7.8 {lappend var triggers read trace, array stack var} {
    proc bar {} {
	catch {unset ::myvar}
	catch {unset ::result}
	trace variable ::myvar r foo
	proc foo {args} {append ::result $args}
	lappend ::myvar(b) a b
	list [catch {set ::result} msg] $msg
    }
    bar
} {0 {::myvar b r}}
test appendComp-7.9 {append var does not trigger read trace} {
    proc bar {} {
	catch {unset myvar}
	catch {unset ::result}
	trace variable myvar r foo
	proc foo {args} {append ::result $args}
	append myvar a
	info exists ::result
    }
    bar
} {0}

catch {unset i x result y}
catch {rename foo ""}
catch {rename bar ""}
catch {rename check ""}
catch {rename bar {}}

# cleanup
::tcltest::cleanupTests
return
