/* wptCryptdisk.cpp
 *	Copyright (C) 2004 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <windows.h>
#include <winioctl.h>
#include <stdio.h>
#include <ctype.h>

#include "../resource.h"
#include "wptCryptdisk.h"
#include "wptTypes.h"
#include "wptW32API.h"
#include "wptNLS.h"
#include "wptErrors.h"

/*DWORD SHFormatDrive(HWND hwnd, UINT drive, UINT fmtID, UINT options);*/
typedef DWORD (WINAPI *sh_format_drive_p) (HWND, UINT, UINT, UINT);

static mount_list_t mounted=NULL;
static HMODULE shell32=NULL;
static sh_format_drive_p shfmt=NULL;


static int 
is_nt4 (void)
{
    static int nt_flag = -1;
    OSVERSIONINFO osinf;

    if (nt_flag != -1)
	return nt_flag;
    
    memset (&osinf, 0, sizeof osinf);
    osinf.dwOSVersionInfoSize = sizeof osinf;
    GetVersionEx (&osinf);
    if (osinf.dwPlatformId == VER_PLATFORM_WIN32_NT) {
	nt_flag = 1;
	shell32 = LoadLibrary ("SHELL32.DLL");
	shfmt = (sh_format_drive_p)GetProcAddress (shell32, "SHFormatDrive");
	if (!shfmt) 
	    BUG (0);
    }
    else
	nt_flag = 0;
    return nt_flag;
}

static int
check_filedisk (void)
{
    HKEY hk;
    int rc = -1;

    rc = RegOpenKeyEx (HKEY_LOCAL_MACHINE, 
		       "SYSTEM\\CurrentControlSet\\Services\\FileDisk",
		       0, KEY_READ, &hk);
    if (rc == ERROR_SUCCESS) {
	RegCloseKey (hk);
	rc = 0;
    }
    return rc;
}


int
cryptdisk_available (void)
{
    return (is_nt4 () == 1 && check_filedisk () == 0)? 1 : 0;
}


void
cryptdisk_cleanup (void)
{
    if (shell32)
	FreeLibrary (shell32);
    shell32 = NULL;
}


static int
num_of_devices (void)
{
    HKEY hk;
    BYTE buf[32];
    DWORD n=0, type=0;
    int rc;

    rc = RegOpenKeyEx (HKEY_LOCAL_MACHINE,
		       "SYSTEM\\CurrentControlSet\\Services\\FileDisk\\Parameters",
		       0, KEY_READ, &hk);
    if (rc != ERROR_SUCCESS)
	return -1;

    n = sizeof buf-1;
    memset (buf, 0, sizeof buf);
    rc = RegQueryValueEx (hk, "NumberOfDevices", 0, &type, buf, &n);
    if (rc != ERROR_SUCCESS || type != REG_DWORD)
	n=0;
    else
	n = *buf;
    RegCloseKey (hk);
    return n;
}


static HWND
start_cryptdisk_server (HANDLE * r_proc)
{
    STARTUPINFO si;
    PROCESS_INFORMATION pi;

    memset (&si, 0, sizeof si);
    memset (&pi, 0, sizeof pi);
    si.cb = sizeof si;
    
    if (CreateProcess (NULL, "CryptdiskSrv", NULL, NULL, FALSE, 0,
		       NULL, NULL, &si, &pi) == FALSE) {
	msg_box (NULL, _("Could not execute Cryptdisk Server."),
		 _("Cryptdisk Error"), MB_ERR);
	return NULL;
    }
    *r_proc = pi.hProcess;
    Sleep (300);
    return FindWindow ("CryptdiskSrv", NULL);
}


static int
cryptdisk_mount (int devnum,
		 const char * imgfile,
		 u32 size,
		 const char * hdlet)
{
    HWND h;
    HANDLE proc, wait_ev;
    COPYDATASTRUCT cds;
    struct private_ofi_s ofi;

    h = start_cryptdisk_server (&proc);
    if (!h)
	return WPTERR_GENERAL;
    memset (&ofi, 0, sizeof ofi);
    ofi.cmd = CMD_MOUNT;
    ofi.devnum = devnum;
    ofi.filesize.LowPart = size;
    ofi.drvlet = *hdlet;
    ofi.readonly = 0;
    strcpy ((char *)ofi.fname, imgfile);
    ofi.flen = strlen (imgfile);

    wait_ev = OpenEvent (SYNCHRONIZE, FALSE, "cryptdisksrv.wait");
    if (!wait_ev)
	BUG (0);
    memset (&cds, 0, sizeof cds);
    cds.cbData = sizeof ofi;
    cds.lpData = &ofi;
    SendMessage (h, WM_COPYDATA, (WPARAM)GetDesktopWindow (), (LPARAM)&cds);
    WaitForSingleObject (wait_ev, INFINITE);
    CloseHandle (wait_ev);
    CloseHandle (proc);
    return 0;
}



int
cryptdisk_unmount (char drivelet, int forced)
{
    HWND h;
    HANDLE proc;
    COPYDATASTRUCT cds;
    struct private_ofi_s ofi;

    h = start_cryptdisk_server (&proc);
    if (!h)
	return WPTERR_GENERAL;

    memset (&ofi, 0, sizeof ofi);
    ofi.cmd = CMD_UMOUNT;
    ofi.drvlet = drivelet;
    ofi.forced = forced;

    memset (&cds, 0, sizeof cds);
    cds.lpData = &ofi;
    cds.cbData = sizeof ofi;
    SendMessage (h, WM_COPYDATA, (WPARAM)GetDesktopWindow (), (LPARAM)&cds);
    CloseHandle (proc);

    return 0;
}

static void 
print_lasterr (char * prefix)
{
    LPVOID msg;

    FormatMessage(
        FORMAT_MESSAGE_ALLOCATE_BUFFER |
        FORMAT_MESSAGE_FROM_SYSTEM |
        FORMAT_MESSAGE_IGNORE_INSERTS,
        NULL,
        GetLastError(),
        0,
        (LPTSTR) &msg,
        0,
        NULL
        );
    MessageBox (NULL, (LPTSTR)msg, prefix, MB_ICONERROR|MB_OK);
    LocalFree (msg);
}


static int
filedisk_status (char drivelet, LARGE_INTEGER * size, int * rdonly)
{
    char                    VolumeName[] = "\\\\.\\ :";
    HANDLE                  Device;
    POPEN_FILE_INFORMATION  ofi;
    DWORD                   BytesReturned;
    int n= sizeof (OPEN_FILE_INFORMATION)+ MAX_PATH;

    VolumeName[4] = drivelet;

    Device = CreateFile (VolumeName,
			 GENERIC_READ,
			 FILE_SHARE_READ | FILE_SHARE_WRITE,
			 NULL,
			 OPEN_EXISTING,
			 FILE_FLAG_NO_BUFFERING,
			NULL);

    if (Device == INVALID_HANDLE_VALUE) {
        print_lasterr(&VolumeName[4]);
        return WPTERR_CDISK_OPEN;
    }

    ofi = (POPEN_FILE_INFORMATION)malloc (n);

    if (!DeviceIoControl (Device,
			  IOCTL_FILE_DISK_QUERY_FILE,
			  NULL,
			  0,
			  ofi,
			  sizeof(OPEN_FILE_INFORMATION) + MAX_PATH,
			  &BytesReturned, NULL)) {
        print_lasterr (&VolumeName[4]);
        return WPTERR_CDISK_QUERY;
    }

    if (BytesReturned < sizeof(OPEN_FILE_INFORMATION)) {
        SetLastError (ERROR_INSUFFICIENT_BUFFER);
        print_lasterr (&VolumeName[4]);
        return WPTERR_CDISK_QUERY;
    }

    if (size)
	*size = ofi->FileSize;
    if (rdonly)
	*rdonly = ofi->ReadOnly;

    return 0;
}


static mount_list_t
new_mount_list (const char * drive, int devnum)
{
    mount_list_t n;

    n = (mount_list_t)calloc (1, sizeof *n);
    if (n) {
	strcpy (n->drive, drive);
	n->devnum = devnum;
	n->ttl = 1;
	filedisk_status (*drive, &n->size, &n->rdonly);
    }
    return n;
}


static mount_list_t
find_mount_list (mount_list_t root, const char * drive)
{
    mount_list_t n;

    for (n=root; n; n=n->next) {
	if (n->ttl == 0)
	    continue;
	if (!strncmp (n->drive, drive, strlen (drive)))
	    return n;
    }
    return NULL;
}


static void
add_mount_list (mount_list_t root, mount_list_t node)
{
    mount_list_t n;

    if (find_mount_list (root, node->drive))
	return;

    for (n=root; n->next; n=n->next)
	;
    n->next = node;
}


static void
remove_mount_list (mount_list_t root, mount_list_t node)
{
    mount_list_t n;

    n = find_mount_list (root, node->drive);
    if (n)
	n->ttl = 0;
}


static void
free_mount_list (mount_list_t root)
{
    mount_list_t n;

    while (root) {
	n = root->next;
	free (root);
	root = n;
    }
}


static int
last_devnum (mount_list_t root)
{
    mount_list_t n;
    int devnum=0;

    if (!root)
	return 0;

    for (n=root; n; n=n->next) {
	if (devnum < n->devnum)
	    devnum=n->devnum;
    }
    return devnum+1;
}


static char
init_drives (HWND dlg, int ctlid)
{
    char buf[512], drv[5];
    DWORD n=0, i=0;

    n=GetLogicalDriveStrings (sizeof buf-1, buf);
    for (i=0; i < n; i++) {
	memcpy (drv, buf+i, 3); drv[3]=0;
	i += 3;
    }
    if (!dlg && ctlid == -1)
	return ++drv[0];

    for (++drv[0]; drv[0] < 'Z'; drv[0]++)
	SendDlgItemMessage (dlg, ctlid, CB_ADDSTRING, 0,
			    (LPARAM)(const char *)drv);
    SendDlgItemMessage (dlg, ctlid, CB_SETCURSEL, 0, 0);
    return 0;
}


static int
check_freespace (const char * imgfile, u32 size)
{
    ULARGE_INTEGER tocaller, total, totfree;
    char buf[128] = {0};

    if (!strchr (imgfile, ':'))
	GetCurrentDirectory (sizeof buf-1, buf);
    else
	strncpy (buf, imgfile, sizeof buf-1);
    buf[3] = '\0';
    GetDiskFreeSpaceEx (buf, &tocaller, &total, &totfree);
    if (size > tocaller.LowPart)
	return -1;
    return 0;
}


static void
do_check (HWND dlg)
{
    if (!cryptdisk_serv_run ()) { /* xxx */
	msg_box (dlg, _("The Cryptdisk service seems to be available but "
		        "it is not started yet.\nPlease start the service "
			"and try again."), _("Cryptdisk Error"), MB_ERR);
	EndDialog (dlg, TRUE);
    }
}


BOOL CALLBACK
cryptdisk_new_dlg_proc (HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam)
{
    const char * ciphers [] = {
	"twofish",
	NULL
    };
    const char * s;
    char imgfile[128], key[128], drv[3];
    int i;
    u32 size=0;

    switch (msg) {
    case WM_INITDIALOG:
	do_check (dlg);
	SetDlgItemInt (dlg, IDC_CDNEW_SIZE, DISK_DEFSIZE, FALSE);
	for (i=0; (s=ciphers[i]); i++)
	    SendDlgItemMessage (dlg, IDC_CDNEW_CIPHERS, CB_ADDSTRING,
				     0, (LPARAM)(const char *) s);
	SendDlgItemMessage (dlg, IDC_CDNEW_CIPHERS, CB_SETCURSEL, 0, 0);
	center_window (dlg);
	SetForegroundWindow (dlg);
	break;

    case WM_COMMAND:
	switch (LOWORD (wparam)) {
	case IDOK:
	    i = GetDlgItemText (dlg, IDC_CDNEW_IMGFILE, imgfile, sizeof imgfile-1);
	    if (!i) {
		msg_box (dlg, _("Please enter a name for the image file."),
			 _("Cryptdisk"), MB_ERR);
		return FALSE;
	    }
	    if (file_exist_check (imgfile) == 0) {
		i = msg_box (dlg, _("This volume file already exists.\n"
				    "Do you want to overwrite it?"),
			    _("Cryptdisk Warning"), MB_YESNO|MB_WARN);
		if (i == IDNO)
		    return FALSE;
	    }
	    for (i=0; i < (int)strlen (imgfile); i++)
		imgfile[i] = toupper (imgfile[i]);
	    size = GetDlgItemInt (dlg, IDC_CDNEW_SIZE, NULL, FALSE);
	    if (!size) {
		msg_box (dlg, _("Please enter the size for the volume"),
				_("Cryptdisk"), MB_INFO);
		return FALSE;
	    }
	    size *= 1024; /*KB*/
	    if (check_freespace (imgfile, (size*1024))) {
		msg_box (dlg, _("There is not enough free disk space to "
				"store the volume."), _("Cryptdisk"), MB_ERR);
		return FALSE;
	    }
	    i = GetDlgItemText (dlg, IDC_CDNEW_PASS, key, sizeof key-1);
	    if (!i) {
		msg_box (dlg, _("Please enter a passphrase for the volume."),
			    _("Cryptdisk"), MB_ERR);
		return FALSE;
	    }
	    strcpy (drv, "?:\\");
	    drv[0] = init_drives (NULL, -1); 
	    i = cryptdisk_mount (last_devnum (mounted), imgfile, size, drv);
	    if (!i)
		shfmt (dlg, drv[0] - 65, 0, 0);
	    break;
	case IDCANCEL:
	    EndDialog (dlg, FALSE);
	    break;
	}
	break;

    }

    return FALSE;
}


BOOL CALLBACK
cryptdisk_mount_dlg_proc (HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam)
{
    const char * s_imgfile;
    char buf[8], drive[8], passwd[64], imgfile[128];
    int n, i, devnum=0;
    int rc;

    switch (msg) {
    case WM_INITDIALOG:
	do_check (dlg);
	s_imgfile = (const char *)lparam;
	if (s_imgfile)
	    SetDlgItemText (dlg, IDC_CDMOUNT_IMGFILE, s_imgfile);
	init_drives (dlg, IDC_CDMOUNT_DRV);
	n = num_of_devices ();
	if (n == -1) {
	    msg_box (dlg, _("Cannot determine the number of drives."),
		     _("Cryptdisk Error"), MB_ERR);
	    EndDialog (dlg, FALSE);
	}
	for (i=last_devnum (mounted); i < n; i++) {
	    sprintf (buf, "%d", i);
	    SendDlgItemMessage (dlg, IDC_CDMOUNT_ID, CB_ADDSTRING, 0,
				(LPARAM)(const char *)buf);
	}
	SendDlgItemMessage (dlg, IDC_CDMOUNT_ID, CB_SETCURSEL, 0, 0);
	center_window (dlg);
	SetForegroundWindow (dlg);
	break;

    case WM_COMMAND:
	switch (LOWORD (wparam)) {
	case IDC_CDMOUNT_SELFILE:
	    s_imgfile = get_filename_dlg (dlg, FILE_OPEN, _("Select Crypdisk Volume"), 
					  NULL, NULL);
	    if (s_imgfile && !file_exist_check (s_imgfile))
		SetDlgItemText (dlg, IDC_CDMOUNT_IMGFILE, s_imgfile);
	    break;

	case IDOK:
	    n = item_get_text_length (dlg, IDC_CDMOUNT_IMGFILE);
	    if (!n) {
		msg_box (dlg, _("Please enter the name of the image file."),
			 _("Cryptdisk Error"), MB_ERR);
		return FALSE;
	    }
	    n = item_get_text_length (dlg, IDC_CDMOUNT_PASS);
	    if (!n) {
		msg_box (dlg, _("Please enter a password."), 
			 _("Cryptdisk Error"), MB_ERR);
		return FALSE;
	    }
	    devnum = GetDlgItemInt (dlg, IDC_CDMOUNT_ID, NULL, FALSE);
	    GetDlgItemText (dlg, IDC_CDMOUNT_DRV, drive, sizeof drive-1);
	    GetDlgItemText (dlg, IDC_CDMOUNT_PASS, passwd, sizeof passwd-1);
	    GetDlgItemText (dlg, IDC_CDMOUNT_IMGFILE, imgfile, sizeof imgfile-1);
	    if (file_exist_check  (imgfile)) {
		msg_box (dlg, _("Image file does not exist or could not be accessed."),
			 _("Cryptdisk Error"), MB_ERR);
		return FALSE;
	    }
	    rc = cryptdisk_mount (devnum, imgfile, 0, drive);
	    if (!rc) {
		mount_list_t n = new_mount_list (drive, devnum);
		if (!mounted)
		    mounted = n;
		else
		    add_mount_list (mounted, n);
		EndDialog (dlg, TRUE);
	    }
	    else
		msg_box (dlg, winpt_strerror (rc), _("Cryptdisk Error"), MB_ERR);
	    break;

	case IDCANCEL:
	    EndDialog (dlg, FALSE);
	    break;
	}
	break;
    }
    return FALSE;
}


static void
load_devs_from_mountlist (HWND dlg, int ctlid)
{
    mount_list_t n;
    char buf[128];

    if (!mounted)
	return;

    for (n=mounted; n; n=n->next) {
	if (n->ttl == 0)
	    continue;
	_snprintf (buf, sizeof buf-1, _("Drive %s (ID %d); Size %d MB, Readonly=%s"), 
		    n->drive, n->devnum, n->size.LowPart/1024/1024, 
		    n->rdonly? "true" : "false");
	SendDlgItemMessage (dlg, ctlid, LB_ADDSTRING, 0, (LPARAM)(const char *)buf);
    }

}


static void
do_reaping (void)
{
    mount_list_t n;
    int ndevs=0, numount=0;

    for (n=mounted; n; n=n->next) {
	if (n->ttl == 0)
	    numount++;
	ndevs++;
    }
    if (ndevs == numount) {
	free_mount_list (mounted);
	mounted = NULL;
    }
}


BOOL CALLBACK
cryptdisk_umount_dlg_proc (HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam)
{    
    mount_list_t vol;
    char buf[128];
    int n, rc=0;

    switch (msg) {
    case WM_INITDIALOG:
	do_check (dlg);
	load_devs_from_mountlist (dlg, IDC_CDUMOUNT_LIST);
	center_window (dlg);
	SetForegroundWindow (dlg);
	break;

    case WM_DESTROY:
	do_reaping ();
	break;

    case WM_COMMAND:
	switch (LOWORD (wparam)) {
	case IDOK:
	    n = SendDlgItemMessage (dlg, IDC_CDUMOUNT_LIST, LB_GETCURSEL, 0, 0);
	    if (n == LB_ERR) {
		msg_box (dlg, _("Please select one drive to umount."), _("Cryptdisk"), MB_INFO);
		return FALSE;
	    }
	    SendDlgItemMessage (dlg, IDC_CDUMOUNT_LIST, LB_GETTEXT, n, 
				(LPARAM)(char *)buf);
	    strcpy (buf, buf+6);
	    buf[2] = '\0';
	    vol = find_mount_list (mounted, buf);
	    if (!vol)
		BUG (0);
	    rc = cryptdisk_unmount (buf[0], 0);
	    //if (rc == WPTERR_CDISK_LOCK)
	//	rc = cryptdisk_unmount (buf[0], 0);
	    if (!rc) {
		SendDlgItemMessage (dlg, IDC_CDUMOUNT_LIST, LB_DELETESTRING, (WPARAM)n, 0);
		remove_mount_list (mounted, vol);
	    }
	    else
		msg_box (dlg, winpt_strerror (rc), _("Cryptdisk Error"), MB_ERR);
	    break;
	case IDCANCEL:
	    EndDialog (dlg, FALSE);
	    break;
	}
	break;
    }

    return FALSE;
}


int
cryptdisk_serv_run (void)
{
    SC_HANDLE hd;
    ENUM_SERVICE_STATUS ess[4096];
    DWORD nservs=0, dummy=0;
    DWORD i;
    int rc=0;

    hd = OpenSCManager (NULL, NULL, SC_MANAGER_ENUMERATE_SERVICE);
    if (!hd)
	return 0;

    EnumServicesStatus (hd,  SERVICE_DRIVER, SERVICE_ACTIVE,
			(LPENUM_SERVICE_STATUS)&ess,
			sizeof ess-1, &dummy, &nservs, &dummy);

    for (i=0; i < nservs; i++) {
	if (!strnicmp ("FileDisk", ess[i].lpDisplayName, 8)) {
	    rc = 1;
	    break;
	}
    }

    CloseServiceHandle (hd);
    return rc;
}