/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gtk;

import org.gnu.glib.GObject;
import org.gnu.glib.Type;

/**
 * This widget is similar to DrawingArea in that it's a "blank slate" and 
 * doesn't do anything but paint a blank background by default.  It's different
 * in that it supports scrolling natively and it contains child widgets since
 * it is a Container.
 * 
 * @see Container
 */
public class Layout extends Container 
{
	
	/**
	 * Construct a new Layout object.
	 * 
	 * @param hAdj The horizontal Adjustment object.
	 * @param vAdj The vertical Adjustment object.
	 */
	public Layout(Adjustment hAdj, Adjustment vAdj) {
		setHandle(Layout.gtk_layout_new(hAdj.getHandle(), vAdj.getHandle()));
		initializeEventHandlers();
	}
	
	protected Layout() {}
	
	/**
	 * Construct a new Layout from a handle to a native resource.
	 */
	public Layout(int handle) {
	    setHandle(handle);
		initializeEventHandlers();
	}
	
	protected void noopInit() {}
	
	/**
	 * Add a child Widget to this Layout.
	 * 
	 * @param child The child Widget to add to this Layout.
	 * @param x The X coordinate to position the child.
	 * @param y The Y coordinate to position the child.
	 */
	public void addChild(Widget child, int x, int y) {
		Layout.gtk_layout_put(getHandle(), child.getHandle(), x, y);
	}
	
	/**
	 * Move a child Widget to a new position
	 * 
	 * @param child The child Widget to move.
	 * @param x The X coordinate for the new position.
	 * @param y The Y coordinate for the new position.
	 */
	public void moveChild(Widget child, int x, int y) {
		Layout.gtk_layout_move(getHandle(), child.getHandle(), x, y);
	}
	
	/**
	 * Set the size of the scrollable area of the layout.
	 * 
	 * @param width The width for the new size.
	 * @param height The height for the new size.
	 */
	public void setSize(int width, int height) {
		Layout.gtk_layout_set_size(getHandle(), width, height);
	}
	
	/**
	 * Retrieve the size of the scrollabel area of the layout.
	 * 
	 * @return The size of the scrollabel area.
	 */
	public Requisition getSize() {
		int [] width = new int[1];
		int [] height = new int[1];
		Layout.gtk_layout_get_size(getHandle(), width, height);
		return new Requisition(width[0], height[0]);
	}
	
	/**
	 * Return the Horizontal Adjustment.
	 */
	public Adjustment getHorizontalAdjustment() {
		int han = Layout.gtk_layout_get_hadjustment(getHandle());
		GObject obj = retrieveGObject(han);
		if (null != obj)
			return (Adjustment)obj;
		return new Adjustment(han);
	}
	
	/**
	 * Set the Horizontal Adjustment.
	 */
	public void setHorizontalAdjustment(Adjustment adjust) {
		Layout.gtk_layout_set_hadjustment(getHandle(), adjust.getHandle());
	}
	
	/**
	 * Return the Vertical Adjustment.
	 */
	public Adjustment getVerticalAdjustment() {
		int han = Layout.gtk_layout_get_vadjustment(getHandle());
		GObject obj = retrieveGObject(han);
		if (null != obj)
			return (Adjustment)obj;
		return new Adjustment(han);
	}
	
	/**
	 * Set the Vertical Adjustment.
	 */
	public void setVerticalAdjustment(Adjustment adjust) {
		Layout.gtk_layout_set_vadjustment(getHandle(), adjust.getHandle());
	}
	
	/**
	 * Retrieve the runtime type used by the GLib library.
	 */
	public static Type getType() {
		return new Type(gtk_layout_get_type());
	}

    /****************************************
     * BEGINNING OF GENERATED CODE
     ****************************************/
    native static final protected int gtk_layout_get_type ();
    native static final protected int gtk_layout_new (int hadjustment, int vadjustment);
    native static final protected void gtk_layout_put (int layout, int childWidget, int x, int y);
    native static final protected void gtk_layout_move (int layout, int childWidget, int x, int y);
    native static final protected void gtk_layout_set_size (int layout, int width, int height);
    native static final protected void gtk_layout_get_size (int layout, int [] width, int [] height);
    native static final protected int gtk_layout_get_hadjustment (int layout);
    native static final protected int gtk_layout_get_vadjustment (int layout);
    native static final protected void gtk_layout_set_hadjustment (int layout, int adjustment);
    native static final protected void gtk_layout_set_vadjustment (int layout, int adjustment);
    /****************************************
     * END OF GENERATED CODE
     ****************************************/
}

