/* This file is part of GNU Pies.
   Copyright (C) 2009 Sergey Poznyakoff

   GNU Pies is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3, or (at your option)
   any later version.

   GNU Pies is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with GNU Pies.  If not, see <http://www.gnu.org/licenses/>. */

#include "pies.h"
#include "xvasprintf.h"

unsigned debug_level;
int source_info_option;
int diag_output = DIAG_TO_STDERR;

void
diag_setup (int flags)
{
  if (flags)
    diag_output = flags;
  if (diag_output & DIAG_TO_SYSLOG)
    openlog (log_tag, LOG_PID, log_facility);
}

void
syslog_printer (int prio, const char *fmt, va_list ap)
{
#if HAVE_VSYSLOG
  vsyslog (prio, fmt, ap);
#else
  char buf[128];
  vsnprintf (buf, sizeof buf, fmt, ap);
  syslog (prio, "%s", buf);
#endif
}

void
vlogmsg (int prio, const char *fmt, va_list ap)
{
  if (DIAG_OUTPUT (DIAG_TO_STDERR))
    {
      fprintf (stderr, "%s: ", program_name);
      vfprintf (stderr, fmt, ap);
      fprintf (stderr, "\n");
    }
  if (DIAG_OUTPUT (DIAG_TO_SYSLOG))
    syslog_printer (prio, fmt, ap);
}

void
logmsg (int prio, const char *fmt, ...)
{
  va_list ap;
  va_start (ap, fmt);
  vlogmsg (prio, fmt, ap);
  va_end (ap);
}

void
debug_msg (const char *fmt, ...)
{
  va_list ap;
  va_start (ap, fmt);
  vlogmsg (LOG_DEBUG, fmt, ap);
  va_end (ap);
}


static struct obstack log_stk;
static int log_stk_init;

void
logmsg_vprintf (int prio, const char *fmt, va_list ap)
{
  char *str, *p;
  
  str = xvasprintf (fmt, ap);

  if (!log_stk_init)
    {
      obstack_init (&log_stk);
      log_stk_init = 1;
    }
  for (p = str; *p; )
    {
      size_t len = strcspn (p, "\n");
      if (len)
	obstack_grow (&log_stk, p, len);
      p += len;
      if (*p)
	{
	  char *msg;
	  
	  obstack_1grow (&log_stk, 0);
	  msg = obstack_finish (&log_stk);
	  logmsg (prio, "%s", msg);
	  obstack_free (&log_stk, msg);
	  p++;
	}
    }
  free (str);
}

void
logmsg_printf (int prio, const char *fmt, ...)
{
  va_list ap;
  va_start (ap, fmt);
  logmsg_vprintf (prio, fmt, ap);
  va_end (ap);
}

void
grecs_print_diag (grecs_locus_t *locus, int err, int errcode, const char *msg)
{
  if (locus)
    {
      if (errcode)
	logmsg (err ? LOG_ERR : LOG_WARNING, "%s:%lu: %s: %s",
		locus->file, (unsigned long)locus->line, msg,
		strerror (errcode));
      else
	logmsg (err ? LOG_ERR : LOG_WARNING, "%s:%lu: %s",
		locus->file, (unsigned long)locus->line, msg);
    }
  else
    {
      if (errcode)
	logmsg (err ? LOG_ERR : LOG_WARNING, "%s: %s", msg,
		strerror (errcode));
      else
	logmsg (err ? LOG_ERR : LOG_WARNING, "%s", msg);
    }
}
  
