#!/bin/bash

set -e

PR_URL="$1"

if [ -z "$PR_URL" ]; then
    echo "Usage: $0 <PR_URL>"
    echo ""
    echo "Check if the PR references any non-help-wanted issues and, if so, comment"
    echo "on it explaining why the team might close/dismiss it."
    exit 1
fi

# Skip if PR is from a bot or org member
if [ "$PR_AUTHOR_TYPE" = "Bot" ] || [ "$PR_AUTHOR_ASSOCIATION" = "MEMBER" ] || [ "$PR_AUTHOR_ASSOCIATION" = "OWNER" ]; then
    echo "Skipping check for PR $PR_URL as it is from a bot ($PR_AUTHOR_TYPE) or an org member ($PR_AUTHOR_ASSOCIATION: MEMBER/OWNER)"
    exit 0
fi

# Skip if PR is a draft
if [ "$(gh pr view "${PR_URL}" --json isDraft --jq '.isDraft')" != "false" ]; then
    echo "Skipping check for PR $PR_URL as it is a draft"
    exit 0
fi

# Extract PR number from URL for logging
PR_NUM="$(basename "$PR_URL")"

# Extract cli/cli closing issues references from PR
CLOSING_ISSUES="$(gh pr view "$PR_URL" --json closingIssuesReferences --jq '.closingIssuesReferences[] | select(.repository.name == "cli" and .repository.owner.login == "cli") | .number')"

if [ -z "$CLOSING_ISSUES" ]; then
    echo "No closing issues found for PR #$PR_NUM"
    exit 0
fi

# Check each closing issue for 'help-wanted' label
ISSUES_WITHOUT_HELP_WANTED=()

for issue_num in $CLOSING_ISSUES; do
    echo "Checking issue #$issue_num for 'help wanted' label..."

    # Get issue labels
    LABELS=$(gh issue view "$issue_num" --json labels --jq '.labels[].name')

    # Skip if the issue has the gh-attestion or gh-codespace label
    # This is because the codeowners for these commands may not be public
    # cli org members, and so unless we authenticate with a PAT, we can't
    # know who is an external contributor or not.
    # So we skip these issues to avoid falsely writing a comment
    # on each PR opened by these codeowners.
    if echo "$LABELS" | grep -q -e "gh-attestation" -e "gh-codespace"; then
        echo "Issue #$issue_num is skipped due to labels"
        continue
    fi

    # Check if 'help wanted' label exists
    if ! echo "$LABELS" | grep -q "help wanted"; then
        ISSUES_WITHOUT_HELP_WANTED+=("$issue_num")
        echo "Issue #$issue_num does not have 'help wanted' label"
    else
        echo "Issue #$issue_num has 'help wanted' label"
    fi
done

# If we found issues without 'help wanted' label, post a comment
if [ ${#ISSUES_WITHOUT_HELP_WANTED[@]} -gt 0 ]; then
    echo "Found ${#ISSUES_WITHOUT_HELP_WANTED[@]} issues without 'help wanted' label"

    # Build issue list for comment
    ISSUE_LIST=""
    for issue_num in "${ISSUES_WITHOUT_HELP_WANTED[@]}"; do
        ISSUE_LIST="$ISSUE_LIST- #$issue_num"$'\n'
    done

    # Create comment message
    gh pr comment "$PR_URL" --body-file - <<EOF
Thank you for your pull request! 🎉

This PR appears to fix the following issues that are not labeled with \`help wanted\`:

$ISSUE_LIST
As outlined in our [Contributing Guidelines](https://github.com/cli/cli/blob/trunk/.github/CONTRIBUTING.md), we expect that PRs are only created for issues that have been labeled \`help wanted\`.

While we appreciate your initiative, please note that:

- **PRs for non-\`help wanted\` issues may not be reviewed immediately** as they might not align with our current priorities
- **The issue might already be assigned** to a team member or planned for a specific release
- **We may need to close this PR**. For example, if it conflicts with ongoing work or architectural decisions

**What happens next:**
- Our team will review this PR and the associated issues
- We may add the \`help wanted\` label to the issues, if appropriate, and review this pull request
- In some cases, we may need to close the PR. For example, if it doesn't fit our current roadmap

Thank you for your understanding and contribution to the project! 🙏

*This comment was automatically generated by cliAutomation.*
EOF

    echo "Posted comment on PR #$PR_NUM"
else
    echo "All closing issues have 'help wanted' label - no action needed"
fi
