/*****************************************************************************
* Copyright (c) 2016 CEA LIST.
*
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
*
* Contributors:
*  Arnault Lapitre (CEA LIST) arnault.lapitre@cea.fr
*  - Initial API and implementation
*****************************************************************************/
package org.eclipse.efm.formalml.xtext.scoping

import com.google.inject.Inject
import java.util.HashSet
import java.util.Set
import org.eclipse.efm.ecore.formalml.common.AbstractElement
import org.eclipse.efm.ecore.formalml.common.NamedElement
import org.eclipse.efm.ecore.formalml.common.TypedElement
import org.eclipse.efm.ecore.formalml.common.VisibilityKind
import org.eclipse.efm.ecore.formalml.datatype.DataStructuredType
import org.eclipse.efm.ecore.formalml.datatype.DataSupportedType
import org.eclipse.efm.ecore.formalml.datatype.DataType
import org.eclipse.efm.ecore.formalml.datatype.DataTypeReference
import org.eclipse.efm.ecore.formalml.datatype.DatatypePackage
import org.eclipse.efm.ecore.formalml.datatype.EnumerationLiteral
import org.eclipse.efm.ecore.formalml.datatype.EnumerationType
import org.eclipse.efm.ecore.formalml.datatype.PrimitiveInstanceKind
import org.eclipse.efm.ecore.formalml.datatype.PrimitiveInstanceType
import org.eclipse.efm.ecore.formalml.expression.CastExpression
import org.eclipse.efm.ecore.formalml.expression.ExpressionPackage
import org.eclipse.efm.ecore.formalml.expression.InvokeExpression
import org.eclipse.efm.ecore.formalml.expression.LeftHandSideExpression
import org.eclipse.efm.ecore.formalml.expression.LiteralReferenceElement
import org.eclipse.efm.ecore.formalml.expression.LiteralReferenceExpression
import org.eclipse.efm.ecore.formalml.expression.TupleExpression
import org.eclipse.efm.ecore.formalml.expression.ValueElementSpecification
import org.eclipse.efm.ecore.formalml.expression.ValueElementSpecificationKind
import org.eclipse.efm.ecore.formalml.expression.ValueElementSpecificationScheme
import org.eclipse.efm.ecore.formalml.infrastructure.Behavior
import org.eclipse.efm.ecore.formalml.infrastructure.ChannelDirection
import org.eclipse.efm.ecore.formalml.infrastructure.ComPoint
import org.eclipse.efm.ecore.formalml.infrastructure.ComProtocol
import org.eclipse.efm.ecore.formalml.infrastructure.InfrastructurePackage
import org.eclipse.efm.ecore.formalml.infrastructure.InstanceMachine
import org.eclipse.efm.ecore.formalml.infrastructure.Machine
import org.eclipse.efm.ecore.formalml.infrastructure.Port
import org.eclipse.efm.ecore.formalml.infrastructure.PropertyDefinition
import org.eclipse.efm.ecore.formalml.infrastructure.Route
import org.eclipse.efm.ecore.formalml.infrastructure.Routine
import org.eclipse.efm.ecore.formalml.infrastructure.Signal
import org.eclipse.efm.ecore.formalml.infrastructure.SlotProperty
import org.eclipse.efm.ecore.formalml.infrastructure.Variable
import org.eclipse.efm.ecore.formalml.statemachine.Region
import org.eclipse.efm.ecore.formalml.statemachine.Statemachine
import org.eclipse.efm.ecore.formalml.statemachine.StatemachinePackage
import org.eclipse.efm.ecore.formalml.statemachine.Transition
import org.eclipse.efm.ecore.formalml.statement.AbstractComStatement
import org.eclipse.efm.ecore.formalml.statement.ActivityStatement
import org.eclipse.efm.ecore.formalml.statement.InputComStatement
import org.eclipse.efm.ecore.formalml.statement.InvokeStatement
import org.eclipse.efm.ecore.formalml.statement.OutputComStatement
import org.eclipse.efm.ecore.formalml.statement.StatementPackage
import org.eclipse.efm.formalml.xtext.typing.FormalMLTypeProvider
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.EReference
import org.eclipse.xtext.EcoreUtil2
import org.eclipse.xtext.scoping.IScope
import org.eclipse.xtext.scoping.Scopes
import org.eclipse.xtext.xbase.lib.Functions.Function1
import org.eclipse.efm.ecore.formalml.expression.QuantifiedLogicalExpression
import org.eclipse.efm.ecore.formalml.expression.LiteralNullExpression

/**
 * This class contains custom scoping description.
 * 
 * See https://www.eclipse.org/Xtext/documentation/303_runtime_concepts.html#scoping
 * on how and when to use it.
 */
class FormalMLScopeProvider extends AbstractFormalMLScopeProvider {

	@Inject extension FormalMLTypeProvider
	
	
	////////////////////////////////////////////////////////////////////////////
	// GET SCOPE
	////////////////////////////////////////////////////////////////////////////
	
//	static val DEFINED_SCOPES = #{
//		// OK
//		"scope_Transition_target(Pseudostate.Transition context, EReference reference)",
//		"scope_Transition_target(State.Transition context, EReference reference)",
//		"scope_Transition_target(StartState.Transition context, EReference reference)",
//
//		// OK
//		"scope_AbstractComStatement_port(BlockStatement.OutputComStatement context, EReference reference)",
//		"scope_AbstractComStatement_port(BlockStatement.InputComStatement context, EReference reference)",
//		
//		// OK
//		"scope_InvokeStatement_invokable(BlockStatement.InvokeStatement context, EReference reference)",
//		
//		// OK
//		"scope_InstanceMachine_model(System.InstanceMachine context, EReference reference)",
//
//		// Statement
//		"scope_LiteralReferenceElement_value(InputComStatement.LiteralReferenceElement context, EReference reference)",
//		"scope_LiteralReferenceElement_value(ActivityStatement.LiteralReferenceElement context, EReference reference)",
//		"scope_LiteralReferenceElement_value(OutputComStatement.LiteralReferenceElement context, EReference reference)",
//		"scope_LiteralReferenceElement_value(ForEachStatement.LiteralReferenceElement context, EReference reference)",
//		"scope_LiteralReferenceElement_value(GuardStatement.LiteralReferenceElement context, EReference reference)",
//		"scope_LiteralReferenceElement_value(IfStatement.LiteralReferenceElement context, EReference reference)",
//		"scope_LiteralReferenceElement_value(InterruptStatement.LiteralReferenceElement context, EReference reference)",
//		// Expression
//		"scope_LiteralReferenceElement_value(ValueElementSpecification.LiteralReferenceElement context, EReference reference)",
//		"scope_LiteralReferenceElement_value(LeftHandSideExpression.LiteralReferenceElement context, EReference reference)",		
//		"scope_LiteralReferenceElement_value(ArithmeticAssociativeExpression.LiteralReferenceElement context, EReference reference)",
//		"scope_LiteralReferenceElement_value(RelationalBinaryExpression.LiteralReferenceElement context, EReference reference)",
//		"scope_LiteralReferenceElement_value(BitwiseAssociativeExpression.LiteralReferenceElement context, EReference reference)",
//		"scope_LiteralReferenceElement_value(LiteralCollectionExpression.LiteralReferenceElement context, EReference reference)",
//		"scope_LiteralReferenceElement_value(LogicalAssociativeExpression.LiteralReferenceElement context, EReference reference)",
//		"scope_LiteralReferenceElement_value(LogicalUnaryExpression.LiteralReferenceElement context, EReference reference)",
//		"scope_LiteralReferenceElement_value(EqualityBinaryExpression.LiteralReferenceElement context, EReference reference)",
//		"scope_LiteralReferenceElement_value(AssignmentExpression.LiteralReferenceElement context, EReference reference)",
//		"scope_LiteralReferenceElement_value(InvokeExpression.LiteralReferenceElement context, EReference reference)",
//		"scope_LiteralReferenceElement_value(MixTupleExpression.LiteralReferenceElement context, EReference reference)",
//		
//		"scope_ValueElementSpecification_element(LiteralCollectionExpression.ValueElementSpecification context, EReference reference)",
//		"scope_LiteralReferenceElement_value(PositionalTupleExpression.LiteralReferenceElement context, EReference reference)",
//		
//		"scope_LiteralReferenceElement_value(ExpressionAsMachine.LiteralReferenceElement context, EReference reference)",
//		
//		// Other
//		"scope_LiteralReferenceElement_value(Variable.LiteralReferenceElement context, EReference reference)",
//
//		// Communication Element OK
//		"scope_ValueElementSpecification_element(ComPoint.ValueElementSpecification context, EReference reference)",
//		"scope_ValueElementSpecification_element(ComProtocol.ValueElementSpecification context, EReference reference)",
//		// Statement OK
//		"scope_ValueElementSpecification_element(OutputComStatement.ValueElementSpecification context, EReference reference)",
//		"scope_ValueElementSpecification_element(InputComStatement.ValueElementSpecification context, EReference reference)",
//		// Expression OK
//		"scope_ValueElementSpecification_element(ValueElementSpecification.ValueElementSpecification context, EReference reference)",
//		"scope_ValueElementSpecification_element(AssignmentExpression.ValueElementSpecification context, EReference reference)",
//		"scope_ValueElementSpecification_element(ArithmeticAssociativeExpression.ValueElementSpecification context, EReference reference)",
//		"scope_ValueElementSpecification_element(Variable.ValueElementSpecification context, EReference reference)",
//		"scope_ValueElementSpecification_element(LeftHandSideExpression.ValueElementSpecification context, EReference reference)",
//		"scope_ValueElementSpecification_element(PositionalTupleExpression.ValueElementSpecification context, EReference reference)",
//		"scope_ValueElementSpecification_element(EqualityBinaryExpression.ValueElementSpecification context, EReference reference)",
//		"scope_ValueElementSpecification_element(MixTupleExpression.ValueElementSpecification context, EReference reference)",
//		// Other OK
//		"scope_ValueElementSpecification_element(IntervalType.ValueElementSpecification context, EReference reference)",
//		
//		
//		"scope_DataTypeReference_typeref(Variable.DataTypeReference context, EReference reference)",
//		"scope_DataTypeReference_typeref(Parameter.DataTypeReference context, EReference reference)",
//		"scope_DataTypeReference_typeref(CollectionType.DataTypeReference context, EReference reference)",
//		"scope_DataTypeReference_typeref(LiteralCollectionExpression.DataTypeReference context, EReference reference)",
//		
//		// OK
//		"scope_Route_signals(ModelOfInteraction.Route context, EReference reference)",
//		// OK
//		"scope_SlotProperty_xliaProperty(InstanceMachine.SlotProperty context, EReference reference)",
//
//TO DO "scope_PrimitiveInstanceType_model(LiteralNullExpression.PrimitiveInstanceType context, EReference reference)",
//
//		// OK
//		"scope_CastExpression_datatype(InterruptStatement.CastExpression context, EReference reference)"
//	}
//	
//	static var Set< String > fRequireScoping = new HashSet( DEFINED_SCOPES )
	
	////////////////////////////////////////////////////////////////////////////
	// getScope
	////////////////////////////////////////////////////////////////////////////
	
	override getScope(EObject context, EReference reference)
	{
		val requiredScoping = "scope_" + reference.EContainingClass.name
			 + "_" + reference.name + "(" + context.eContainer?.eClass.name
			 + "." + context.eClass.name + " context, EReference reference)"
			 
		val scope = getScopeImpl(context, reference, requiredScoping)
	
//		if( scope == IScope::NULLSCOPE )
//		{
//			print( "IScope::NULLSCOPE for: ")
//			println( requiredScoping )
//		}
//		else
//		{
//			println( requiredScoping )
//			print( "\tIScope: ")
//			println( scope )
//		}
		
		scope
	}
	
	
	def getScopeImpl(EObject context, EReference reference, String requiredScoping)
	{
		switch( reference )
		{
			case StatemachinePackage.Literals.TRANSITION__TARGET:
			{
				return scope_Transition_target(context as Transition, reference)
			}
			
			case ExpressionPackage.Literals.LITERAL_REFERENCE_ELEMENT__VALUE:
			{
				return scope_LiteralReferenceElement_value(
					context as LiteralReferenceElement, reference
				)
			}
			
			case ExpressionPackage.Literals.VALUE_ELEMENT_SPECIFICATION__ELEMENT:
			{
				return scope_ValueElementSpecification_element(
					context as ValueElementSpecification, reference)
			}
			
			case ExpressionPackage.Literals.CAST_EXPRESSION__DATATYPE:
			{
				return scope_CastExpression_datatype(
					context as CastExpression, reference)
			}
			
			case StatementPackage.Literals.ABSTRACT_COM_STATEMENT__PORT:
			{
				return scope_AbstractComStatement_port(
					context as AbstractComStatement, reference)
			}
			
			case StatementPackage.Literals.INVOKE_STATEMENT__INVOKABLE:
			{
				return scope_InvokeStatement_invokable(
					context as InvokeStatement, reference)
			}
			
			case DatatypePackage.Literals.DATA_TYPE_REFERENCE__TYPEREF:
			{
				return scope_DataTypeReference_typeref(
					context as DataTypeReference, reference)
			}
			
			case InfrastructurePackage.Literals.INSTANCE_MACHINE__MODEL:
			{
				return scope_InstanceMachine_model(
					context as InstanceMachine, reference)
			}
			
			case InfrastructurePackage.Literals.ROUTE__SIGNALS:
			{
				return scope_Route_signals(context as Route, reference)
			}
			
			case InfrastructurePackage.Literals.SLOT_PROPERTY__XLIA_PROPERTY:
			{
				return scope_SlotProperty_xliaProperty(
					context as SlotProperty, reference
				)
			}
			
			case DatatypePackage.Literals.PRIMITIVE_INSTANCE_TYPE__MODEL:
			{
				return scope_PrimitiveInstanceType_model(
					context as PrimitiveInstanceType, reference)
			}
			
//			case InfrastructurePackage.Literals.COM_PROTOCOL__BUFFER:
//			{
////			createScopeForComProtocolBuffer(
//					context as ComProtocol, reference)
//
//				return super.getScope(context, reference)
//			}
			
			default: {
				print( "TO DO case: " )
				println( requiredScoping )
		
				val scope = super.getScope(context, reference)
				
				print( "\tdefault::IScope: ")
				println( scope )
				
				return scope
			}
		}
	}
	
	////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////
	
	def scope_Transition_target(Transition context, EReference reference)
	{
		val parentScope = IScope::NULLSCOPE

		val region = EcoreUtil2.getContainerOfType(context, typeof(Region))

		var thisMachine = EcoreUtil2.getContainerOfType(region, typeof(Machine))

		Scopes::scopeFor(thisMachine.property.selectTypedElement(
				PrimitiveInstanceKind.VERTEX),
			Scopes::scopeFor(region.vertex, parentScope))
	}
		
	////////////////////////////////////////////////////////////////////////////
	// scope_AbstractComStatement_port
	////////////////////////////////////////////////////////////////////////////
	
	def scope_AbstractComStatement_port(
		AbstractComStatement context, EReference reference)
	{
		switch( context ) {
			OutputComStatement: {
				return context.scopeForComElement(
					PrimitiveInstanceKind.COM_POINT, ChannelDirection.OUTPUT)
			}
			InputComStatement: {
				return context.scopeForComElement(
					PrimitiveInstanceKind.COM_POINT, ChannelDirection.INPUT)
			}
			default: {
				val parentScope = IScope::NULLSCOPE

				return parentScope
			}
		}
	}
	
	
	def Iterable< Port > selectPort(
		Iterable< Port > elements, ChannelDirection dir)
	{

		if( dir == ChannelDirection.INOUT ) {
			elements
		}
		else {
			elements.filter[ (direction == dir) ||
				(direction == ChannelDirection.INOUT) ]
		}
	}

	def Iterable< Signal > selectSignal(
		Iterable< Signal > elements, ChannelDirection dir)
	{
		if( dir == ChannelDirection.INOUT ) {
			elements
		}
		else {
			elements.filter[ (direction == dir) ||
				(direction == ChannelDirection.INOUT) ]
		}
	}
	
	def scopeForComElement(AbstractElement context,
		PrimitiveInstanceKind expected, ChannelDirection direction)
	{
		for( var EObject it = context ; it !== null; it = it.eContainer())
		{
			switch( it ) {
				Routine: return it.scopeForHierarchicLeftElement(
					ValueElementSpecificationScheme.COM_POINT)

				Machine: {
					var thisMachine = EcoreUtil2.getContainerOfType(context, typeof(Machine))
			
					while( (thisMachine !== null)
						 && thisMachine.port.empty && thisMachine.signal.empty)
					{
						thisMachine = EcoreUtil2.getContainerOfType(
							thisMachine.eContainer, typeof(Machine) )
					}
			
					var parentScope = IScope::NULLSCOPE
					if( thisMachine !== null ) {
						parentScope =
							Scopes::scopeFor(thisMachine.property.selectTypedElement(expected),
								Scopes::scopeFor(thisMachine.signal.selectSignal(direction),
									Scopes::scopeFor(thisMachine.port.selectPort(direction),
										parentScope)))
					}
			
					return parentScope
				}
			}
		}

		IScope::NULLSCOPE
	}
	
	
	////////////////////////////////////////////////////////////////////////////
	// scope_LiteralReferenceElement_value
	////////////////////////////////////////////////////////////////////////////
	
	def scope_LiteralReferenceElement_value(
		LiteralReferenceElement context, EReference reference)
	{
		var parentScope = IScope::NULLSCOPE

		var container = context.eContainer

		for( ; container !== null ; container = container.eContainer )
		{
			switch( container )
			{
				ValueElementSpecification : 
						return container.scopeForThis(context)

				LeftHandSideExpression : return container.scopeForThis(context)

				TupleExpression : return container.scopeForThis(context)
				
				QuantifiedLogicalExpression : return container.scopeForThis(context)

				InvokeExpression : return container.scopeForThis(context)

				InputComStatement : return container.scopeForThis(context)
				OutputComStatement: return container.scopeForThis(context)

				ActivityStatement : return container.scopeForThis(context)

				Transition : return container.scopeForThis(context)

				ComPoint: return container.scopeForThis(context)

				Routine : return container.scopeForAnyElement(context)

				Machine : return container.scopeForAnyElement(context)

				PrimitiveInstanceType :
						return container.scopeForAnyModelElement(context)

				DataTypeReference : return container.scopeForThis(context)
			}
		}

//		switch( container ) {
//			// Expression
//			Expression: {
//			}
//			
//			// Statement
//			Statement: {
//			}
//			
//			// Other
//			Variable: {
//			}
//			
//			default: {
//				
//			}
//		}
		
		parentScope
	}
		
	
	////////////////////////////////////////////////////////////////////////////
	// scope_ValueElementSpecification_element
	////////////////////////////////////////////////////////////////////////////
	
	def scope_ValueElementSpecification_element(
		ValueElementSpecification context, EReference reference)
	{
		var parentScope = IScope::NULLSCOPE

		val type = context.target.typeFor ?:
			EcoreUtil2.getContainerOfType(context, typeof(Machine))

		if( (type === null) || type.isPrimitive )
		{
			return parentScope
		}

		val container = context.eContainer
		
		if( context.expected == ValueElementSpecificationScheme.ANY )
		{
			switch( container )
			{
				ComPoint:
					context.expected = ValueElementSpecificationScheme.COM_POINT

				ComProtocol:
					context.expected = ValueElementSpecificationScheme.BUFFER

				InputComStatement :
					context.expected = ValueElementSpecificationScheme.COM_POINT

				OutputComStatement:
					context.expected = ValueElementSpecificationScheme.COM_POINT
					
//				IntervalType:
//					context.expected = ValueElementSpecificationScheme.
			}
		}

//		switch( container )
//		{
//			Expression: {
//			}
//			
//			OutputComStatement: {
//			}
//			InputComStatement: {
//			}
//			
//			ComPoint: {
//			}
//			ComProtocol: {
//			}
//			
//			IntervalType: {
//			}
//			
//			default: {
//			}
//		}
		

		if( type instanceof Machine )
		{
			val typeMachine = type as Machine

			if( context.expected == ValueElementSpecificationScheme.BUFFER )
			{
				typeMachine.scopeForHierarchicLeftElement(context.expected)
			}
			else if( context.expected == ValueElementSpecificationScheme.COM_POINT )
			{
				switch( container )
				{
					ComPoint: typeMachine.scopeForSelfMachineComPoint(
						(context.eContainer as ComPoint).direction)

					InputComStatement : Scopes::scopeFor(
						typeMachine.selectLeftElements(context), parentScope)

					OutputComStatement: Scopes::scopeFor(
						typeMachine.selectRigthElements(context), parentScope)
				}
			}
			else if( (context.target instanceof LiteralReferenceExpression) &&
					(! (context.target instanceof LiteralReferenceElement)) )
			{
				Scopes::scopeFor(
					typeMachine.selectLeftElements(context), parentScope)
			}
			else if( typeMachine.isAncestorOf(context) )
			{
				for( machine : typeMachine.reverseMachineHierarchy )
				{
					parentScope = Scopes::scopeFor(
						machine.selectLeftElements(context), parentScope)
				}

				Scopes::scopeFor(
					typeMachine.selectLeftElements(context), parentScope)
			}
			else {
				for( machine : typeMachine.reverseMachineHierarchy )
				{
					parentScope = Scopes::scopeFor(
						machine.selectPublicLeftElements(context), parentScope)
				}

				Scopes::scopeFor(
					typeMachine.selectPublicLeftElements(context), parentScope)
			}
		}
		else if( type instanceof DataType )
		{
			Scopes::scopeFor(type.selectElements(context), parentScope)
		}
		else {
			parentScope
		}
	}

	////////////////////////////////////////////////////////////////////////////
	// scope_CastExpression_datatype
	////////////////////////////////////////////////////////////////////////////
	
	def scope_CastExpression_datatype(CastExpression context, EReference reference)
	{
		return super.getScope(context, reference)
	}

	////////////////////////////////////////////////////////////////////////////
	// scope_InvokeStatement_invokable
	////////////////////////////////////////////////////////////////////////////
	
	def scope_InvokeStatement_invokable(
		InvokeStatement context, EReference reference)
	{
		var parentScope = IScope::NULLSCOPE
		
		var thisMachine = EcoreUtil2.getContainerOfType(context, typeof(Machine))
		
		if( context.isExecRoutine )
		{
			while( (thisMachine !== null) && thisMachine.routine.empty )
			{
				thisMachine = EcoreUtil2.getContainerOfType(
					thisMachine.eContainer, typeof(Machine) )
			}
	
			if( thisMachine !== null )
			{
				val machineHasRoutine = [ Machine m | ! m.routine.empty ]
				
				for( machine : thisMachine.reverseMachineHierarchy(machineHasRoutine) )
				{
					if( ! machine.routine.empty )
					{
						parentScope = Scopes::scopeFor(
							machine.routine.selectNonPrivate, parentScope)
					}
				}
						
				parentScope = Scopes::scopeFor(thisMachine.routine, parentScope)
			}
		}
		else if( context.callProcedure )
		{
			while( (thisMachine !== null) && thisMachine.procedure.empty )
			{
				thisMachine = EcoreUtil2.getContainerOfType(
					thisMachine.eContainer, typeof(Machine) )
			}
	
			if( thisMachine !== null )
			{
				val machineHasProcedure = [ Machine m | ! m.procedure.empty ]
				
				for( machine : thisMachine.reverseMachineHierarchy(machineHasProcedure) )
				{
					if( ! machine.procedure.empty )
					{
						parentScope = Scopes::scopeFor(
							machine.procedure.selectNonPrivate, parentScope)
					}
				}
						
				parentScope =Scopes::scopeFor(thisMachine.procedure, parentScope)
			}
		}
		
		parentScope
	}

	////////////////////////////////////////////////////////////////////////////
	// scope_DataTypeReference_typeref
	////////////////////////////////////////////////////////////////////////////
	
	def scope_DataTypeReference_typeref(
		DataTypeReference context, EReference reference)
	{
//		var parentScope = IScope::NULLSCOPE
//
//		val container = context.eContainer
//
//		switch( container )
//		{
//			LiteralReferenceElement: {
//				context.scopeForThis(container)
//			}
//			
//			Variable: {
//				
//			}
//			
//			Parameter: {
//				
//			}
//			
//			CollectionType: {
//				
//			}
//			
//			LiteralCollectionExpression: {
//				
//			}
//			
//			default: {
//			}
//		}
//				
//		parentScope
		
		val thisMachine = EcoreUtil2.getContainerOfType(context, typeof(Machine))

		thisMachine.scopeForHierarchicRigthElement(
			ValueElementSpecificationScheme.TYPEDEF)
	}

	////////////////////////////////////////////////////////////////////////////
	// scope_InstanceMachine_model
	////////////////////////////////////////////////////////////////////////////
	
	def scope_InstanceMachine_model(InstanceMachine context, EReference reference)
	{
		context.scopeForHierarchicLeftElement(ValueElementSpecificationScheme.MODEL)
	}

	////////////////////////////////////////////////////////////////////////////
	// scope_Route_signals
	////////////////////////////////////////////////////////////////////////////
	
	def scope_Route_signals(Route context, EReference reference)
	{
		context.scopeForComElement(
			PrimitiveInstanceKind.SIGNAL, ChannelDirection.INPUT)
	}

	////////////////////////////////////////////////////////////////////////////
	// scope_SlotProperty_xliaProperty
	////////////////////////////////////////////////////////////////////////////
	
	def scope_SlotProperty_xliaProperty(
		SlotProperty context, EReference reference)
	{
		val instance = EcoreUtil2.getContainerOfType(
			context, typeof(InstanceMachine) )
		
		val model = instance.model

		var parentScope = Scopes::scopeFor(
				model.property.selectNonFinalNorConst,
					Scopes::scopeFor(model.buffer, IScope::NULLSCOPE))
			
		if( model instanceof Behavior )
		{
			parentScope = Scopes::scopeFor(model.parameter, parentScope)
		}
		
		parentScope
	}


	////////////////////////////////////////////////////////////////////////////
	// scope_PrimitiveInstanceType_model
	////////////////////////////////////////////////////////////////////////////
	
	def scope_PrimitiveInstanceType_model(
		PrimitiveInstanceType context, EReference reference)
	{
		context.scopeForAnyModelElement(context)
	}

	////////////////////////////////////////////////////////////////////////////
	// Tools / Utils
	////////////////////////////////////////////////////////////////////////////

	def Iterable< ? extends NamedElement> selectPublic(
		Iterable< ? extends NamedElement> elements) {

		elements.filter[ visibility == VisibilityKind.PUBLIC ]
	}

	def Iterable< ? extends NamedElement> selectNonPublic(
		Iterable< ? extends NamedElement> elements) {

		elements.filter[ visibility != VisibilityKind.PUBLIC ]
	}

	def Iterable< ? extends NamedElement> selectNonPrivate(
		Iterable< ? extends NamedElement> elements) {

		elements.filter[ visibility != VisibilityKind.PRIVATE ]
	}


	def Iterable< ? extends PropertyDefinition > selectNonFinalNorConst(
		Iterable< ? extends PropertyDefinition > elements) {
		val nonFinalNorConstPredicate = [ PropertyDefinition P |
			(P.modifier.final == false) &&
			((! (P instanceof Variable)) || ((P as Variable).const == false)) ]

		elements.filter( nonFinalNorConstPredicate )
	}

	def isAncestorOf(EObject container, EObject element)
	{
		var ancestor = element.eContainer

		while( (ancestor !== null) && (ancestor != container) )
		{
			ancestor = ancestor.eContainer
		}

		( ancestor !== null )
	}


	// select Element by Type
	def Iterable< ? extends TypedElement > selectTypedElement(
		Iterable< ? extends TypedElement > elements,
		PrimitiveInstanceKind expected)
	{

		val typePredicate = [ TypedElement element |
			(element.type instanceof PrimitiveInstanceType)
			 && ((element.type as PrimitiveInstanceType).expected == expected) ]

		elements.filter( typePredicate )
	}

	// Machine Hierarcy -- Reverse View List
	def reverseMachineHierarchy(Machine machine)
	{
		val hierarchy = <Machine>newArrayList()

		var container = EcoreUtil2.getContainerOfType(
				machine.eContainer, typeof(Machine))

		while( container !== null ) {
			hierarchy.add( container )

			container = EcoreUtil2.getContainerOfType(
				container.eContainer, typeof(Machine))
		}

		hierarchy.reverseView
	}

	def reverseMachineHierarchy(
		Machine machine, Function1<Machine, Boolean> predicate)
	{
		val hierarchy = <Machine>newArrayList()

		var container = EcoreUtil2.getContainerOfType(
				machine.eContainer, typeof(Machine))

		while( (container !== null) && predicate.apply(container) )
		{
			hierarchy.add( container )

			container = EcoreUtil2.getContainerOfType(
				container.eContainer, typeof(Machine))
		}

		hierarchy.reverseView
	}


	////////////////////////////////////////////////////////////////////////////
	// select MACHINE Elements
	////////////////////////////////////////////////////////////////////////////

	def selectLeftElements(Machine machine, ValueElementSpecification ves)
	{
		if( ves.kind == ValueElementSpecificationKind.PARAMETER )
		{
			machine.routine + machine.procedure
		}
		else
		{
			selectLeftElements(machine, ves.expected)
		}
	}

	def selectPublicLeftElements(Machine machine, ValueElementSpecification ves)
	{
		if( ves.kind == ValueElementSpecificationKind.PARAMETER )
		{
			machine.routine + machine.procedure
		}
		else
		{
			selectPublicLeftElements(machine, ves.expected)
		}
	}


	def selectRigthElements(Machine machine, ValueElementSpecification ves)
	{
		if( ves.kind == ValueElementSpecificationKind.PARAMETER )
		{
			machine.routine + machine.procedure
		}
		else
		{
			selectLeftElements(machine, ves.expected)
		}
	}

	def Iterable< ? extends NamedElement> selectElements(
		DataType type, ValueElementSpecification ves)
	{
		if( ves.kind != ValueElementSpecificationKind.PARAMETER )
		{
			switch( type )
			{
				EnumerationType: type.literal

				DataStructuredType: type.property

				DataSupportedType : type.support.selectElements(ves)

				DataTypeReference :
				{
					if( type.typeref !== null )
					{
						type.typeref.selectElements(ves)
					}
					else if( type.support !== null )
					{
						type.support.selectElements(ves)
					}
				}
			}
		}
	}


	////////////////////////////////////////////////////////////////////////////
	// select MACHINE Elements
	////////////////////////////////////////////////////////////////////////////

	def selectLeftElements(
		Machine block, ValueElementSpecificationScheme expected)
	{
		switch( expected )
		{
			case ANY : block.property +
				block.channel + block.buffer + block.port + block.signal +
				block.machine + block.instance + block.behavior +
				block.routine + block.procedure + block.typedef

			case VARIABLE : block.property

			case TYPEDEF  : block.typedef
			
			case PORT     : block.port
			case SIGNAL   : block.signal
			case MESSAGE  : block.port + block.signal
			case COM_POINT: block.port + block.signal

			case INSTANCE : block.instance
			case MACHINE  : block.machine + block.instance + block.behavior
			case MODEL    : block.machine + block.behavior

			case BUFFER   : block.buffer

			case CHANNEL  : block.channel

			case VERTEX   : {
			}

			case PROCEDURE: block.procedure
			case ROUTINE  : block.routine
		}
	}


	def selectPublicLeftElements(
		Machine block, ValueElementSpecificationScheme expected)
	{
		selectLeftElements(block, expected).selectPublic
	}

	def selectNonPublicLeftElements(
		Machine block, ValueElementSpecificationScheme expected)
	{
		selectLeftElements(block, expected).selectNonPublic
	}


	def selectRigthElements(
		Machine block, ValueElementSpecificationScheme expected)
	{
		switch( expected )
		{
			case ANY : block.property + block.enumLiterals +
				block.channel + block.buffer + block.port + block.signal +
				block.machine + block.instance + block.behavior +
				block.routine + block.procedure + block.typedef

			case VARIABLE : block.property

			case TYPEDEF  : block.typedef

			case PORT     : block.port +
							block.property.selectProperty(expected)

			case SIGNAL   : block.signal +
							block.property.selectProperty(expected)

			case MESSAGE  : block.port + block.signal +
							block.property.selectProperty(expected)

			case COM_POINT: block.port + block.signal +
							block.property.selectProperty(expected)

			case INSTANCE : block.instance +
							block.property.selectProperty(expected)

			case MACHINE  : block.machine + block.instance + block.behavior +
							block.property.selectProperty(expected)

			case MODEL    : block.machine + block.behavior

			case BUFFER   : block.buffer +
							block.property.selectProperty(expected)

			case CHANNEL  : block.channel +
							block.property.selectProperty(expected)

			case VERTEX   : block.property.selectProperty(expected)

			case PROCEDURE: block.procedure
			case ROUTINE  : block.routine
		}
	}

	def selectPublicRigthElements(
		Machine block, ValueElementSpecificationScheme expected)
	{
		selectRigthElements(block, expected).selectPublic
	}

	def selectNonPublicRigthElements(
		Machine block, ValueElementSpecificationScheme expected)
	{
		selectRigthElements(block, expected).selectNonPublic
	}

	def enumLiterals(Machine machine)
	{
		val literals = <EnumerationLiteral>newArrayList()

		for( it : machine.typedef )
		{
			val type = it.typeSpecifier
			switch( type )
			{
				EnumerationType: literals += type.literal
			}
		}

		literals
	}

	def selectStructuredElements(
		Machine block, ValueElementSpecificationScheme expected)
	{
		val roots = <NamedElement>newArrayList()

		switch( expected )
		{
			case ANY : {
				for( it : block.property )
				{
					if( it.type.isStructured )
					{
						roots += it
					}
				}
				for( it : block.typedef )
				{
					if( it instanceof EnumerationType )
					{
						roots += it
					}
				}
				roots += block.machine + block.instance + block.behavior
			}

			case VARIABLE : {
				for( it : block.property )
				{
					if( it.type.isStructured )
					{
						roots += it
					}
				}
			}

			case INSTANCE : roots += block.instance +
							block.property.selectProperty(expected)

			case MACHINE  : roots += block.machine + block.instance +
							block.behavior +
							block.property.selectProperty(expected)

			case MODEL    : roots += block.machine + block.behavior

			default: {
			}
		}

		roots
	}

	////////////////////////////////////////////////////////////////////////////
	// select BEHAVIOR Elements
	////////////////////////////////////////////////////////////////////////////

	def selectLeftElements(
		Behavior block, ValueElementSpecificationScheme expected)
	{
//		val elements = (block as Machine).selectLeftElements(expected)
//
//		switch( expected ) {
//			case ANY : elements + block.parameter
//
//			case VARIABLE : elements + block.parameter
//
//			default: elements
//		}

		switch( expected )
		{
			case ANY : block.property + block.parameter +
				block.channel + block.buffer + block.port + block.signal +
				block.machine + block.instance + block.behavior +
				block.routine + block.procedure + block.typedef

			case VARIABLE : block.property + block.parameter

			case TYPEDEF : block.typedef

			case PORT     : block.port
			case SIGNAL   : block.signal
			case MESSAGE  : block.port + block.signal
			case COM_POINT: block.port + block.signal

			case INSTANCE : block.instance
			case MACHINE  : block.machine + block.instance + block.behavior
			case MODEL    : block.machine + block.behavior

			case BUFFER   : block.buffer

			case CHANNEL  : block.channel

			case VERTEX   : {

			}

			case PROCEDURE: block.procedure
			case ROUTINE  : block.routine
		}
	}


	def selectRigthElements(
		Behavior block, ValueElementSpecificationScheme expected)
	{
		switch( expected )
		{
			case ANY : block.property + block.parameter + block.enumLiterals +
				block.channel + block.buffer + block.port + block.signal +
				block.machine + block.instance + block.behavior +
				block.routine + block.procedure + block.typedef

			case VARIABLE : block.property + block.parameter

			case TYPEDEF  : block.typedef

			case PORT     : block.port +
							block.property.selectProperty(expected) +
							block.parameter.selectProperty(expected)

			case SIGNAL   : block.signal +
							block.property.selectProperty(expected) +
							block.parameter.selectProperty(expected)

			case MESSAGE  : block.port + block.signal +
							block.property.selectProperty(expected) +
							block.parameter.selectProperty(expected)

			case COM_POINT: block.port + block.signal +
							block.property.selectProperty(expected) +
							block.parameter.selectProperty(expected)

			case INSTANCE : block.instance +
							block.property.selectProperty(expected) +
							block.parameter.selectProperty(expected)

			case MACHINE  : block.machine  + block.instance + block.behavior +
							block.property.selectProperty(expected) +
							block.parameter.selectProperty(expected)

			case MODEL    : block.machine + block.behavior

			case BUFFER   : block.buffer +
							block.property.selectProperty(expected) +
							block.parameter.selectProperty(expected)

			case CHANNEL  : block.channel +
							block.property.selectProperty(expected) +
							block.parameter.selectProperty(expected)

			case VERTEX   : block.property.selectProperty(expected) +
							block.parameter.selectProperty(expected)

			case PROCEDURE: block.procedure
			case ROUTINE  : block.routine
		}
	}


	def selectProperty(
		Iterable< ? extends PropertyDefinition> properties,
		ValueElementSpecificationScheme expected)
	{
		switch( expected ) {
			case ANY : properties

			case VARIABLE: properties

			default: {
				val selection = <PropertyDefinition>newArrayList()

				for( it : properties ) {
					val type = it.type.typeSpecifier
					switch( type ) {
						PrimitiveInstanceType : {
							switch( expected ) {
								case PORT: {
									if( type.expected ==
										PrimitiveInstanceKind.PORT)
									{ selection += it }
								}
								case SIGNAL: {
									if( type.expected ==
										PrimitiveInstanceKind.SIGNAL)
									{ selection += it }
								}
								case MESSAGE: {
									if( type.expected ==
										PrimitiveInstanceKind.MESSAGE)
									{ selection += it }
								}
								case COM_POINT: {
									if( (type.expected ==
											PrimitiveInstanceKind.PORT)
									|| (type.expected ==
											PrimitiveInstanceKind.SIGNAL)
									|| (type.expected ==
											PrimitiveInstanceKind.COM_POINT) )
									{ selection += it }
								}
								case INSTANCE: {
									if( type.expected ==
										PrimitiveInstanceKind.MACHINE)
									{ selection += it }
								}
								case MACHINE: {
									if( type.expected ==
										PrimitiveInstanceKind.MACHINE)
									{ selection += it }
								}

								case BUFFER: {
									if( type.expected ==
										PrimitiveInstanceKind.BUFFER)
									{ selection += it }
								}
								case CHANNEL: {
									if( type.expected ==
										PrimitiveInstanceKind.CHANNEL)
									{ selection += it }
								}

								case VERTEX: {
									if( type.expected ==
										PrimitiveInstanceKind.VERTEX)
									{ selection += it }
								}

								default: {
								}
							}
						}
					}
				}

				selection
			}
		}
	}

	////////////////////////////////////////////////////////////////////////////
	// select ROUTINE Elements
	////////////////////////////////////////////////////////////////////////////

	def selectLeftElements(
		Routine routine, ValueElementSpecificationScheme expected)
	{
		switch( expected )
		{
			case ANY     :
			{
				var lvalue = routine.parameter
				if( routine.domain !== null )
				{
					lvalue += routine.domain.parameter
				}
				if( routine.codomain !== null )
				{
					lvalue += routine.codomain.parameter
				}

				lvalue
			}

			case VARIABLE: {
				var lvalue = routine.parameter
				if( routine.domain !== null )
				{
					lvalue += routine.domain.parameter
				}
				if( routine.codomain !== null )
				{
					lvalue += routine.codomain.parameter
				}

				lvalue
			}

			case MACHINE: {
				val thisMachine =
						EcoreUtil2.getContainerOfType(routine, typeof(Machine))
				
				var lvalue = thisMachine.machine + thisMachine.instance

				lvalue
			}

			default: {
				var lvalue = routine.parameter.selectProperty(expected)
				if( routine.domain !== null )
				{
					lvalue = lvalue +
						routine.domain.parameter.selectProperty(expected)
				}
				if( routine.codomain !== null )
				{
					lvalue = lvalue +
						routine.codomain.parameter.selectProperty(expected)
				}

				lvalue
			}
		}
	}

	def selectRigthElements(
		Routine routine, ValueElementSpecificationScheme expected)
	{
		var properties = routine.parameter.selectProperty(expected)

		if( routine.domain !== null )
		{
			properties = properties +
				routine.domain.parameter.selectProperty(expected)
		}
		if( routine.codomain !== null )
		{
			properties = properties +
				routine.codomain.parameter.selectProperty(expected)
		}

		properties
	}

	def selectStructuredElements(
		Routine routine, ValueElementSpecificationScheme expected)
	{
		val roots = <PropertyDefinition>newArrayList()

		for( it : routine.parameter )
		{
			if( it.type.isStructured )
			{
				roots += it
			}
		}

		if( routine.domain !== null )
		{
			for( it : routine.domain.parameter )
			{
				if( it.type.isStructured ) {
					roots += it
				}
			}
		}
		if( routine.codomain !== null )
		{
			for( it : routine.codomain.parameter )
			{
				if( it.type.isStructured )
				{
					roots += it
				}
			}
		}

		roots
	}

	////////////////////////////////////////////////////////////////////////////
	// ComPoint --> NamedElement
	////////////////////////////////////////////////////////////////////////////

	def scopeForSelfMachineInstance(Machine thisMachine, IScope outer)
	{
		val selfMachine =
			if( (thisMachine.name === null) &&
				(thisMachine.eContainer instanceof Machine) )
			{
				thisMachine.eContainer as Machine
			}
			else
			{
				thisMachine
			}

		Scopes::scopeFor(selfMachine.instance,
			Scopes::scopeFor(selfMachine.machine,
				Scopes::scopeFor(selfMachine.behavior, outer)))
	}


	def scopeForSelfMachineComPoint(
		Machine thisMachine, ChannelDirection direction)
	{
		var parentScope = Scopes::scopeFor(
			thisMachine.port.selectPort(direction),
			Scopes::scopeFor(
				thisMachine.signal.selectSignal(direction),
				IScope::NULLSCOPE))

		if( (thisMachine.name === null) &&
			(thisMachine.eContainer instanceof Machine) )
		{
			val selfMachine = thisMachine.eContainer as Machine

			if( selfMachine.main == thisMachine )
			{
				parentScope = Scopes::scopeFor(
					selfMachine.port.selectPort(direction),
					Scopes::scopeFor(
						selfMachine.signal.selectSignal(direction), parentScope))
			}
		}

		parentScope
	}

	////////////////////////////////////////////////////////////////////////////
	// scopeForThis
	////////////////////////////////////////////////////////////////////////////

	def scopeForThis(
		LeftHandSideExpression container, LiteralReferenceElement context)
	{
		for( var EObject it = container ; it !== null; it = it.eContainer())
		{
			switch( it ) {
				Routine: return it.scopeForHierarchicLeftElement(
					ValueElementSpecificationScheme.VARIABLE)

				Machine: return it.scopeForHierarchicLeftElement(
					ValueElementSpecificationScheme.VARIABLE)
			}
		}

		IScope::NULLSCOPE
	}


	def scopeForThis(InputComStatement container, LiteralReferenceElement context)
	{
		switch( context ) {
			case container.port:
				container.scopeForHierarchicRigthElement(
					ValueElementSpecificationScheme.COM_POINT)

			case container.route :
				container.scopeForHierarchicRigthElement(
					ValueElementSpecificationScheme.CHANNEL)

			case container.target:
				container.scopeForHierarchicRigthElement(
					ValueElementSpecificationScheme.MACHINE)

			default:
				container.scopeForHierarchicLeftElement(
					ValueElementSpecificationScheme.VARIABLE)
		}
	}


	def scopeForThis(OutputComStatement container, LiteralReferenceElement context)
	{
		switch( context ) {
			case container.port:
				container.scopeForHierarchicRigthElement(
					ValueElementSpecificationScheme.COM_POINT)

			case container.route :
				container.scopeForHierarchicRigthElement(
					ValueElementSpecificationScheme.CHANNEL)

			case container.target:
				container.scopeForHierarchicRigthElement(
					ValueElementSpecificationScheme.MACHINE)

			default:
				container.scopeForHierarchicRigthElement(
					ValueElementSpecificationScheme.ANY)
		}
	}


	def scopeForThis(ActivityStatement container, LiteralReferenceElement context)
	{
		if( context == container.machine )
		{
			container.scopeForHierarchicLeftElement(
				ValueElementSpecificationScheme.MACHINE)
		}
		else
		{
			container.scopeForHierarchicLeftElement(
				ValueElementSpecificationScheme.ANY)
		}
	}


	def scopeForThis(Transition container, LiteralReferenceElement context)
	{
		val thisRegion = EcoreUtil2.getContainerOfType(container, typeof(Region))

		if( context == container.target ) {
			thisRegion.scopeForRigthElement(
				ValueElementSpecificationScheme.VERTEX)
		}
		else {
			val thisMachine =
				EcoreUtil2.getContainerOfType(thisRegion, typeof(Machine))

			thisMachine.scopeForHierarchicRigthElement(
				ValueElementSpecificationScheme.ANY)
		}
	}

	def scopeForThis(ComPoint container, LiteralReferenceElement context)
	{
		val thisMachine = EcoreUtil2.getContainerOfType(container, typeof(Machine))

		thisMachine.scopeForHierarchicLeftElement(
			ValueElementSpecificationScheme.COM_POINT)
	}


	def scopeForThis(DataTypeReference thisType, LiteralReferenceElement context)
	{
		val thisMachine = EcoreUtil2.getContainerOfType(thisType, typeof(Machine))

		if( thisType.multiplicity == context ) {
			thisMachine.scopeForHierarchicRigthElement(
				ValueElementSpecificationScheme.TYPEDEF)
		}
		else {
			thisMachine.scopeForHierarchicRigthElement(
				ValueElementSpecificationScheme.TYPEDEF)
		}
	}


	def scopeForThis(
		ValueElementSpecification thisElement, LiteralReferenceElement context)
	{
		for( var EObject it = context ; it !== null; it = it.eContainer()) {
			switch( it ) {
				Routine: return it.scopeForHierarchicStructuredElement(
					ValueElementSpecificationScheme.ANY)

				Machine: return it.scopeForHierarchicStructuredElement(
					ValueElementSpecificationScheme.ANY)
			}
		}

		IScope::NULLSCOPE
	}

	def scopeForThis(TupleExpression thisTuple, LiteralReferenceElement context)
	{
		thisTuple.scopeForHierarchicRigthElement(
			ValueElementSpecificationScheme.ANY)
	}

	def scopeForThis(
		QuantifiedLogicalExpression thisQExpression,
		LiteralReferenceElement context)
	{
		Scopes::scopeFor(thisQExpression.variable,
			thisQExpression.scopeForHierarchicRigthElement(
				ValueElementSpecificationScheme.VARIABLE) )
	}


	def scopeForThis(
		InvokeExpression thisInvoke, LiteralReferenceElement context)
	{
		if( (context.eContainer == thisInvoke) &&
			(context.kind == ValueElementSpecificationKind.PARAMETER) )
		{
				if( thisInvoke.callProcedure )
				{
					thisInvoke.scopeForHierarchicRigthElement(
						ValueElementSpecificationScheme.PROCEDURE)
				}
				else
				{
					thisInvoke.scopeForHierarchicRigthElement(
						ValueElementSpecificationScheme.ROUTINE)
				}
		}
		else {
			thisInvoke.scopeForHierarchicRigthElement(
				ValueElementSpecificationScheme.ANY)
		}
	}

//	def scopeForThis(Expression container, LiteralReferenceElement context) {
//		container.scopeForHierarchicRigthElement(ValueElementSpecificationScheme.ANY)
//	}


	////////////////////////////////////////////////////////////////////////////
	// scopeForAnyElement
	////////////////////////////////////////////////////////////////////////////

	def scopeForAnyElement(Routine thisRoutine, LiteralReferenceElement context)
	{
		thisRoutine.scopeForHierarchicRigthElement(
			ValueElementSpecificationScheme.ANY)
	}

	def scopeForAnyElement(Machine thisBlock, LiteralReferenceElement context)
	{
		thisBlock.scopeForHierarchicRigthElement(
			ValueElementSpecificationScheme.ANY)
	}


	def scopeForAnyModelElement(
		PrimitiveInstanceType thisType, AbstractElement context)
	{
		val thisMachine =
				EcoreUtil2.getContainerOfType(thisType, typeof(Machine))
				
		if( context.eContainer instanceof LiteralNullExpression )
		{
			thisMachine.scopeForHierarchicLeftElement(
				ValueElementSpecificationScheme.MODEL)
		}
		else switch( thisType.expected )
		{
			case PORT: {
				thisMachine.scopeForHierarchicLeftElement(
					ValueElementSpecificationScheme.PORT)
			}
			case SIGNAL: {
				thisMachine.scopeForHierarchicLeftElement(
					ValueElementSpecificationScheme.SIGNAL)
			}
			case MESSAGE: {
				thisMachine.scopeForHierarchicLeftElement(
					ValueElementSpecificationScheme.MESSAGE)
			}
			case COM_POINT: {
				thisMachine.scopeForHierarchicLeftElement(
					ValueElementSpecificationScheme.COM_POINT)
			}

			case MACHINE: {
				thisMachine.scopeForHierarchicLeftElement(
					ValueElementSpecificationScheme.MODEL)
			}

			case BUFFER: {
				thisMachine.scopeForHierarchicLeftElement(
					ValueElementSpecificationScheme.BUFFER)
			}
			case CHANNEL: {
				thisMachine.scopeForHierarchicLeftElement(
					ValueElementSpecificationScheme.CHANNEL)
			}

			case VERTEX   : {
				thisMachine.scopeForHierarchicLeftElement(
					ValueElementSpecificationScheme.VERTEX)
			}
			default: {
				thisMachine.scopeForHierarchicLeftElement(
					ValueElementSpecificationScheme.ANY)
			}
		}
	}


	////////////////////////////////////////////////////////////////////////////
	// scopeForAnyElement
	////////////////////////////////////////////////////////////////////////////

	def scopeForLeftElement(
		Region thisRegion, ValueElementSpecificationScheme expected)
	{
		if( expected == ValueElementSpecificationScheme.VERTEX )
		{
			Scopes::scopeFor(thisRegion.vertex, IScope::NULLSCOPE)
		}

		IScope::NULLSCOPE
	}

	def scopeForRigthElement(
		Region thisRegion, ValueElementSpecificationScheme expected) {
		if( expected == ValueElementSpecificationScheme.VERTEX ) {
			val thisStatemachine =
				EcoreUtil2.getContainerOfType(thisRegion, typeof(Statemachine))

			Scopes::scopeFor(thisRegion.vertex,
				Scopes::scopeFor(thisStatemachine.selectRigthElements(expected),
					IScope::NULLSCOPE))
		}

		IScope::NULLSCOPE
	}


	////////////////////////////////////////////////////////////////////////////
	// scopeForLeftElement / scopeForRigthElement
	////////////////////////////////////////////////////////////////////////////

	def scopeForLeftElement(
		Machine thisMachine, ValueElementSpecificationScheme expected)
	{
		var parentScope = IScope::NULLSCOPE

		if( (thisMachine.name === null) &&
			(thisMachine.eContainer instanceof Machine) )
		{
			val selfMachine = thisMachine.eContainer as Machine

			if( selfMachine.main == thisMachine )
			{
				parentScope = Scopes::scopeFor(
					selfMachine.selectLeftElements(expected), parentScope)
			}
		}

		Scopes::scopeFor(thisMachine.selectLeftElements(expected), parentScope)
	}

	def scopeForRigthElement(
		Machine thisMachine, ValueElementSpecificationScheme expected)
	{
		var parentScope = IScope::NULLSCOPE

		if( (thisMachine.name === null) &&
			(thisMachine.eContainer instanceof Machine) )
		{
			val selfMachine = thisMachine.eContainer as Machine

			if( selfMachine.main == thisMachine )
			{
				parentScope = Scopes::scopeFor(
					selfMachine.selectRigthElements(expected), parentScope)
			}
		}

		Scopes::scopeFor(thisMachine.selectRigthElements(expected), parentScope)
	}


	def scopeForHierarchicLeftElement(
		Machine thisMachine, ValueElementSpecificationScheme expected)
	{
		var parentScope = IScope::NULLSCOPE

		for( block : thisMachine.reverseMachineHierarchy )
		{
			switch( block )
			{
				Behavior: parentScope = Scopes::scopeFor(
					block.selectLeftElements(expected), parentScope)

				default: parentScope = Scopes::scopeFor(
					block.selectLeftElements(expected), parentScope)
			}
		}

		switch( thisMachine )
		{
			Behavior: Scopes::scopeFor(
				thisMachine.selectLeftElements(expected), parentScope)

			default: Scopes::scopeFor(
				thisMachine.selectLeftElements(expected), parentScope)
		}
	}

	////////////////////////////////////////////////////////////////////////////
	// scopeForHierarchicLeftElement / scopeForHierarchicRigthElement
	////////////////////////////////////////////////////////////////////////////

	def scopeForHierarchicRigthElement(
		Machine thisMachine, ValueElementSpecificationScheme expected)
	{
		var parentScope = IScope::NULLSCOPE

		for( block : thisMachine.reverseMachineHierarchy )
		{
			switch( block )
			{
				Behavior: parentScope = Scopes::scopeFor(
					block.selectRigthElements(expected), parentScope)

				default: parentScope = Scopes::scopeFor(
					block.selectRigthElements(expected), parentScope)
			}
		}

		switch( thisMachine )
		{
			Behavior: Scopes::scopeFor(
				thisMachine.selectRigthElements(expected), parentScope)

			default: Scopes::scopeFor(
				thisMachine.selectRigthElements(expected), parentScope)
		}
	}


	def scopeForHierarchicStructuredElement(
		Machine thisMachine, ValueElementSpecificationScheme expected)
	{
		var parentScope = IScope::NULLSCOPE

		for( machine : thisMachine.reverseMachineHierarchy )
		{
			switch( machine )
			{
				Behavior: parentScope = Scopes::scopeFor(
					machine.selectStructuredElements(expected), parentScope)

				default: parentScope = Scopes::scopeFor(
					machine.selectStructuredElements(expected), parentScope)
			}
		}

		switch( thisMachine )
		{
			Behavior: Scopes::scopeFor(
				thisMachine.selectStructuredElements(expected), parentScope)

			default: Scopes::scopeFor(
				thisMachine.selectStructuredElements(expected), parentScope)
		}
	}


	def scopeForHierarchicStructuredElement(
		Routine thisRoutine, ValueElementSpecificationScheme expected)
	{
		val thisMachine =
			EcoreUtil2.getContainerOfType(thisRoutine, typeof(Machine))

		Scopes::scopeFor(thisRoutine.selectStructuredElements(expected),
			scopeForHierarchicStructuredElement(thisMachine, expected))
	}


	def scopeForHierarchicLeftElement(
		Routine thisRoutine, ValueElementSpecificationScheme expected)
	{
		val thisMachine =
			EcoreUtil2.getContainerOfType(thisRoutine, typeof(Machine))

		Scopes::scopeFor(thisRoutine.selectLeftElements(expected),
			scopeForHierarchicLeftElement(thisMachine, expected))
	}

	def scopeForHierarchicRigthElement(
		Routine thisRoutine, ValueElementSpecificationScheme expected)
	{
		val thisMachine =
			EcoreUtil2.getContainerOfType(thisRoutine, typeof(Machine))

		Scopes::scopeFor(thisRoutine.selectRigthElements(expected),
			scopeForHierarchicRigthElement(thisMachine, expected))
	}



	def scopeForHierarchicLeftElement(
		EObject context, ValueElementSpecificationScheme expected)
	{
		for( var it = context ; it !== null; it = it.eContainer() )
		{
			switch( it )
			{
				Routine: return it.scopeForHierarchicLeftElement(expected)

				Machine: return it.scopeForHierarchicLeftElement(expected)
			}
		}

		IScope::NULLSCOPE
	}

	def scopeForHierarchicRigthElement(
		EObject context, ValueElementSpecificationScheme expected)
	{
		for( var it = context ; it !== null; it = it.eContainer() )
		{
			switch( it )
			{
				Routine: return it.scopeForHierarchicRigthElement(expected)

				Machine: return it.scopeForHierarchicRigthElement(expected)
			}
		}

		IScope::NULLSCOPE
	}



}
