/*******************************************************************************
 * Copyright (c) 2005 - 2006 Joel Cheuoua & others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Joel Cheuoua - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.codegen.jet.editor.presentation;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.emf.codegen.ecore.genmodel.provider.GenModelItemProviderAdapterFactory;
import org.eclipse.emf.codegen.jet.editor.JETEditorGenerator;
import org.eclipse.emf.codegen.jet.editor.JETURLClassLoader;
import org.eclipse.emf.codegen.jet.editor.JavaClassLoaderFactory;
import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.edit.provider.ComposedAdapterFactory;
import org.eclipse.emf.edit.provider.ReflectiveItemProviderAdapterFactory;
import org.eclipse.emf.edit.provider.resource.ResourceItemProviderAdapterFactory;
import org.eclipse.emf.edit.ui.provider.AdapterFactoryContentProvider;
import org.eclipse.emf.edit.ui.provider.AdapterFactoryLabelProvider;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.internal.ui.JavaPlugin;
import org.eclipse.jdt.internal.ui.javaeditor.JavaSourceViewer;
import org.eclipse.jdt.ui.PreferenceConstants;
import org.eclipse.jdt.ui.text.JavaSourceViewerConfiguration;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.text.Document;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.source.SourceViewer;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.dialogs.ContainerSelectionDialog;
import org.eclipse.ui.dialogs.ResourceSelectionDialog;
import org.eclipse.ui.forms.ManagedForm;
import org.eclipse.ui.forms.SectionPart;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.forms.widgets.Section;

/**
 * @author Joel
 */
public class JETJavaPreviewViewer extends Viewer {

  private SourceViewer sourceViewer;
  private Object fInput;
  private ScrolledForm sForm;
  
  private Text inputResource = null;
  private Button browseWorkspace = null;
 
  private Tree tree = null;
  private TreeViewer treeViewer = null;
  private ISelection selection;
  
  
  public JETJavaPreviewViewer(Composite parent) {
    Display display = parent.getDisplay();
    FormToolkit formToolkit = new FormToolkit(display);
    sForm = formToolkit.createScrolledForm(parent);
    ManagedForm overviewForm = new ManagedForm(formToolkit, sForm);
    Composite body = sForm.getBody();

    GridLayout layout = new GridLayout();
    layout.numColumns = 1;
    layout.marginWidth = 10;
    layout.horizontalSpacing = 10;
    layout.verticalSpacing = 10;
    body.setLayout(layout);

    createInputArgumentsSection(overviewForm, body, formToolkit);
    createJavaTextViewerSection(overviewForm, body, formToolkit);
    
    overviewForm.initialize();
    overviewForm.refresh();    
    
  }
  
  /**
   * @param mForm
   * @param body
   * @param formToolkit
   */
  private void createInputArgumentsSection(ManagedForm mForm, Composite body, FormToolkit formToolkit) {
    final Section section = formToolkit.createSection(body, Section.TITLE_BAR);
    section.setText("Input Argument");
    Composite client = formToolkit.createComposite(section);
    GridLayout layout = new GridLayout();
    layout.marginWidth = formToolkit.getBorderStyle() != SWT.NULL ? 0 : 2;
    layout.numColumns = 2;
    client.setLayout(layout);
    
    inputResource = formToolkit.createText(client, "<Empty>");
    inputResource.setEditable(false);
    GridData gd = new GridData();
    gd.horizontalSpan = 1;
    gd.grabExcessHorizontalSpace = true;
    gd.horizontalAlignment = GridData.FILL;
    gd.grabExcessVerticalSpace = false;
    inputResource.setLayoutData(gd);
    
    browseWorkspace = formToolkit.createButton(client, "Browse...", SWT.NONE);
    browseWorkspace.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        ResourceSelectionDialog resSelectionDialog = 
          new ResourceSelectionDialog(section.getShell(), ResourcesPlugin.getWorkspace().getRoot(), "Select a workspace resource");
        if (resSelectionDialog.open() == ContainerSelectionDialog.OK) {
          Object[] result = resSelectionDialog.getResult();
          if (result.length == 1) {
            String inputFile = ((IResource) result[0]).getFullPath().toString();
            inputResource.setText(inputFile);
            ResourceSet resourceSet = new ResourceSetImpl();
            URI uri = URI.createPlatformResourceURI(inputFile);
            Resource resource = resourceSet.getResource(uri, true);
            if (resource != null)
              treeViewer.setInput(resourceSet);
          }
        }
      }
    });
    
    tree = formToolkit.createTree(client, SWT.BORDER | SWT.SINGLE);
    gd = new GridData();
    gd.horizontalSpan = 2;
    gd.horizontalAlignment = GridData.FILL;
    gd.verticalAlignment = GridData.FILL;
    gd.grabExcessVerticalSpace = true;
    tree.setLayoutData(gd);
    treeViewer = new TreeViewer(tree);
    
    List factories = new ArrayList();
    factories.add(new ResourceItemProviderAdapterFactory());
    factories.add(new GenModelItemProviderAdapterFactory());
    factories.add(new ReflectiveItemProviderAdapterFactory());

    AdapterFactory adapterFactory = new ComposedAdapterFactory(factories);
    treeViewer.setContentProvider(new AdapterFactoryContentProvider(adapterFactory));
    treeViewer.setLabelProvider(new AdapterFactoryLabelProvider(adapterFactory));
    
    Button button = formToolkit.createButton(client,"Apply", SWT.PUSH | SWT.FLAT);    
    button.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        IFileEditorInput fileInput = (IFileEditorInput) fInput;
        IJavaProject javaProject = JavaCore.create(fileInput.getFile().getProject());
        JETEditorGenerator jetGenerator = new JETEditorGenerator();
        Object generatedObject = null;
        sourceViewer.setInput(null);
        TreeItem[] selection = tree.getSelection();
        if (selection.length == 1) {
          generatedObject = selection[0].getData();
        }                
        if (generatedObject != null) {
          ClassLoader objectClassLoader = generatedObject.getClass().getClassLoader();
          ClassLoader classLoader = JavaClassLoaderFactory.getClassLoader(javaProject, objectClassLoader);
          ClassLoader currentClassLoader = jetGenerator.getClass().getClassLoader();
          ((JETURLClassLoader)classLoader).addClassLoaderDelegate(currentClassLoader);          
          jetGenerator.setClassLoader(classLoader);
          
          String string = jetGenerator.generateJava(generatedObject, fileInput.getFile().getLocation().toString(),  new NullProgressMonitor());
          if (string != null) {
            IDocument document = new Document(string);
            JavaPlugin.getDefault().getJavaTextTools().setupJavaDocumentPartitioner(document);
            sourceViewer.setInput(document);
          } else {
            IDocument document = new Document("/** Cannot generate preview with the given input ... Consult the logfile for details.*/");
            JavaPlugin.getDefault().getJavaTextTools().setupJavaDocumentPartitioner(document);
            sourceViewer.setInput(document);            
          }
        } else {
          IDocument document = new Document("/** Cannot generate preview ... no input was provided.*/");
          JavaPlugin.getDefault().getJavaTextTools().setupJavaDocumentPartitioner(document);
          sourceViewer.setInput(document);
        }
      }
    });
    
    section.setClient(client);
    
    gd = new GridData();
    gd.grabExcessHorizontalSpace = true;
    gd.horizontalAlignment = GridData.FILL;
    gd.verticalAlignment = GridData.BEGINNING;
    section.setLayoutData(gd);
    SectionPart sectionPart = new SectionPart(section);
    mForm.addPart(sectionPart);
  }
  
  /**
   * @param mForm
   * @param body
   * @param formToolkit
   */
  private void createJavaTextViewerSection(ManagedForm mForm, Composite body, FormToolkit formToolkit) {
    Section section = formToolkit.createSection(body, Section.TITLE_BAR);
    section.setText("Preview"); //$NON-NLS-1$
    
    IPreferenceStore store= JavaPlugin.getDefault().getCombinedPreferenceStore();
    sourceViewer= new JavaSourceViewer(section, null, null, false, SWT.H_SCROLL + SWT.V_SCROLL, store);    
    sourceViewer.configure(new JavaSourceViewerConfiguration(JavaPlugin.getDefault().getJavaTextTools().getColorManager(), store, null, null));    
    sourceViewer.getTextWidget().setFont(JFaceResources.getFont(PreferenceConstants.EDITOR_TEXT_FONT));
    sourceViewer.setEditable(false);
    
    section.setClient(sourceViewer.getControl());
    
    GridData gd = new GridData();
    gd.horizontalAlignment = GridData.FILL;
    gd.verticalAlignment = GridData.FILL;
    gd.grabExcessHorizontalSpace = true;
    gd.grabExcessVerticalSpace = true;
    
    section.setLayoutData(gd);
    SectionPart sectionPart = new SectionPart(section);
    mForm.addPart(sectionPart);
  }
  
  public Control getControl() {
    return sForm;
  }
  
  public void setInput(Object input) {
    fInput= input;
  }
  
  public Object getInput() {
    return fInput;
  }
  
  public ISelection getSelection() {
    return selection;
  }
  
  public void setSelection(ISelection s, boolean reveal) {
    this.selection = s;
  }
  
  public void refresh() {
  }

}
