/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        if (typeof b !== "function" && b !== null)
            throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
var __values = (this && this.__values) || function(o) {
    var s = typeof Symbol === "function" && Symbol.iterator, m = s && o[s], i = 0;
    if (m) return m.call(o);
    if (o && typeof o.length === "number") return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
    throw new TypeError(s ? "Object is not iterable." : "Symbol.iterator is not defined.");
};
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
import { Color } from '../../../base/common/color';
import { Disposable } from '../../../base/common/lifecycle';
import * as strings from '../../../base/common/strings';
import { EDITOR_FONT_DEFAULTS, filterValidationDecorations } from '../config/editorOptions';
import { Position } from '../core/position';
import { Range } from '../core/range';
import { TokenizationRegistry } from '../modes';
import { tokenizeLineToHTML } from '../modes/textToHtmlTokenizer';
import { MinimapTokensColorTracker } from './minimapTokensColorTracker';
import * as viewEvents from '../view/viewEvents';
import { ViewLayout } from '../viewLayout/viewLayout';
import { IdentityLinesCollection, SplitLinesCollection } from './splitLinesCollection';
import { MinimapLinesRenderingData, ViewLineRenderingData } from './viewModel';
import { ViewModelDecorations } from './viewModelDecorations';
import { RunOnceScheduler } from '../../../base/common/async';
import * as platform from '../../../base/common/platform';
import { Cursor } from '../controller/cursor';
import { CursorConfiguration } from '../controller/cursorCommon';
import { ViewModelEventDispatcher, FocusChangedEvent, ScrollChangedEvent, ViewZonesChangedEvent, ReadOnlyEditAttemptEvent } from './viewModelEventDispatcher';
var USE_IDENTITY_LINES_COLLECTION = true;
var ViewModel = /** @class */ (function (_super) {
    __extends(ViewModel, _super);
    function ViewModel(editorId, configuration, model, domLineBreaksComputerFactory, monospaceLineBreaksComputerFactory, scheduleAtNextAnimationFrame) {
        var _this = _super.call(this) || this;
        _this._editorId = editorId;
        _this._configuration = configuration;
        _this.model = model;
        _this._eventDispatcher = new ViewModelEventDispatcher();
        _this.onEvent = _this._eventDispatcher.onEvent;
        _this.cursorConfig = new CursorConfiguration(_this.model.getLanguageIdentifier(), _this.model.getOptions(), _this._configuration);
        _this._tokenizeViewportSoon = _this._register(new RunOnceScheduler(function () { return _this.tokenizeViewport(); }, 50));
        _this._updateConfigurationViewLineCount = _this._register(new RunOnceScheduler(function () { return _this._updateConfigurationViewLineCountNow(); }, 0));
        _this._hasFocus = false;
        _this._viewportStartLine = -1;
        _this._viewportStartLineTrackedRange = null;
        _this._viewportStartLineDelta = 0;
        if (USE_IDENTITY_LINES_COLLECTION && _this.model.isTooLargeForTokenization()) {
            _this._lines = new IdentityLinesCollection(_this.model);
        }
        else {
            var options = _this._configuration.options;
            var fontInfo = options.get(38 /* fontInfo */);
            var wrappingStrategy = options.get(118 /* wrappingStrategy */);
            var wrappingInfo = options.get(125 /* wrappingInfo */);
            var wrappingIndent = options.get(117 /* wrappingIndent */);
            _this._lines = new SplitLinesCollection(_this.model, domLineBreaksComputerFactory, monospaceLineBreaksComputerFactory, fontInfo, _this.model.getOptions().tabSize, wrappingStrategy, wrappingInfo.wrappingColumn, wrappingIndent);
        }
        _this.coordinatesConverter = _this._lines.createCoordinatesConverter();
        _this._cursor = _this._register(new Cursor(model, _this, _this.coordinatesConverter, _this.cursorConfig));
        _this.viewLayout = _this._register(new ViewLayout(_this._configuration, _this.getLineCount(), scheduleAtNextAnimationFrame));
        _this._register(_this.viewLayout.onDidScroll(function (e) {
            if (e.scrollTopChanged) {
                _this._tokenizeViewportSoon.schedule();
            }
            _this._eventDispatcher.emitSingleViewEvent(new viewEvents.ViewScrollChangedEvent(e));
            _this._eventDispatcher.emitOutgoingEvent(new ScrollChangedEvent(e.oldScrollWidth, e.oldScrollLeft, e.oldScrollHeight, e.oldScrollTop, e.scrollWidth, e.scrollLeft, e.scrollHeight, e.scrollTop));
        }));
        _this._register(_this.viewLayout.onDidContentSizeChange(function (e) {
            _this._eventDispatcher.emitOutgoingEvent(e);
        }));
        _this._decorations = new ViewModelDecorations(_this._editorId, _this.model, _this._configuration, _this._lines, _this.coordinatesConverter);
        _this._registerModelEvents();
        _this._register(_this._configuration.onDidChangeFast(function (e) {
            try {
                var eventsCollector = _this._eventDispatcher.beginEmitViewEvents();
                _this._onConfigurationChanged(eventsCollector, e);
            }
            finally {
                _this._eventDispatcher.endEmitViewEvents();
            }
        }));
        _this._register(MinimapTokensColorTracker.getInstance().onDidChange(function () {
            _this._eventDispatcher.emitSingleViewEvent(new viewEvents.ViewTokensColorsChangedEvent());
        }));
        _this._updateConfigurationViewLineCountNow();
        return _this;
    }
    ViewModel.prototype.dispose = function () {
        // First remove listeners, as disposing the lines might end up sending
        // model decoration changed events ... and we no longer care about them ...
        _super.prototype.dispose.call(this);
        this._decorations.dispose();
        this._lines.dispose();
        this.invalidateMinimapColorCache();
        this._viewportStartLineTrackedRange = this.model._setTrackedRange(this._viewportStartLineTrackedRange, null, 1 /* NeverGrowsWhenTypingAtEdges */);
        this._eventDispatcher.dispose();
    };
    ViewModel.prototype.createLineBreaksComputer = function () {
        return this._lines.createLineBreaksComputer();
    };
    ViewModel.prototype.addViewEventHandler = function (eventHandler) {
        this._eventDispatcher.addViewEventHandler(eventHandler);
    };
    ViewModel.prototype.removeViewEventHandler = function (eventHandler) {
        this._eventDispatcher.removeViewEventHandler(eventHandler);
    };
    ViewModel.prototype._updateConfigurationViewLineCountNow = function () {
        this._configuration.setViewLineCount(this._lines.getViewLineCount());
    };
    ViewModel.prototype.tokenizeViewport = function () {
        var linesViewportData = this.viewLayout.getLinesViewportData();
        var startPosition = this.coordinatesConverter.convertViewPositionToModelPosition(new Position(linesViewportData.startLineNumber, 1));
        var endPosition = this.coordinatesConverter.convertViewPositionToModelPosition(new Position(linesViewportData.endLineNumber, 1));
        this.model.tokenizeViewport(startPosition.lineNumber, endPosition.lineNumber);
    };
    ViewModel.prototype.setHasFocus = function (hasFocus) {
        this._hasFocus = hasFocus;
        this._cursor.setHasFocus(hasFocus);
        this._eventDispatcher.emitSingleViewEvent(new viewEvents.ViewFocusChangedEvent(hasFocus));
        this._eventDispatcher.emitOutgoingEvent(new FocusChangedEvent(!hasFocus, hasFocus));
    };
    ViewModel.prototype.onCompositionStart = function () {
        this._eventDispatcher.emitSingleViewEvent(new viewEvents.ViewCompositionStartEvent());
    };
    ViewModel.prototype.onCompositionEnd = function () {
        this._eventDispatcher.emitSingleViewEvent(new viewEvents.ViewCompositionEndEvent());
    };
    ViewModel.prototype.onDidColorThemeChange = function () {
        this._eventDispatcher.emitSingleViewEvent(new viewEvents.ViewThemeChangedEvent());
    };
    ViewModel.prototype._onConfigurationChanged = function (eventsCollector, e) {
        // We might need to restore the current centered view range, so save it (if available)
        var previousViewportStartModelPosition = null;
        if (this._viewportStartLine !== -1) {
            var previousViewportStartViewPosition = new Position(this._viewportStartLine, this.getLineMinColumn(this._viewportStartLine));
            previousViewportStartModelPosition = this.coordinatesConverter.convertViewPositionToModelPosition(previousViewportStartViewPosition);
        }
        var restorePreviousViewportStart = false;
        var options = this._configuration.options;
        var fontInfo = options.get(38 /* fontInfo */);
        var wrappingStrategy = options.get(118 /* wrappingStrategy */);
        var wrappingInfo = options.get(125 /* wrappingInfo */);
        var wrappingIndent = options.get(117 /* wrappingIndent */);
        if (this._lines.setWrappingSettings(fontInfo, wrappingStrategy, wrappingInfo.wrappingColumn, wrappingIndent)) {
            eventsCollector.emitViewEvent(new viewEvents.ViewFlushedEvent());
            eventsCollector.emitViewEvent(new viewEvents.ViewLineMappingChangedEvent());
            eventsCollector.emitViewEvent(new viewEvents.ViewDecorationsChangedEvent(null));
            this._cursor.onLineMappingChanged(eventsCollector);
            this._decorations.onLineMappingChanged();
            this.viewLayout.onFlushed(this.getLineCount());
            if (this.viewLayout.getCurrentScrollTop() !== 0) {
                // Never change the scroll position from 0 to something else...
                restorePreviousViewportStart = true;
            }
            this._updateConfigurationViewLineCount.schedule();
        }
        if (e.hasChanged(75 /* readOnly */)) {
            // Must read again all decorations due to readOnly filtering
            this._decorations.reset();
            eventsCollector.emitViewEvent(new viewEvents.ViewDecorationsChangedEvent(null));
        }
        eventsCollector.emitViewEvent(new viewEvents.ViewConfigurationChangedEvent(e));
        this.viewLayout.onConfigurationChanged(e);
        if (restorePreviousViewportStart && previousViewportStartModelPosition) {
            var viewPosition = this.coordinatesConverter.convertModelPositionToViewPosition(previousViewportStartModelPosition);
            var viewPositionTop = this.viewLayout.getVerticalOffsetForLineNumber(viewPosition.lineNumber);
            this.viewLayout.setScrollPosition({ scrollTop: viewPositionTop + this._viewportStartLineDelta }, 1 /* Immediate */);
        }
        if (CursorConfiguration.shouldRecreate(e)) {
            this.cursorConfig = new CursorConfiguration(this.model.getLanguageIdentifier(), this.model.getOptions(), this._configuration);
            this._cursor.updateConfiguration(this.cursorConfig);
        }
    };
    ViewModel.prototype._registerModelEvents = function () {
        var _this = this;
        this._register(this.model.onDidChangeRawContentFast(function (e) {
            var e_1, _a, e_2, _b, e_3, _c;
            try {
                var eventsCollector = _this._eventDispatcher.beginEmitViewEvents();
                var hadOtherModelChange = false;
                var hadModelLineChangeThatChangedLineMapping = false;
                var changes = e.changes;
                var versionId = e.versionId;
                // Do a first pass to compute line mappings, and a second pass to actually interpret them
                var lineBreaksComputer = _this._lines.createLineBreaksComputer();
                try {
                    for (var changes_1 = __values(changes), changes_1_1 = changes_1.next(); !changes_1_1.done; changes_1_1 = changes_1.next()) {
                        var change = changes_1_1.value;
                        switch (change.changeType) {
                            case 4 /* LinesInserted */: {
                                try {
                                    for (var _d = (e_2 = void 0, __values(change.detail)), _e = _d.next(); !_e.done; _e = _d.next()) {
                                        var line = _e.value;
                                        lineBreaksComputer.addRequest(line, null);
                                    }
                                }
                                catch (e_2_1) { e_2 = { error: e_2_1 }; }
                                finally {
                                    try {
                                        if (_e && !_e.done && (_b = _d.return)) _b.call(_d);
                                    }
                                    finally { if (e_2) throw e_2.error; }
                                }
                                break;
                            }
                            case 2 /* LineChanged */: {
                                lineBreaksComputer.addRequest(change.detail, null);
                                break;
                            }
                        }
                    }
                }
                catch (e_1_1) { e_1 = { error: e_1_1 }; }
                finally {
                    try {
                        if (changes_1_1 && !changes_1_1.done && (_a = changes_1.return)) _a.call(changes_1);
                    }
                    finally { if (e_1) throw e_1.error; }
                }
                var lineBreaks = lineBreaksComputer.finalize();
                var lineBreaksOffset = 0;
                try {
                    for (var changes_2 = __values(changes), changes_2_1 = changes_2.next(); !changes_2_1.done; changes_2_1 = changes_2.next()) {
                        var change = changes_2_1.value;
                        switch (change.changeType) {
                            case 1 /* Flush */: {
                                _this._lines.onModelFlushed();
                                eventsCollector.emitViewEvent(new viewEvents.ViewFlushedEvent());
                                _this._decorations.reset();
                                _this.viewLayout.onFlushed(_this.getLineCount());
                                hadOtherModelChange = true;
                                break;
                            }
                            case 3 /* LinesDeleted */: {
                                var linesDeletedEvent = _this._lines.onModelLinesDeleted(versionId, change.fromLineNumber, change.toLineNumber);
                                if (linesDeletedEvent !== null) {
                                    eventsCollector.emitViewEvent(linesDeletedEvent);
                                    _this.viewLayout.onLinesDeleted(linesDeletedEvent.fromLineNumber, linesDeletedEvent.toLineNumber);
                                }
                                hadOtherModelChange = true;
                                break;
                            }
                            case 4 /* LinesInserted */: {
                                var insertedLineBreaks = lineBreaks.slice(lineBreaksOffset, lineBreaksOffset + change.detail.length);
                                lineBreaksOffset += change.detail.length;
                                var linesInsertedEvent = _this._lines.onModelLinesInserted(versionId, change.fromLineNumber, change.toLineNumber, insertedLineBreaks);
                                if (linesInsertedEvent !== null) {
                                    eventsCollector.emitViewEvent(linesInsertedEvent);
                                    _this.viewLayout.onLinesInserted(linesInsertedEvent.fromLineNumber, linesInsertedEvent.toLineNumber);
                                }
                                hadOtherModelChange = true;
                                break;
                            }
                            case 2 /* LineChanged */: {
                                var changedLineBreakData = lineBreaks[lineBreaksOffset];
                                lineBreaksOffset++;
                                var _f = __read(_this._lines.onModelLineChanged(versionId, change.lineNumber, changedLineBreakData), 4), lineMappingChanged = _f[0], linesChangedEvent = _f[1], linesInsertedEvent = _f[2], linesDeletedEvent = _f[3];
                                hadModelLineChangeThatChangedLineMapping = lineMappingChanged;
                                if (linesChangedEvent) {
                                    eventsCollector.emitViewEvent(linesChangedEvent);
                                }
                                if (linesInsertedEvent) {
                                    eventsCollector.emitViewEvent(linesInsertedEvent);
                                    _this.viewLayout.onLinesInserted(linesInsertedEvent.fromLineNumber, linesInsertedEvent.toLineNumber);
                                }
                                if (linesDeletedEvent) {
                                    eventsCollector.emitViewEvent(linesDeletedEvent);
                                    _this.viewLayout.onLinesDeleted(linesDeletedEvent.fromLineNumber, linesDeletedEvent.toLineNumber);
                                }
                                break;
                            }
                            case 5 /* EOLChanged */: {
                                // Nothing to do. The new version will be accepted below
                                break;
                            }
                        }
                    }
                }
                catch (e_3_1) { e_3 = { error: e_3_1 }; }
                finally {
                    try {
                        if (changes_2_1 && !changes_2_1.done && (_c = changes_2.return)) _c.call(changes_2);
                    }
                    finally { if (e_3) throw e_3.error; }
                }
                _this._lines.acceptVersionId(versionId);
                _this.viewLayout.onHeightMaybeChanged();
                if (!hadOtherModelChange && hadModelLineChangeThatChangedLineMapping) {
                    eventsCollector.emitViewEvent(new viewEvents.ViewLineMappingChangedEvent());
                    eventsCollector.emitViewEvent(new viewEvents.ViewDecorationsChangedEvent(null));
                    _this._cursor.onLineMappingChanged(eventsCollector);
                    _this._decorations.onLineMappingChanged();
                }
            }
            finally {
                _this._eventDispatcher.endEmitViewEvents();
            }
            // Update the configuration and reset the centered view line
            _this._viewportStartLine = -1;
            _this._configuration.setMaxLineNumber(_this.model.getLineCount());
            _this._updateConfigurationViewLineCountNow();
            // Recover viewport
            if (!_this._hasFocus && _this.model.getAttachedEditorCount() >= 2 && _this._viewportStartLineTrackedRange) {
                var modelRange = _this.model._getTrackedRange(_this._viewportStartLineTrackedRange);
                if (modelRange) {
                    var viewPosition = _this.coordinatesConverter.convertModelPositionToViewPosition(modelRange.getStartPosition());
                    var viewPositionTop = _this.viewLayout.getVerticalOffsetForLineNumber(viewPosition.lineNumber);
                    _this.viewLayout.setScrollPosition({ scrollTop: viewPositionTop + _this._viewportStartLineDelta }, 1 /* Immediate */);
                }
            }
            try {
                var eventsCollector = _this._eventDispatcher.beginEmitViewEvents();
                _this._cursor.onModelContentChanged(eventsCollector, e);
            }
            finally {
                _this._eventDispatcher.endEmitViewEvents();
            }
        }));
        this._register(this.model.onDidChangeTokens(function (e) {
            var viewRanges = [];
            for (var j = 0, lenJ = e.ranges.length; j < lenJ; j++) {
                var modelRange = e.ranges[j];
                var viewStartLineNumber = _this.coordinatesConverter.convertModelPositionToViewPosition(new Position(modelRange.fromLineNumber, 1)).lineNumber;
                var viewEndLineNumber = _this.coordinatesConverter.convertModelPositionToViewPosition(new Position(modelRange.toLineNumber, _this.model.getLineMaxColumn(modelRange.toLineNumber))).lineNumber;
                viewRanges[j] = {
                    fromLineNumber: viewStartLineNumber,
                    toLineNumber: viewEndLineNumber
                };
            }
            _this._eventDispatcher.emitSingleViewEvent(new viewEvents.ViewTokensChangedEvent(viewRanges));
            if (e.tokenizationSupportChanged) {
                _this._tokenizeViewportSoon.schedule();
            }
        }));
        this._register(this.model.onDidChangeLanguageConfiguration(function (e) {
            _this._eventDispatcher.emitSingleViewEvent(new viewEvents.ViewLanguageConfigurationEvent());
            _this.cursorConfig = new CursorConfiguration(_this.model.getLanguageIdentifier(), _this.model.getOptions(), _this._configuration);
            _this._cursor.updateConfiguration(_this.cursorConfig);
        }));
        this._register(this.model.onDidChangeLanguage(function (e) {
            _this.cursorConfig = new CursorConfiguration(_this.model.getLanguageIdentifier(), _this.model.getOptions(), _this._configuration);
            _this._cursor.updateConfiguration(_this.cursorConfig);
        }));
        this._register(this.model.onDidChangeOptions(function (e) {
            // A tab size change causes a line mapping changed event => all view parts will repaint OK, no further event needed here
            if (_this._lines.setTabSize(_this.model.getOptions().tabSize)) {
                try {
                    var eventsCollector = _this._eventDispatcher.beginEmitViewEvents();
                    eventsCollector.emitViewEvent(new viewEvents.ViewFlushedEvent());
                    eventsCollector.emitViewEvent(new viewEvents.ViewLineMappingChangedEvent());
                    eventsCollector.emitViewEvent(new viewEvents.ViewDecorationsChangedEvent(null));
                    _this._cursor.onLineMappingChanged(eventsCollector);
                    _this._decorations.onLineMappingChanged();
                    _this.viewLayout.onFlushed(_this.getLineCount());
                }
                finally {
                    _this._eventDispatcher.endEmitViewEvents();
                }
                _this._updateConfigurationViewLineCount.schedule();
            }
            _this.cursorConfig = new CursorConfiguration(_this.model.getLanguageIdentifier(), _this.model.getOptions(), _this._configuration);
            _this._cursor.updateConfiguration(_this.cursorConfig);
        }));
        this._register(this.model.onDidChangeDecorations(function (e) {
            _this._decorations.onModelDecorationsChanged();
            _this._eventDispatcher.emitSingleViewEvent(new viewEvents.ViewDecorationsChangedEvent(e));
        }));
    };
    ViewModel.prototype.setHiddenAreas = function (ranges) {
        try {
            var eventsCollector = this._eventDispatcher.beginEmitViewEvents();
            var lineMappingChanged = this._lines.setHiddenAreas(ranges);
            if (lineMappingChanged) {
                eventsCollector.emitViewEvent(new viewEvents.ViewFlushedEvent());
                eventsCollector.emitViewEvent(new viewEvents.ViewLineMappingChangedEvent());
                eventsCollector.emitViewEvent(new viewEvents.ViewDecorationsChangedEvent(null));
                this._cursor.onLineMappingChanged(eventsCollector);
                this._decorations.onLineMappingChanged();
                this.viewLayout.onFlushed(this.getLineCount());
                this.viewLayout.onHeightMaybeChanged();
            }
        }
        finally {
            this._eventDispatcher.endEmitViewEvents();
        }
        this._updateConfigurationViewLineCount.schedule();
    };
    ViewModel.prototype.getVisibleRangesPlusViewportAboveBelow = function () {
        var layoutInfo = this._configuration.options.get(124 /* layoutInfo */);
        var lineHeight = this._configuration.options.get(53 /* lineHeight */);
        var linesAround = Math.max(20, Math.round(layoutInfo.height / lineHeight));
        var partialData = this.viewLayout.getLinesViewportData();
        var startViewLineNumber = Math.max(1, partialData.completelyVisibleStartLineNumber - linesAround);
        var endViewLineNumber = Math.min(this.getLineCount(), partialData.completelyVisibleEndLineNumber + linesAround);
        return this._toModelVisibleRanges(new Range(startViewLineNumber, this.getLineMinColumn(startViewLineNumber), endViewLineNumber, this.getLineMaxColumn(endViewLineNumber)));
    };
    ViewModel.prototype.getVisibleRanges = function () {
        var visibleViewRange = this.getCompletelyVisibleViewRange();
        return this._toModelVisibleRanges(visibleViewRange);
    };
    ViewModel.prototype._toModelVisibleRanges = function (visibleViewRange) {
        var visibleRange = this.coordinatesConverter.convertViewRangeToModelRange(visibleViewRange);
        var hiddenAreas = this._lines.getHiddenAreas();
        if (hiddenAreas.length === 0) {
            return [visibleRange];
        }
        var result = [], resultLen = 0;
        var startLineNumber = visibleRange.startLineNumber;
        var startColumn = visibleRange.startColumn;
        var endLineNumber = visibleRange.endLineNumber;
        var endColumn = visibleRange.endColumn;
        for (var i = 0, len = hiddenAreas.length; i < len; i++) {
            var hiddenStartLineNumber = hiddenAreas[i].startLineNumber;
            var hiddenEndLineNumber = hiddenAreas[i].endLineNumber;
            if (hiddenEndLineNumber < startLineNumber) {
                continue;
            }
            if (hiddenStartLineNumber > endLineNumber) {
                continue;
            }
            if (startLineNumber < hiddenStartLineNumber) {
                result[resultLen++] = new Range(startLineNumber, startColumn, hiddenStartLineNumber - 1, this.model.getLineMaxColumn(hiddenStartLineNumber - 1));
            }
            startLineNumber = hiddenEndLineNumber + 1;
            startColumn = 1;
        }
        if (startLineNumber < endLineNumber || (startLineNumber === endLineNumber && startColumn < endColumn)) {
            result[resultLen++] = new Range(startLineNumber, startColumn, endLineNumber, endColumn);
        }
        return result;
    };
    ViewModel.prototype.getCompletelyVisibleViewRange = function () {
        var partialData = this.viewLayout.getLinesViewportData();
        var startViewLineNumber = partialData.completelyVisibleStartLineNumber;
        var endViewLineNumber = partialData.completelyVisibleEndLineNumber;
        return new Range(startViewLineNumber, this.getLineMinColumn(startViewLineNumber), endViewLineNumber, this.getLineMaxColumn(endViewLineNumber));
    };
    ViewModel.prototype.getCompletelyVisibleViewRangeAtScrollTop = function (scrollTop) {
        var partialData = this.viewLayout.getLinesViewportDataAtScrollTop(scrollTop);
        var startViewLineNumber = partialData.completelyVisibleStartLineNumber;
        var endViewLineNumber = partialData.completelyVisibleEndLineNumber;
        return new Range(startViewLineNumber, this.getLineMinColumn(startViewLineNumber), endViewLineNumber, this.getLineMaxColumn(endViewLineNumber));
    };
    ViewModel.prototype.saveState = function () {
        var compatViewState = this.viewLayout.saveState();
        var scrollTop = compatViewState.scrollTop;
        var firstViewLineNumber = this.viewLayout.getLineNumberAtVerticalOffset(scrollTop);
        var firstPosition = this.coordinatesConverter.convertViewPositionToModelPosition(new Position(firstViewLineNumber, this.getLineMinColumn(firstViewLineNumber)));
        var firstPositionDeltaTop = this.viewLayout.getVerticalOffsetForLineNumber(firstViewLineNumber) - scrollTop;
        return {
            scrollLeft: compatViewState.scrollLeft,
            firstPosition: firstPosition,
            firstPositionDeltaTop: firstPositionDeltaTop
        };
    };
    ViewModel.prototype.reduceRestoreState = function (state) {
        if (typeof state.firstPosition === 'undefined') {
            // This is a view state serialized by an older version
            return this._reduceRestoreStateCompatibility(state);
        }
        var modelPosition = this.model.validatePosition(state.firstPosition);
        var viewPosition = this.coordinatesConverter.convertModelPositionToViewPosition(modelPosition);
        var scrollTop = this.viewLayout.getVerticalOffsetForLineNumber(viewPosition.lineNumber) - state.firstPositionDeltaTop;
        return {
            scrollLeft: state.scrollLeft,
            scrollTop: scrollTop
        };
    };
    ViewModel.prototype._reduceRestoreStateCompatibility = function (state) {
        return {
            scrollLeft: state.scrollLeft,
            scrollTop: state.scrollTopWithoutViewZones
        };
    };
    ViewModel.prototype.getTabSize = function () {
        return this.model.getOptions().tabSize;
    };
    ViewModel.prototype.getTextModelOptions = function () {
        return this.model.getOptions();
    };
    ViewModel.prototype.getLineCount = function () {
        return this._lines.getViewLineCount();
    };
    /**
     * Gives a hint that a lot of requests are about to come in for these line numbers.
     */
    ViewModel.prototype.setViewport = function (startLineNumber, endLineNumber, centeredLineNumber) {
        this._viewportStartLine = startLineNumber;
        var position = this.coordinatesConverter.convertViewPositionToModelPosition(new Position(startLineNumber, this.getLineMinColumn(startLineNumber)));
        this._viewportStartLineTrackedRange = this.model._setTrackedRange(this._viewportStartLineTrackedRange, new Range(position.lineNumber, position.column, position.lineNumber, position.column), 1 /* NeverGrowsWhenTypingAtEdges */);
        var viewportStartLineTop = this.viewLayout.getVerticalOffsetForLineNumber(startLineNumber);
        var scrollTop = this.viewLayout.getCurrentScrollTop();
        this._viewportStartLineDelta = scrollTop - viewportStartLineTop;
    };
    ViewModel.prototype.getActiveIndentGuide = function (lineNumber, minLineNumber, maxLineNumber) {
        return this._lines.getActiveIndentGuide(lineNumber, minLineNumber, maxLineNumber);
    };
    ViewModel.prototype.getLinesIndentGuides = function (startLineNumber, endLineNumber) {
        return this._lines.getViewLinesIndentGuides(startLineNumber, endLineNumber);
    };
    ViewModel.prototype.getLineContent = function (lineNumber) {
        return this._lines.getViewLineContent(lineNumber);
    };
    ViewModel.prototype.getLineLength = function (lineNumber) {
        return this._lines.getViewLineLength(lineNumber);
    };
    ViewModel.prototype.getLineMinColumn = function (lineNumber) {
        return this._lines.getViewLineMinColumn(lineNumber);
    };
    ViewModel.prototype.getLineMaxColumn = function (lineNumber) {
        return this._lines.getViewLineMaxColumn(lineNumber);
    };
    ViewModel.prototype.getLineFirstNonWhitespaceColumn = function (lineNumber) {
        var result = strings.firstNonWhitespaceIndex(this.getLineContent(lineNumber));
        if (result === -1) {
            return 0;
        }
        return result + 1;
    };
    ViewModel.prototype.getLineLastNonWhitespaceColumn = function (lineNumber) {
        var result = strings.lastNonWhitespaceIndex(this.getLineContent(lineNumber));
        if (result === -1) {
            return 0;
        }
        return result + 2;
    };
    ViewModel.prototype.getDecorationsInViewport = function (visibleRange) {
        return this._decorations.getDecorationsViewportData(visibleRange).decorations;
    };
    ViewModel.prototype.getViewLineRenderingData = function (visibleRange, lineNumber) {
        var mightContainRTL = this.model.mightContainRTL();
        var mightContainNonBasicASCII = this.model.mightContainNonBasicASCII();
        var tabSize = this.getTabSize();
        var lineData = this._lines.getViewLineData(lineNumber);
        var allInlineDecorations = this._decorations.getDecorationsViewportData(visibleRange).inlineDecorations;
        var inlineDecorations = allInlineDecorations[lineNumber - visibleRange.startLineNumber];
        return new ViewLineRenderingData(lineData.minColumn, lineData.maxColumn, lineData.content, lineData.continuesWithWrappedLine, mightContainRTL, mightContainNonBasicASCII, lineData.tokens, inlineDecorations, tabSize, lineData.startVisibleColumn);
    };
    ViewModel.prototype.getViewLineData = function (lineNumber) {
        return this._lines.getViewLineData(lineNumber);
    };
    ViewModel.prototype.getMinimapLinesRenderingData = function (startLineNumber, endLineNumber, needed) {
        var result = this._lines.getViewLinesData(startLineNumber, endLineNumber, needed);
        return new MinimapLinesRenderingData(this.getTabSize(), result);
    };
    ViewModel.prototype.getAllOverviewRulerDecorations = function (theme) {
        return this._lines.getAllOverviewRulerDecorations(this._editorId, filterValidationDecorations(this._configuration.options), theme);
    };
    ViewModel.prototype.invalidateOverviewRulerColorCache = function () {
        var e_4, _a;
        var decorations = this.model.getOverviewRulerDecorations();
        try {
            for (var decorations_1 = __values(decorations), decorations_1_1 = decorations_1.next(); !decorations_1_1.done; decorations_1_1 = decorations_1.next()) {
                var decoration = decorations_1_1.value;
                var opts = decoration.options.overviewRuler;
                if (opts) {
                    opts.invalidateCachedColor();
                }
            }
        }
        catch (e_4_1) { e_4 = { error: e_4_1 }; }
        finally {
            try {
                if (decorations_1_1 && !decorations_1_1.done && (_a = decorations_1.return)) _a.call(decorations_1);
            }
            finally { if (e_4) throw e_4.error; }
        }
    };
    ViewModel.prototype.invalidateMinimapColorCache = function () {
        var e_5, _a;
        var decorations = this.model.getAllDecorations();
        try {
            for (var decorations_2 = __values(decorations), decorations_2_1 = decorations_2.next(); !decorations_2_1.done; decorations_2_1 = decorations_2.next()) {
                var decoration = decorations_2_1.value;
                var opts = decoration.options.minimap;
                if (opts) {
                    opts.invalidateCachedColor();
                }
            }
        }
        catch (e_5_1) { e_5 = { error: e_5_1 }; }
        finally {
            try {
                if (decorations_2_1 && !decorations_2_1.done && (_a = decorations_2.return)) _a.call(decorations_2);
            }
            finally { if (e_5) throw e_5.error; }
        }
    };
    ViewModel.prototype.getValueInRange = function (range, eol) {
        var modelRange = this.coordinatesConverter.convertViewRangeToModelRange(range);
        return this.model.getValueInRange(modelRange, eol);
    };
    ViewModel.prototype.getModelLineMaxColumn = function (modelLineNumber) {
        return this.model.getLineMaxColumn(modelLineNumber);
    };
    ViewModel.prototype.validateModelPosition = function (position) {
        return this.model.validatePosition(position);
    };
    ViewModel.prototype.validateModelRange = function (range) {
        return this.model.validateRange(range);
    };
    ViewModel.prototype.deduceModelPositionRelativeToViewPosition = function (viewAnchorPosition, deltaOffset, lineFeedCnt) {
        var modelAnchor = this.coordinatesConverter.convertViewPositionToModelPosition(viewAnchorPosition);
        if (this.model.getEOL().length === 2) {
            // This model uses CRLF, so the delta must take that into account
            if (deltaOffset < 0) {
                deltaOffset -= lineFeedCnt;
            }
            else {
                deltaOffset += lineFeedCnt;
            }
        }
        var modelAnchorOffset = this.model.getOffsetAt(modelAnchor);
        var resultOffset = modelAnchorOffset + deltaOffset;
        return this.model.getPositionAt(resultOffset);
    };
    ViewModel.prototype.getEOL = function () {
        return this.model.getEOL();
    };
    ViewModel.prototype.getPlainTextToCopy = function (modelRanges, emptySelectionClipboard, forceCRLF) {
        var e_6, _a, e_7, _b, e_8, _c;
        var newLineCharacter = forceCRLF ? '\r\n' : this.model.getEOL();
        modelRanges = modelRanges.slice(0);
        modelRanges.sort(Range.compareRangesUsingStarts);
        var hasEmptyRange = false;
        var hasNonEmptyRange = false;
        try {
            for (var modelRanges_1 = __values(modelRanges), modelRanges_1_1 = modelRanges_1.next(); !modelRanges_1_1.done; modelRanges_1_1 = modelRanges_1.next()) {
                var range = modelRanges_1_1.value;
                if (range.isEmpty()) {
                    hasEmptyRange = true;
                }
                else {
                    hasNonEmptyRange = true;
                }
            }
        }
        catch (e_6_1) { e_6 = { error: e_6_1 }; }
        finally {
            try {
                if (modelRanges_1_1 && !modelRanges_1_1.done && (_a = modelRanges_1.return)) _a.call(modelRanges_1);
            }
            finally { if (e_6) throw e_6.error; }
        }
        if (!hasNonEmptyRange) {
            // all ranges are empty
            if (!emptySelectionClipboard) {
                return '';
            }
            var modelLineNumbers = modelRanges.map(function (r) { return r.startLineNumber; });
            var result_1 = '';
            for (var i = 0; i < modelLineNumbers.length; i++) {
                if (i > 0 && modelLineNumbers[i - 1] === modelLineNumbers[i]) {
                    continue;
                }
                result_1 += this.model.getLineContent(modelLineNumbers[i]) + newLineCharacter;
            }
            return result_1;
        }
        if (hasEmptyRange && emptySelectionClipboard) {
            // mixed empty selections and non-empty selections
            var result_2 = [];
            var prevModelLineNumber = 0;
            try {
                for (var modelRanges_2 = __values(modelRanges), modelRanges_2_1 = modelRanges_2.next(); !modelRanges_2_1.done; modelRanges_2_1 = modelRanges_2.next()) {
                    var modelRange = modelRanges_2_1.value;
                    var modelLineNumber = modelRange.startLineNumber;
                    if (modelRange.isEmpty()) {
                        if (modelLineNumber !== prevModelLineNumber) {
                            result_2.push(this.model.getLineContent(modelLineNumber));
                        }
                    }
                    else {
                        result_2.push(this.model.getValueInRange(modelRange, forceCRLF ? 2 /* CRLF */ : 0 /* TextDefined */));
                    }
                    prevModelLineNumber = modelLineNumber;
                }
            }
            catch (e_7_1) { e_7 = { error: e_7_1 }; }
            finally {
                try {
                    if (modelRanges_2_1 && !modelRanges_2_1.done && (_b = modelRanges_2.return)) _b.call(modelRanges_2);
                }
                finally { if (e_7) throw e_7.error; }
            }
            return result_2.length === 1 ? result_2[0] : result_2;
        }
        var result = [];
        try {
            for (var modelRanges_3 = __values(modelRanges), modelRanges_3_1 = modelRanges_3.next(); !modelRanges_3_1.done; modelRanges_3_1 = modelRanges_3.next()) {
                var modelRange = modelRanges_3_1.value;
                if (!modelRange.isEmpty()) {
                    result.push(this.model.getValueInRange(modelRange, forceCRLF ? 2 /* CRLF */ : 0 /* TextDefined */));
                }
            }
        }
        catch (e_8_1) { e_8 = { error: e_8_1 }; }
        finally {
            try {
                if (modelRanges_3_1 && !modelRanges_3_1.done && (_c = modelRanges_3.return)) _c.call(modelRanges_3);
            }
            finally { if (e_8) throw e_8.error; }
        }
        return result.length === 1 ? result[0] : result;
    };
    ViewModel.prototype.getRichTextToCopy = function (modelRanges, emptySelectionClipboard) {
        var languageId = this.model.getLanguageIdentifier();
        if (languageId.id === 1 /* PlainText */) {
            return null;
        }
        if (modelRanges.length !== 1) {
            // no multiple selection support at this time
            return null;
        }
        var range = modelRanges[0];
        if (range.isEmpty()) {
            if (!emptySelectionClipboard) {
                // nothing to copy
                return null;
            }
            var lineNumber = range.startLineNumber;
            range = new Range(lineNumber, this.model.getLineMinColumn(lineNumber), lineNumber, this.model.getLineMaxColumn(lineNumber));
        }
        var fontInfo = this._configuration.options.get(38 /* fontInfo */);
        var colorMap = this._getColorMap();
        var fontFamily = fontInfo.fontFamily === EDITOR_FONT_DEFAULTS.fontFamily ? fontInfo.fontFamily : "'" + fontInfo.fontFamily + "', " + EDITOR_FONT_DEFAULTS.fontFamily;
        return {
            mode: languageId.language,
            html: ("<div style=\""
                + ("color: " + colorMap[1 /* DefaultForeground */] + ";")
                + ("background-color: " + colorMap[2 /* DefaultBackground */] + ";")
                + ("font-family: " + fontFamily + ";")
                + ("font-weight: " + fontInfo.fontWeight + ";")
                + ("font-size: " + fontInfo.fontSize + "px;")
                + ("line-height: " + fontInfo.lineHeight + "px;")
                + "white-space: pre;"
                + "\">"
                + this._getHTMLToCopy(range, colorMap)
                + '</div>')
        };
    };
    ViewModel.prototype._getHTMLToCopy = function (modelRange, colorMap) {
        var startLineNumber = modelRange.startLineNumber;
        var startColumn = modelRange.startColumn;
        var endLineNumber = modelRange.endLineNumber;
        var endColumn = modelRange.endColumn;
        var tabSize = this.getTabSize();
        var result = '';
        for (var lineNumber = startLineNumber; lineNumber <= endLineNumber; lineNumber++) {
            var lineTokens = this.model.getLineTokens(lineNumber);
            var lineContent = lineTokens.getLineContent();
            var startOffset = (lineNumber === startLineNumber ? startColumn - 1 : 0);
            var endOffset = (lineNumber === endLineNumber ? endColumn - 1 : lineContent.length);
            if (lineContent === '') {
                result += '<br>';
            }
            else {
                result += tokenizeLineToHTML(lineContent, lineTokens.inflate(), colorMap, startOffset, endOffset, tabSize, platform.isWindows);
            }
        }
        return result;
    };
    ViewModel.prototype._getColorMap = function () {
        var colorMap = TokenizationRegistry.getColorMap();
        var result = ['#000000'];
        if (colorMap) {
            for (var i = 1, len = colorMap.length; i < len; i++) {
                result[i] = Color.Format.CSS.formatHex(colorMap[i]);
            }
        }
        return result;
    };
    //#region model
    ViewModel.prototype.pushStackElement = function () {
        this.model.pushStackElement();
    };
    //#endregion
    //#region cursor operations
    ViewModel.prototype.getPrimaryCursorState = function () {
        return this._cursor.getPrimaryCursorState();
    };
    ViewModel.prototype.getLastAddedCursorIndex = function () {
        return this._cursor.getLastAddedCursorIndex();
    };
    ViewModel.prototype.getCursorStates = function () {
        return this._cursor.getCursorStates();
    };
    ViewModel.prototype.setCursorStates = function (source, reason, states) {
        var _this = this;
        this._withViewEventsCollector(function (eventsCollector) { return _this._cursor.setStates(eventsCollector, source, reason, states); });
    };
    ViewModel.prototype.getCursorColumnSelectData = function () {
        return this._cursor.getCursorColumnSelectData();
    };
    ViewModel.prototype.setCursorColumnSelectData = function (columnSelectData) {
        this._cursor.setCursorColumnSelectData(columnSelectData);
    };
    ViewModel.prototype.getPrevEditOperationType = function () {
        return this._cursor.getPrevEditOperationType();
    };
    ViewModel.prototype.setPrevEditOperationType = function (type) {
        this._cursor.setPrevEditOperationType(type);
    };
    ViewModel.prototype.getSelection = function () {
        return this._cursor.getSelection();
    };
    ViewModel.prototype.getSelections = function () {
        return this._cursor.getSelections();
    };
    ViewModel.prototype.getPosition = function () {
        return this._cursor.getPrimaryCursorState().modelState.position;
    };
    ViewModel.prototype.setSelections = function (source, selections, reason) {
        var _this = this;
        if (reason === void 0) { reason = 0 /* NotSet */; }
        this._withViewEventsCollector(function (eventsCollector) { return _this._cursor.setSelections(eventsCollector, source, selections, reason); });
    };
    ViewModel.prototype.saveCursorState = function () {
        return this._cursor.saveState();
    };
    ViewModel.prototype.restoreCursorState = function (states) {
        var _this = this;
        this._withViewEventsCollector(function (eventsCollector) { return _this._cursor.restoreState(eventsCollector, states); });
    };
    ViewModel.prototype._executeCursorEdit = function (callback) {
        if (this._cursor.context.cursorConfig.readOnly) {
            // we cannot edit when read only...
            this._eventDispatcher.emitOutgoingEvent(new ReadOnlyEditAttemptEvent());
            return;
        }
        this._withViewEventsCollector(callback);
    };
    ViewModel.prototype.executeEdits = function (source, edits, cursorStateComputer) {
        var _this = this;
        this._executeCursorEdit(function (eventsCollector) { return _this._cursor.executeEdits(eventsCollector, source, edits, cursorStateComputer); });
    };
    ViewModel.prototype.startComposition = function () {
        var _this = this;
        this._cursor.setIsDoingComposition(true);
        this._executeCursorEdit(function (eventsCollector) { return _this._cursor.startComposition(eventsCollector); });
    };
    ViewModel.prototype.endComposition = function (source) {
        var _this = this;
        this._cursor.setIsDoingComposition(false);
        this._executeCursorEdit(function (eventsCollector) { return _this._cursor.endComposition(eventsCollector, source); });
    };
    ViewModel.prototype.type = function (text, source) {
        var _this = this;
        this._executeCursorEdit(function (eventsCollector) { return _this._cursor.type(eventsCollector, text, source); });
    };
    ViewModel.prototype.compositionType = function (text, replacePrevCharCnt, replaceNextCharCnt, positionDelta, source) {
        var _this = this;
        this._executeCursorEdit(function (eventsCollector) { return _this._cursor.compositionType(eventsCollector, text, replacePrevCharCnt, replaceNextCharCnt, positionDelta, source); });
    };
    ViewModel.prototype.paste = function (text, pasteOnNewLine, multicursorText, source) {
        var _this = this;
        this._executeCursorEdit(function (eventsCollector) { return _this._cursor.paste(eventsCollector, text, pasteOnNewLine, multicursorText, source); });
    };
    ViewModel.prototype.cut = function (source) {
        var _this = this;
        this._executeCursorEdit(function (eventsCollector) { return _this._cursor.cut(eventsCollector, source); });
    };
    ViewModel.prototype.executeCommand = function (command, source) {
        var _this = this;
        this._executeCursorEdit(function (eventsCollector) { return _this._cursor.executeCommand(eventsCollector, command, source); });
    };
    ViewModel.prototype.executeCommands = function (commands, source) {
        var _this = this;
        this._executeCursorEdit(function (eventsCollector) { return _this._cursor.executeCommands(eventsCollector, commands, source); });
    };
    ViewModel.prototype.revealPrimaryCursor = function (source, revealHorizontal) {
        var _this = this;
        this._withViewEventsCollector(function (eventsCollector) { return _this._cursor.revealPrimary(eventsCollector, source, revealHorizontal, 0 /* Smooth */); });
    };
    ViewModel.prototype.revealTopMostCursor = function (source) {
        var viewPosition = this._cursor.getTopMostViewPosition();
        var viewRange = new Range(viewPosition.lineNumber, viewPosition.column, viewPosition.lineNumber, viewPosition.column);
        this._withViewEventsCollector(function (eventsCollector) { return eventsCollector.emitViewEvent(new viewEvents.ViewRevealRangeRequestEvent(source, viewRange, null, 0 /* Simple */, true, 0 /* Smooth */)); });
    };
    ViewModel.prototype.revealBottomMostCursor = function (source) {
        var viewPosition = this._cursor.getBottomMostViewPosition();
        var viewRange = new Range(viewPosition.lineNumber, viewPosition.column, viewPosition.lineNumber, viewPosition.column);
        this._withViewEventsCollector(function (eventsCollector) { return eventsCollector.emitViewEvent(new viewEvents.ViewRevealRangeRequestEvent(source, viewRange, null, 0 /* Simple */, true, 0 /* Smooth */)); });
    };
    ViewModel.prototype.revealRange = function (source, revealHorizontal, viewRange, verticalType, scrollType) {
        this._withViewEventsCollector(function (eventsCollector) { return eventsCollector.emitViewEvent(new viewEvents.ViewRevealRangeRequestEvent(source, viewRange, null, verticalType, revealHorizontal, scrollType)); });
    };
    //#endregion
    //#region viewLayout
    ViewModel.prototype.getVerticalOffsetForLineNumber = function (viewLineNumber) {
        return this.viewLayout.getVerticalOffsetForLineNumber(viewLineNumber);
    };
    ViewModel.prototype.getScrollTop = function () {
        return this.viewLayout.getCurrentScrollTop();
    };
    ViewModel.prototype.setScrollTop = function (newScrollTop, scrollType) {
        this.viewLayout.setScrollPosition({ scrollTop: newScrollTop }, scrollType);
    };
    ViewModel.prototype.setScrollPosition = function (position, type) {
        this.viewLayout.setScrollPosition(position, type);
    };
    ViewModel.prototype.deltaScrollNow = function (deltaScrollLeft, deltaScrollTop) {
        this.viewLayout.deltaScrollNow(deltaScrollLeft, deltaScrollTop);
    };
    ViewModel.prototype.changeWhitespace = function (callback) {
        var hadAChange = this.viewLayout.changeWhitespace(callback);
        if (hadAChange) {
            this._eventDispatcher.emitSingleViewEvent(new viewEvents.ViewZonesChangedEvent());
            this._eventDispatcher.emitOutgoingEvent(new ViewZonesChangedEvent());
        }
    };
    ViewModel.prototype.setMaxLineWidth = function (maxLineWidth) {
        this.viewLayout.setMaxLineWidth(maxLineWidth);
    };
    //#endregion
    ViewModel.prototype._withViewEventsCollector = function (callback) {
        try {
            var eventsCollector = this._eventDispatcher.beginEmitViewEvents();
            callback(eventsCollector);
        }
        finally {
            this._eventDispatcher.endEmitViewEvents();
        }
    };
    return ViewModel;
}(Disposable));
export { ViewModel };
