/* @(#)fifo.h	1.20 05/07/19 Copyright 1989-2004 J. Schilling */
/*
 *	Definitions for a "fifo" that uses
 *	shared memory between two processes
 *
 *	Copyright (c) 1989-2004 J. Schilling
 */
/*
 * Copyright Jrg Schilling. All rights reserved.
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only.
 * See the file CDDL.Schily.txt in this distribution or
 * http://opensource.org/licenses/cddl1.php for details.
 */

#ifndef	_FIFO_H
#define	_FIFO_H

#ifndef	_BITSTRING_H
#include "bitstring.h"
#endif

typedef	struct	{
	BOOL	reading;	/* true if currently reading from tape	    */
	int	swapflg;	/* -1: init, 0: FALSE, 1: TRUE		    */
	int	hdrtype;	/* Hdrtype used for read/write		    */
	int	volno;		/* Volume #				    */
	int	nblocks;	/* Blocksize for each transfer in TBLOCK    */
	long	blocksize;	/* Blocksize for each transfer in bytes	    */
	long	lastsize;	/* Size of last transfer (for backtape)	    */
	Llong	blocks;		/* Full blocks transfered on Volume	    */
	Llong	parts;		/* Bytes fom partial transferes on Volume   */
	Llong	Tblocks;	/* Total blocks transfered		    */
	Llong	Tparts;		/* Total Bytes fom parttial transferes	    */
	off_t	cur_size;	/* The size of the current file (multivol)  */
	off_t	cur_off;	/* The off for the current file (multivol)  */
	off_t	old_size;	/* The size for the last block write	    */
	off_t	old_off;	/* The off for the last block wrte	    */
} m_stats;

typedef struct {
	char	*putptr;	/* put pointer within shared memory	    */
	char	*getptr;	/* get pointer within shared memory	    */
	char	*base;		/* base of fifo within shared memory segment */
	char	*end;		/* end of real shared memory segment	    */
	int	size;		/* size of fifo within shared memory segment */
	int	ibs;		/* input transfer size			    */
	int	obs;		/* output transfer size			    */
	int	rsize;		/* rest size between head struct and .base  */
	unsigned long	icnt;	/* input count (incremented on each put)    */
	unsigned long	ocnt;	/* output count (incremented on each get)   */
	int	hiw;		/* highwater mark			    */
	int	low;		/* lowwater mark			    */
	int	flags;		/* fifo flags				    */
	int	ferrno;		/* errno from fifo background process	    */
	int	gp[2];		/* sync pipe for get process		    */
	int	pp[2];		/* sync pipe for put process		    */
	int	puts;		/* fifo put count statistic		    */
	int	gets;		/* fifo get get statistic		    */
	int	empty;		/* fifo was empty count statistic	    */
	int	full;		/* fifo was full count statistic	    */
	int	maxfill;	/* max # of bytes in fifo		    */
	int	moves;		/* # of moves of residual bytes		    */
	Llong	mbytes;		/* # of residual bytes moved		    */
	m_stats	stats;		/* statistics				    */
	bitstr_t *bmap;		/* Bitmap used to store TCB positions	    */
	int	bmlast;		/* Last bits # in use in above Bitmap	    */
	GINFO	ginfo;		/* To share GINFO for P.1-2001 'g' headers  */
} m_head;

#define	gpin	gp[0]		/* get pipe in  */
#define	gpout	gp[1]		/* get pipe out */
#define	ppin	pp[0]		/* put pipe in  */
#define	ppout	pp[1]		/* put pipe out */

#define	FIFO_AMOUNT(p)	((p)->icnt - (p)->ocnt)

#define	FIFO_IBLOCKED	0x001	/* input  (put side) is blocked	*/
#define	FIFO_OBLOCKED	0x002	/* output (get side) is blocked	*/
#define	FIFO_FULL	0x004	/* fifo is full			*/
#define	FIFO_MEOF	0x008	/* EOF on input (put side)	*/
#define	FIFO_MERROR	0x010	/* error on input (put side)	*/
#define	FIFO_EXIT	0x020	/* exit() on non tape side	*/
#define	FIFO_EXERRNO	0x040	/* errno from non tape side	*/

#define	FIFO_IWAIT	0x200	/* input (put side) waits after first record */
#define	FIFO_I_CHREEL	0x400	/* change input tape reel if fifo gets empty */
#define	FIFO_O_CHREEL	0x800	/* change output tape reel if fifo gets empty */

#define	FIFO_M1		0x1000	/* Semaphore claimed by newvolhdr()	*/
#define	FIFO_M2		0x2000	/* Semaphore claimed by cr_file()	*/

#if	!defined(HAVE_SMMAP) && !defined(HAVE_USGSHM) && !defined(HAVE_DOSALLOCSHAREDMEM)
#undef	FIFO			/* We cannot have a FIFO on this platform */
#endif

#ifdef	FIFO
/*
 * Critical section handling for multi volume support.
 *
 * This code is used to protect access to stat->cur_size & stat->cur_off
 * when preparing the multi volume header.
 */
extern	BOOL	use_fifo;

#define	fifo_enter_critical()	if (use_fifo) { \
				extern  m_head  *mp;		\
								\
				while (mp->flags & FIFO_M1)	\
					usleep(100000);		\
				mp->flags |= FIFO_M2;		\
			}

#define	fifo_leave_critical()	if (use_fifo) { \
				extern  m_head  *mp;		\
								\
				mp->flags &= ~FIFO_M2;		\
			}

#define	fifo_lock_critical()	if (use_fifo) { \
				extern  m_head  *mp;		\
								\
				mp->flags |= FIFO_M1;		\
				while (mp->flags & FIFO_M2)	\
					usleep(100000);		\
			}

#define	fifo_unlock_critical()	if (use_fifo) { \
				extern  m_head  *mp;		\
								\
				mp->flags &= ~FIFO_M1;		\
			}
#else
#define	fifo_enter_critical()
#define	fifo_leave_critical()
#define	fifo_lock_critical()
#define	fifo_unlock_critical()
#endif

#endif /* _FIFO_H */
