/*
 * Copyright (c)  2000
 * SWsoft  company
 *
 * This material is provided "as is", with absolutely no warranty expressed
 * or implied. Any use is at your own risk.
 *
 * Permission to use or copy this software for any purpose is hereby granted 
 * without fee, provided the above notices are retained on all copies.
 * Permission to modify the code and to distribute modified code is granted,
 * provided the above notices are retained, and a notice that the code was
 * modified is included with the above copyright notice.
 *
 */

//--------------------------------------------------------------------
// MySQL OLE DB Provider 
// Functionality: minimum
// Release: 0.1
//
// @doc
//
// @module COMMAND.CPP | CCommand object implementation
//
//

// Includes ------------------------------------------------------------------

#include "hfiles.h"
#include "headers.h"
#include "Errors.h"

// Get OLE DB data type from wide char
DBTYPE GetOledbTypeFromName(LPOLESTR wszName, BOOL* pbIsMySqlBlob = NULL ); 


#include "mysql.h"

// Code ----------------------------------------------------------------------

// CCommand::CCommand --------------------------------------------------
//
// @mfunc Constructor for this class
//
// @rdesc NONE
//
CCommand::CCommand
    (
    LPUNKNOWN pUnkOuter         //@parm IN | Outer Unkown Pointer
    )
{
    CLEAR_CONSTRUCT( CCommand );

	//  Initialize simple member vars
	m_pUnkOuter			= pUnkOuter ? pUnkOuter : this;
	m_cCols				= -1;
	m_bBookmark			= true;

    // Increment global object count.
    OBJECT_CONSTRUCTED();
}


// CCommand::~Command -------------------------------------------------
//
// @mfunc Destructor for this class
//
// @rdesc NONE
//
CCommand:: ~CCommand
    (
    void
    )
{
	//Unprepare command
	Unprepare();

    // Free properties management object
    delete m_pUtilProp;

    // Free accessors.
    // Each accessor is allocated via new/delete.
    // We store an array of ptrs to each accessor (m_pextbufferAccessor).
    if (NULL != m_pExtBufferAccessor)
    {
        ULONG       hAccessor, hAccessorLast;
        PACCESSOR   pAccessor;

        m_pExtBufferAccessor->GetFirstLastItemH( hAccessor, hAccessorLast );
        for (; hAccessor <= hAccessorLast; hAccessor++)
        {
            m_pExtBufferAccessor->GetItemOfExtBuffer( hAccessor, &pAccessor );
            delete [] pAccessor;
        }
    
		delete m_pExtBufferAccessor;
    }

    // Free contained interfaces
	delete m_pICommand;
    delete m_pICommandText;
	delete m_pICommandPrepare;
	delete m_pICommandWithParameters;
	delete m_pIConvertType;
	delete m_pIAccessor;
	delete m_pIColumnsInfo;
	delete m_pICommandProperties;
	delete m_pISupportErrorInfo;

	if (m_pwszCommand != NULL)
		g_pIMalloc->Free(m_pwszCommand);

	ClearColumnInfo();

	if (m_pDBSession != NULL)
		m_pDBSession->Release();

    // Decrement global object count.
    OBJECT_DESTRUCTED();

    return;
}


// CCommand::Init --------------------------------------------------------
//
// @mfunc Initialize the command Object
//
// @rdesc Did the Initialization Succeed
//      @flag  TRUE | Initialization succeeded
//      @flag  FALSE | Initialization failed
//
BOOL CCommand::Init
    (
	PCDBSESSION	pDBSession		//IN	| parent DBSession object
    )
{
    TRACE( "CCommand::Init" );
	
	LPUNKNOWN   pIUnknown = (LPUNKNOWN) this;

    if (m_pUnkOuter)
        pIUnknown = m_pUnkOuter;

	// Establish parent object pointer
	assert(pDBSession);
	m_pDBSession = pDBSession;
	m_pDBSession->AddRef();

    // Allocate properties management object
    m_pUtilProp = new CUtilProp( m_pDBSession->m_pCDataSource, m_pDBSession->m_pUtilProp );

    // Create the ExtBuffer array.
    // This is an array of pointers to malloc'd accessors.
    m_pExtBufferAccessor = (LPEXTBUFFER) new CExtBuffer;
    if (m_pExtBufferAccessor == NULL ||
		FAILED(m_pExtBufferAccessor->FInit( 1, sizeof( PACCESSOR ), g_dwPageSize)))
	{
        return E_FAIL;
	}

    // Allocate contained interface objects
    m_pICommand					= new CImpICommand(this, pIUnknown);
    m_pICommandText				= new CImpICommandText(this, pIUnknown);
    m_pICommandPrepare			= new CImpICommandPrepare(this, pIUnknown);
	m_pICommandWithParameters	= new CImpICommandWithParameters(this, pIUnknown);
	m_pIConvertType				= new CImpIConvertType(pIUnknown);
	m_pIAccessor				= new CImpIAccessor(m_pExtBufferAccessor, NULL, pIUnknown);
	m_pIColumnsInfo				= new CImpIColumnsInfo(&m_cCols, &m_bBookmark, &m_pColInfo, this, pIUnknown);
	m_pICommandProperties		= new CImpICommandProperties(this, pIUnknown);
	m_pISupportErrorInfo		= new CImpISupportErrorInfo(this, pIUnknown);
	
	return (BOOL)  (m_pUtilProp &&
					m_pICommand && 
					m_pICommandText &&
					m_pICommandPrepare &&
					m_pICommandWithParameters &&
					m_pIConvertType &&
					m_pIAccessor &&
					m_pIColumnsInfo &&
					m_pICommandProperties &&
					m_pISupportErrorInfo);
	
}



// CCommand::QueryInterface -----------------------------------------------
//
// @mfunc Returns a pointer to a specified interface. Callers use
// QueryInterface to determine which interfaces the called object
// supports.
//
// @rdesc HRESULT indicating the status of the method
//      @flag S_OK          | Interface is supported and ppvObject is set.
//      @flag E_NOINTERFACE | Interface is not supported by the object
//      @flag E_INVALIDARG  | One or more arguments are invalid.
//
STDMETHODIMP CCommand::QueryInterface
    (
    REFIID riid,        //@parm IN | Interface ID of the interface being queried for.
    LPVOID * ppv        //@parm OUT | Pointer to interface that was instantiated
    )
{
	TRACE( "CCommand::QueryInterface" );

    // Is the pointer bad?
    if (ppv == NULL)
        return E_INVALIDARG;

    //  Place NULL in *ppv in case of failure
    *ppv = NULL;

    //  This is the non-delegating IUnknown implementation
    if (riid == IID_IUnknown)
        *ppv = (LPVOID) this;
    else if (riid == IID_ICommand)
        *ppv = (LPVOID) m_pICommand;
    else if (riid == IID_ICommandText)
        *ppv = (LPVOID) m_pICommandText;
    else if (riid == IID_ICommandPrepare)
        *ppv = (LPVOID) m_pICommandPrepare;
    else if (riid == IID_ICommandWithParameters)
        *ppv = (LPVOID) m_pICommandWithParameters;
    else if (riid == IID_IConvertType)
        *ppv = (LPVOID) m_pIConvertType;
	else if (riid == IID_IAccessor)
		*ppv = (LPVOID) m_pIAccessor;
	else if (riid == IID_IColumnsInfo)
		*ppv = (LPVOID) m_pIColumnsInfo;
	else if (riid == IID_ICommandProperties)
		*ppv = (LPVOID) m_pICommandProperties;
#ifdef _DEBUG
	else if (riid == IID_ISupportErrorInfo)
		*ppv = (LPVOID) m_pISupportErrorInfo;
#endif

    if (*ppv == NULL)
		return E_NOINTERFACE;

    //  If we're going to return an interface, AddRef it first
    ((LPUNKNOWN) *ppv)->AddRef();
    
	return S_OK;
}

// CCommand::AddRef -------------------------------------------------------
//
// @mfunc Increments a persistence count for the object
//
// @rdesc Current reference count
//
STDMETHODIMP_( ULONG ) CCommand::AddRef
     (
     void
     )
{
    return ++m_cRef;
}


// CCommand::Release ------------------------------------------------------
//
// @mfunc Decrements a persistence count for the object and if
// persistence count is 0, the object destroys itself.
//
// @rdesc Current reference count
//
STDMETHODIMP_( ULONG ) CCommand::Release
     (
     void
     )
{
    if (!--m_cRef)
        {
        delete this;
        return 0;
        }

    return m_cRef;
}


//-----------------------------------------------------------------------------
// Extract next param starting from pchEnd.
// Place first param symbol in pchBegin. Place last param symbol in pchEnd
void NextParam(char** ppchBegin, char** ppchEnd)
{
	if (*ppchEnd == NULL)
	{
		*ppchBegin = NULL;
		return;
	}

	char* pch = *ppchEnd;

	//Determine first param sybol
	while (*pch != 0)
	{
		//Check if it is begin of string
		if (*pch == '\'')
		{
			//So it is string, skip it
			while (*pch != 0)
			{
				if (*pch == '\'')
				{
					pch++;
					break;
				}
			}

			continue;
		}


		//Previous letter is any delimiter, current letter is '?'
		if ((!__iscsym(*(pch - 1)) || pch == *ppchBegin) && *pch == '?')
			break;
		
		pch++;
	}

	//Check if we found param
	if (*pch == 0)
	{ 
		*ppchBegin = NULL;
		*ppchEnd = NULL;
		return;
	}

	//Store first symbol
	*ppchBegin = pch;

	//Determine last param sybol
	while (*pch != 0)
	{
		//Next letter is any delimiter
		if (!__iscsym(*(pch + 1)))
			break;
		
		pch++;
	}

	if (*pch == 0)
		*ppchEnd = NULL;
	else
		*ppchEnd = pch;
	
	return;
 }


//-----------------------------------------------------------------------------
// CCommand::Prepare
//
// @mfunc Prepare command
//
HRESULT CCommand::Prepare(bool bCheckRowsets)
{    
	TRACE( "CCommand::Prepare" );
	
	if (bCheckRowsets && m_cRowsets != 0)
		return DB_E_OBJECTOPEN;
	
	if (m_pszSQL != NULL)
		return S_OK;

	//Check if command text is set
	if (m_pwszCommand == NULL)
#ifdef _DEBUG
	{
		ClearError();
		return Error(DB_E_NOCOMMAND, 100, "Test Error");
	}
#else
		return DB_E_NOCOMMAND;
#endif

    // Create buffer for SQL text
	int nLen = wcslen(m_pwszCommand);
    m_pszSQL = new char[nLen + 1];
    if (m_pszSQL == NULL)
        return E_OUTOFMEMORY;
    
	// Convert SQL text
	int cCharsCopied = WideCharToMultiByte(CP_ACP, 0,
									   m_pwszCommand, nLen,
									   m_pszSQL, nLen + 1,
									   NULL, NULL);
	m_pszSQL[nLen] = 0;

    if (cCharsCopied != nLen)
	{
		delete [] m_pszSQL;
        return E_FAIL;
	}

	//Check if dumb consumer passes on table name
	if (strchr(m_pszSQL, ' ') == NULL)
	{
		//Init new string
		const char* pszPrefix = "SELECT * FROM ";
		char* pszCorrectedSQL = new char[nLen + strlen(pszPrefix) + 1];
		if (pszCorrectedSQL == NULL)
			return E_OUTOFMEMORY;

		//Change our string to SELECT * FROM TableName
		strcpy(pszCorrectedSQL, pszPrefix);
		strcat(pszCorrectedSQL, m_pszSQL);

		//Replace old string
		delete [] m_pszSQL;
		m_pszSQL = pszCorrectedSQL;
	}

	//Check if SQL text contains parameters
	char* pchBegin	= m_pszSQL;
	char* pchEnd	= m_pszSQL;
	NextParam(&pchBegin, &pchEnd);
	m_bParameters = (pchBegin != NULL);

	//Add spaces before and after comparisions signs
	ULONG nSize = 0;
	char* pch = m_pszSQL;
	while (*pch != 0)
	{
		nSize = 0;
		switch (*pch)
		{
		case '\'':
			//So it is string, skip it
			while (*pch != 0)
			{
				if (*pch == '\'')
				{
					pch++;
					break;
				}
			}
			continue;

		case '>':
			if (*(pch + 1) == '=')
				nSize = 2;
			else
				nSize = 1;
			break;
		
		case '<':
			if (*(pch + 1) == '=' || *(pch + 1) == '>')
				nSize = 2;
			else
				nSize = 1;

			break;
		
		case '=':
			nSize = 1;
			break;
		
		case '!':
			if (*(pch + 1) == '=')
				nSize = 2;
			break;
		}

		//Check if found comparision sign without spaces before and after
		if (nSize != 0 && 
			((pch != m_pszSQL && *(pch - 1) != ' ') ||
			 *(pch + nSize + 1) != ' '))
		{
			//Allocate new string
			LPSTR pszSQL = new char[strlen(m_pszSQL) + 1 + 2];
			//Copy all chars before sign
			ULONG nBeforeSign = pch - m_pszSQL;
			strncpy(pszSQL, m_pszSQL, nBeforeSign);

			//Add space before sign
			if (pch != m_pszSQL && *(pch - 1) != ' ')
			{
				pszSQL[nBeforeSign] = ' ';
				nBeforeSign++;
			}

			//Copy sign
			strncpy(pszSQL + nBeforeSign, pch, nSize);

			//Add space after sign 
			if (*(pch + nSize) != ' ')
			{
				pszSQL[nBeforeSign + nSize] = ' ';
				nBeforeSign++;
			}

			//Copy rest of string
			strcpy(pszSQL + nBeforeSign + nSize, pch + nSize);

			//Place new string into the old place
			delete [] m_pszSQL;
			m_pszSQL = pszSQL;
			pch = pszSQL + nBeforeSign + nSize - 1;
		}

		if( nSize == 2 )
			pch += 2;
		else
			pch ++;
	}

	return S_OK;
}


//-----------------------------------------------------------------------------
// CCommand::Unprepare
//
// @mfunc Unprepare command
//
HRESULT CCommand::Unprepare()
{    
	TRACE( "CCommand::Unprepare" );
	
	if (m_cRowsets != 0)
		return DB_E_OBJECTOPEN;

	if (m_pszSQL != NULL)
	{
		delete m_pszSQL;
		m_pszSQL = NULL;
	}

	return S_OK;
}

//-----------------------------------------------------------------------------
// CCommand::FillColumnInfo
//
// @mfunc Fills internal array with column info
//
// @rdesc 
//		@flag S_OK   | Succeed
//		@flag E_FAIL | Error
//
HRESULT CCommand::FillColumnInfo(CRowset *pRowset)
{
	TRACE( "CCommand::FillColumnInfo");
	
	//Check if we are already filled it
	if (m_pColInfo != NULL)
		return S_OK;

	//Check if we need to execute query first to fill column info
	if (pRowset == NULL)
	{
		HRESULT hr;
		hr = ((CImpICommand*)m_pICommand)->Execute(NULL, IID_IUnknown, NULL, NULL, (IUnknown**)&pRowset);
		if (FAILED(hr))
			return E_FAIL; //hr

		if (pRowset == NULL)
		{
			m_cCols = 0;
			return S_OK;
		}
	}
	else
		pRowset->AddRef(); 

	//Store column count
	m_cCols = pRowset->m_cCols;
	//Create column array
	m_pColInfo = new DBCOLUMNINFO[pRowset->m_cCols];
	//Copy column info
	memcpy(m_pColInfo, &pRowset->m_rgdbcolinfo[0], pRowset->m_cCols * sizeof(DBCOLUMNINFO));
	//Copy column names in special way
	for (int i = 1; i < pRowset->m_cCols; i++)
	{
		m_pColInfo[i].pwszName = new WCHAR[wcslen(pRowset->m_rgdbcolinfo[i].pwszName) + 1];
		wcscpy(m_pColInfo[i].pwszName, pRowset->m_rgdbcolinfo[i].pwszName);
	}

	pRowset->Release();

	return S_OK;
}


//-----------------------------------------------------------------------------
// CCommand::ClearColumnInfo
//
// @mfunc Clears internal array
//
// @rdesc 
//		@flag S_OK   | Succeed
//
HRESULT CCommand::ClearColumnInfo()
{
   TRACE( "CCommand::ClearColumnInfo" );

	
	if (m_pColInfo != NULL)
	{
		for (DWORD i = 1; i < m_cCols; i++)
			delete [] m_pColInfo[i].pwszName;
	
		delete [] m_pColInfo;
	}

	m_pColInfo = NULL;
	m_cCols	= -1;
	return S_OK;
}


//-----------------------------------------------------------------------------
// CImpICommand::Cancel 
//
// @mfunc Cancels the current command execution
//
// @rdesc 
//		@flag S_OK   | Operation canceled
//		@flag E_FAIL | Not implemented
//
STDMETHODIMP CImpICommand::Cancel
	(
	void
	)
{
	//Not implemented
	return E_FAIL;
}


//Substitute parameter values to SQL string
HRESULT CImpICommand::SubstituteParameters(LPSTR* ppszSQL, DBPARAMS* pParams)
{
	INTERFACE_METHOD_START( "ICommand::SubstituteParameters" );

	DBPARAMINFO* pParamInfo = m_pObj->m_pICommandWithParameters->m_pParamInfo;
	ULONG ulParamCount = m_pObj->m_pICommandWithParameters->m_nCount;

	*ppszSQL = m_pObj->m_pszSQL;

	if (!m_pObj->m_bParameters)
		return S_OK;

	if (pParams == NULL)
		return DB_E_PARAMNOTOPTIONAL;

	HRESULT hr;
	PACCESSOR pAccessor;
	assert( m_pObj->m_pExtBufferAccessor );
    hr = m_pObj->m_pExtBufferAccessor->GetItemOfExtBuffer((ULONG) pParams->hAccessor, &pAccessor );
    if (FAILED( hr ) || pAccessor == NULL)
        return DB_E_BADACCESSORHANDLE;

    assert( pAccessor );
    ULONG cBindings = pAccessor->cBindings;
    DBBINDING* pBinding  = pAccessor->rgBindings;

	assert(m_pObj->m_pszSQL != NULL);
	char* pchBegin = m_pObj->m_pszSQL;
	char* pchEnd = pchBegin;
	ULONG ulParam = 0;
	ULONG ulParsedParam = 0;
	ULONG ulBind = 0;
	bool bFound;
	bool bQuotations;
	ULONG ulOrdinal;

	while (true)
	{
		NextParam(&pchBegin, &pchEnd);

		if (pchBegin == NULL)
			break;
		
		ulParsedParam++;

		if (pParamInfo != NULL)
		{
			//Find param in param info 
			bFound = false;
			if (pchBegin == pchEnd)	//Check if it is unnamed parameter
			{
				//Check if all params are unnamed
				if (pParamInfo[0].pwszName != NULL)
					return DB_E_ERRORSINCOMMAND;

				//Find param by ordinal
				for (ulParam = 0; ulParam < ulParamCount; ulParam++)
				{
					if (ulParsedParam == pParamInfo[ulParam].iOrdinal)
					{
						bFound = true;
						break;
					}
				}
			}
			else
			{
				//Check if all params are named
				if (pParamInfo[0].pwszName == NULL)
					return DB_E_ERRORSINCOMMAND;

				//Convert param name to wide string
				int len = pchEnd - pchBegin;
				LPWSTR wszParamName = new WCHAR[len + 1];

				if (0 == MultiByteToWideChar( CP_ACP, MB_PRECOMPOSED, pchBegin + 1, len, wszParamName, len))
				{
					delete [] wszParamName;
					return DB_E_CANTCONVERTVALUE;
				}

				wszParamName[len] = 0;


				//Find param by name
				for (ulParam = 0; ulParam < ulParamCount; ulParam++)
				{
					if (0 == wcsicmp(wszParamName, pParamInfo[ulParam].pwszName))
					{
						bFound = true;
						break;
					}
				}
				
				delete [] wszParamName;
			}

			if (!bFound)
				return DB_E_PARAMNOTOPTIONAL;

			ulOrdinal = pParamInfo[ulParam].iOrdinal;
		}
		else
		{
			//No parameter information was set
			//We suppose that it is unnamed parameters
			if (pchBegin != pchEnd)
				return DB_E_PARAMNOTOPTIONAL;

			ulOrdinal = ulParsedParam;
		}

		//Find corresponded binding
		bFound = false;
		for (ulBind = 0; ulBind< cBindings; ulBind++)
		{
			if (ulOrdinal == pBinding[ulBind].iOrdinal)
			{
				bFound = true;
				break;
			}
		}
		
		if (!bFound)
			return DB_E_BADORDINAL;

        //Retrieve parameter value as string

		//Init variables
        DWORD dwSrcType      = pBinding[ulBind].wType;
        DWORD dwPart         = pBinding[ulBind].dwPart;

        void* pSrc           = dwPart & DBPART_VALUE ? ((BYTE*) pParams->pData + pBinding[ulBind].obValue) : NULL;
        ULONG* pulSrcLength  = dwPart & DBPART_LENGTH ? (ULONG *) ((BYTE*) pParams->pData + pBinding[ulBind].obLength) : NULL;
        DWORD* pdwSrcStatus  = dwPart & DBPART_STATUS ? (ULONG *) ((BYTE*) pParams->pData + pBinding[ulBind].obStatus) : NULL;

		//Get buffer size
		ULONG	ulDstMaxLength;
		hr = g_pIDataConvert->GetConversionSize(
				dwSrcType,
				DBTYPE_STR,
				pulSrcLength,
				&ulDstMaxLength,
				pSrc);

		//Allocate string for new SQL (add 10 to size for any case)
		char*	pszNewSQL = new char[strlen(*ppszSQL) + ulDstMaxLength + 10];
		//Copy all chars before parameter
		ULONG ulPrefixSize = pchBegin - *ppszSQL;
		strncpy(pszNewSQL, *ppszSQL, ulPrefixSize);

		if( dwSrcType == DBTYPE_STR || 
			dwSrcType == DBTYPE_WSTR || 
			dwSrcType == DBTYPE_DBDATE || 
			dwSrcType == DBTYPE_DBTIME || 
			dwSrcType == DBTYPE_DBTIMESTAMP || 
			dwSrcType == DBTYPE_BYTES )
				bQuotations = true;
		else
				bQuotations = false;

		//Add first quotation
		if (bQuotations)
		{
			*(pszNewSQL + ulPrefixSize) = '\'';
			ulPrefixSize++;  //Compensate adding quotation
		}

		//Copy value to new SQL string
		DWORD	dwDstStatus;
		ULONG	ulDstLength;
        
		hr = g_pIDataConvert->DataConvert(
                dwSrcType,
                DBTYPE_STR,
                *pulSrcLength,
                &ulDstLength,
                pSrc,
                pszNewSQL + ulPrefixSize,
                ulDstMaxLength,
                *pdwSrcStatus,
                &dwDstStatus,
                0,	// bPrecision for conversion to DBNUMERIC
				0,	// bScale for conversion to DBNUMERIC
				DBDATACONVERT_DEFAULT);
        
		if (FAILED(hr))
            return hr;  // fatal error

		// Write "field = NULL" if necessary
		if( dwDstStatus == DBSTATUS_S_ISNULL )
		{
			if (bQuotations)
				ulPrefixSize--;  //Compensate adding quotation
			strcpy( pszNewSQL + ulPrefixSize, "NULL" );
			ulDstLength = 4; // strlen("NULL")
		}
		//Add last quotation
		else if (bQuotations)
		{
			*(pszNewSQL + ulPrefixSize + ulDstLength) = '\'';
			ulPrefixSize++; //Compensate adding quotation
		}

		//Copy all chars after parameter
		strcpy(pszNewSQL + ulPrefixSize + ulDstLength, pchEnd + 1);

		//Place new SQL string instead of old one
		if (*ppszSQL != m_pObj->m_pszSQL)
			delete [] *ppszSQL;
		
		*ppszSQL = pszNewSQL;
	
		//Adjust data for new loop
		pchEnd = pszNewSQL + ulPrefixSize + ulDstLength;
		pchBegin = pchEnd;
	}

	return S_OK;

	INTERFACE_METHOD_END();
}

//@cmember Cancel
STDMETHODIMP CImpICommandText::Cancel()
{
	INTERFACE_METHOD_START("ICommandText::Cancel");
	return m_pObj->m_pICommand->Cancel();
	INTERFACE_METHOD_END();
}

//@cmember Execute
STDMETHODIMP CImpICommandText::Execute
				(
				IUnknown*	pUnkOuter,
				REFIID		riid,	
				DBPARAMS*	pParams,
				LONG*		pcRowsAffected, 
				IUnknown**	ppRowset
				)
{
	INTERFACE_METHOD_START("ICommandText::Execute");
	return m_pObj->m_pICommand->Execute(pUnkOuter, riid, pParams, pcRowsAffected, ppRowset);
	INTERFACE_METHOD_END();
}

//@cmember GetDBSession
STDMETHODIMP CImpICommandText::GetDBSession(REFIID riid, IUnknown** ppSession)
{
	INTERFACE_METHOD_START("ICommandText::Execute");
	return m_pObj->m_pICommand->GetDBSession(riid, ppSession);
	INTERFACE_METHOD_END();
}

		
//-----------------------------------------------------------------------------
// CImpICommand::Execute 
//
// @mfunc Executes the command
//
// @rdesc 
//		@flag S_OK		| method succeeds
//		@flag E_FAIL	| method fails
//
STDMETHODIMP CImpICommand::Execute
	(
	IUnknown*	pUnkOuter,		// @parm IN		| outer IUnknown
	REFIID		riid,			// @parm IN		| requested IID for recordset
	DBPARAMS*	pParams,		// @parm IN/OUT | execution parameters
	LONG*		pcRowsAffected, // @parm OUT	| count of returned rows
	IUnknown**	ppRowset		// @parm OUT	| ptr to returned recordset
	)
{
	INTERFACE_METHOD_START("ICommand::Execute");

    CRowset*    pRowset = NULL;
	ULONG		ulRowset;

    // NULL out-params in case of error
    if( ppRowset )
	    *ppRowset = NULL;

	// Check interface
	if (riid != IID_NULL && ppRowset == NULL)
        return E_INVALIDARG;
	
	//Check aggregating
	if (pUnkOuter != NULL && riid != IID_IUnknown)
		return DB_E_NOAGGREGATION;

    assert(m_pObj);


	//Prepare statement
	HRESULT hr = m_pObj->Prepare(false);
	if (hr != S_OK)
		return hr;

	//Substitute parameters
	LPSTR pszSQL;

	hr = SubstituteParameters(&pszSQL, pParams);
	if (hr != S_OK)
	{
		if (pszSQL != m_pObj->m_pszSQL)
			delete [] pszSQL;

		return hr;
	}

    // Gather fetching / scrolling backwards properties
	ULONG ulFetchIdx, ulScrollIdx, ulProvStringIdx;
	assert( m_pObj->m_pUtilProp );
	if( !m_pObj->m_pUtilProp->GetPropIndex(DBPROP_CANFETCHBACKWARDS, &ulFetchIdx ) || 
		!m_pObj->m_pUtilProp->GetPropIndex(DBPROP_CANSCROLLBACKWARDS, &ulScrollIdx ) ||
		!m_pObj->m_pDBSession->m_pCDataSource->m_pUtilProp->GetPropIndex(DBPROP_INIT_PROVIDERSTRING, &ulProvStringIdx ) )
	{
		if (pszSQL != m_pObj->m_pszSQL)
			delete [] pszSQL;

		return E_FAIL;
	}
	// Use backwards support (not cheap for P.SQL 2000 yet)
	bool bBackwardsSupport = 
			m_pObj->m_pUtilProp->m_rgproperties[ulFetchIdx].boolVal ||
			m_pObj->m_pUtilProp->m_rgproperties[ulScrollIdx].boolVal;
	
	
	//Create and initialize data object
	DWORD dwAffectedRows = DB_COUNTUNAVAILABLE;
	CData *pData = NULL;
	WORD wAdvancedInterfaces = ROWSET_NONE;
	bool bCheckMemoryAllocation = true;
	switch( m_pObj->m_pDBSession->m_pCDataSource->m_pUtilProp->InternalSqlSupport() )
	{
		case ISS_MYSQL:
		{
			CMySql* pMySql = new CMySql();
			if( pMySql != NULL )
			{
				hr = pMySql->Init( m_pObj->m_pDBSession->m_pCDataSource, NULL, pszSQL, &dwAffectedRows );
				pData = pMySql;
				wAdvancedInterfaces = ROWSET_FIND | ROWSET_SCROLL | ROWSET_CHANGE;
			}
			break;
		}

		default:
			hr = E_FAIL;
	}
    
	// Free string with substitutions
	if (pszSQL != m_pObj->m_pszSQL)
		delete [] pszSQL;

	// Check memory allocation
	if( bCheckMemoryAllocation && pData == NULL )
		return E_OUTOFMEMORY;

    // Check execution result
	if (FAILED(hr))
    {
        delete pData;
        return hr;
    }

	// Check if it was non-SELECT query
	if (hr == S_FALSE) //Special code for non-SELECT query
	{
		//Return row count
		if (pcRowsAffected != NULL)
			*pcRowsAffected = dwAffectedRows;

		delete pData;
		return S_OK;
	}

	if (riid == GUID_NULL)
	{
		delete pData;
		return S_OK;
	}

    // open and initialize a rowset\cursor object
	// Create Rowset:
	// Get number of first inactive rowset
	hr = m_pObj->m_pDBSession->GetFirstInactiveRowset(&ulRowset);
	if (hr == S_FALSE)
	{
        delete  pData;
		return 	DB_E_OBJECTCREATIONLIMITREACHED;	
	}

	// shortcut for (m_pObj->m_pDBSession->m_pRowsets)[ulRowset]
	pRowset = (m_pObj->m_pDBSession->m_pRowsets)[ulRowset];
	// some cautions
	assert(ulRowset < NUM_SUPPORTED_ROWSETS_PER_SESSION);
	assert(pRowset == NULL); //it's clear yet

	// allocate memory for new rowset
    pRowset = new CRowset(pUnkOuter);
    if (!pRowset)
	{
        delete  pData;
        return E_OUTOFMEMORY;
	}

	// mark rowset # ulRowset as active
	hr = m_pObj->m_pDBSession->SetActiveRowset(ulRowset);
	if (hr == S_FALSE)
	{
        delete  pData;
		return 	E_FAIL;	
	}

	// refresh the number of active sessions
	hr = m_pObj->m_pDBSession->CountActiveRowsets();
	if (hr != S_OK)
	{
        delete  pData;
		return 	E_FAIL;	
	}

	// Initialize new rowset
    if (!pRowset->Init(ulRowset, pData, wAdvancedInterfaces, m_pObj))
	{
        delete  pRowset;
        return  DB_E_NOTABLE;
	}


	//At this point we have handed off the pData pointer to the
	//provider so null it out.
	pData = NULL;


	// Get properties from Command object
	ULONG cProperties;
	DBPROPSET* rgProperties;
	hr = m_pObj->m_pUtilProp->GetProperties(PROPSET_ROWSET, 0, NULL, &cProperties, &rgProperties);
	if (FAILED(hr))
	{
        delete pRowset;
		return hr;
	}
	
	// Set the properties
	hr = pRowset->m_pUtilProp->SetProperties(PROPSET_ROWSET, cProperties, rgProperties);
	g_pIMalloc->Free(rgProperties);
	if (FAILED(hr))
	{
        delete pRowset;
		return hr;
	}
	
	// get requested interface pointer on rowset\cursor
    hr = pRowset->QueryInterface(riid, (void**)ppRowset);
    if (FAILED( hr ))
    {
        delete pRowset;
        return hr;
    }

	hr = m_pObj->FillColumnInfo(pRowset);
    if (FAILED(hr))
    {
        delete pRowset;
        return hr;
    }

	//Inherit all accessors from command
	CExtBuffer* pExtBuffer = m_pObj->m_pIAccessor->m_pExtBufferAccessor;
	assert(pExtBuffer != NULL);
	ULONG ulAccessor, ulFirst, ulLast, ulTemp;
	pExtBuffer->GetFirstLastItemH(ulFirst, ulLast);
	for (ulAccessor = ulFirst; ulAccessor <= ulLast; ulAccessor++)
	{
		//Extract accessor from command
		PACCESSOR pCommandAccessor = *(PACCESSOR*)(*pExtBuffer)[ulAccessor];
		PACCESSOR pAccessor = NULL;

		//If command accessor is NULL so we do not need to create rowset accessor
		if (pCommandAccessor != NULL)
		{
			pAccessor = (ACCESSOR*) new BYTE[sizeof(ACCESSOR) + pCommandAccessor->cBindings * sizeof(DBBINDING)];
			if (pAccessor == NULL)
				return E_OUTOFMEMORY;
		}

		//Add accessor to rowset
		hr = pRowset->m_pExtBufferAccessor->InsertIntoExtBuffer(&pAccessor, ulTemp);
		if (FAILED(hr))
		{
			delete pRowset;
			
			if (pAccessor != NULL)
				delete [] pAccessor;

			return hr;
		}

		// Copy the command accessor bindings into the ACCESSOR.
		if (pCommandAccessor != NULL)
		{
			pAccessor->dwAccessorFlags	= pCommandAccessor->dwAccessorFlags;
			pAccessor->cBindings		= pCommandAccessor->cBindings;
			pAccessor->cRef				= 1;		// Establish Reference count.
			pAccessor->bNullAccessor	= pCommandAccessor->bNullAccessor;
			memcpy( &(pAccessor->rgBindings[0]),
					&(pCommandAccessor->rgBindings[0]),
					pCommandAccessor->cBindings * sizeof(DBBINDING));
		}
	}

	//Assign creator pointer. Used for IRowsetInfo::GetSpecification
    pRowset->m_pCreator = m_pObj;     
	pRowset->m_pCreator->AddRef();

    //Assign session pointer
	pRowset->m_pSession = m_pObj->m_pDBSession;     
	pRowset->m_pSession->AddRef();
    m_pObj->m_pDBSession->m_fRowsetCreated = TRUE;

	m_pObj->SetState(STATE_EXECUTE);
	
	return hr;

	INTERFACE_METHOD_END();
}

//-----------------------------------------------------------------------------
// CImpICommand::GetDBSession 
//
// @mfunc Executes the command
//
// @rdesc 
//		@flag S_OK				| method succeeds
//		@flag E_INVALIDARG		| ppSession == NULL
//		@flag E_NOINTERFACE		| Requested interface is not supported 
//
STDMETHODIMP CImpICommand::GetDBSession
	(
	REFIID		riid,		// @parm IN		| IID of session
	IUnknown**	ppSession	// @parm OUT	| ptr to returned session
	)
{
	INTERFACE_METHOD_START("ICommand::GetDBSession");

	//Handle Aggregated DBSession (if aggregated)
	return m_pObj->m_pDBSession->m_pUnkOuter->QueryInterface(riid, (LPVOID*)ppSession);

	INTERFACE_METHOD_END();
}


//-----------------------------------------------------------------------------
// CImpICommandText::GetCommandText 
//
// @mfunc Retrieve command text
//
// @rdesc 
//		@flag S_OK				| Command text returned
//		@flag E_INVALIDARG		| ppwszCommand was NULL
//		@flag DB_E_NOCOMMAND	| No command text was set
//		@flag E_OUTOFMEMORY		| Out of memory
//
STDMETHODIMP CImpICommandText::GetCommandText
	(
	GUID*		pguidDialect,	// @parm IN  | GUID of syntax
	LPOLESTR*	ppwszCommand	// @parm OUT | ptr to command text
	)
{
	INTERFACE_METHOD_START("ICommand::GetCommandText");

	// Check Function Arguments
	if (ppwszCommand == NULL)
	{
		if (pguidDialect != NULL)
			*pguidDialect = GUID_NULL;

		return E_INVALIDARG;
	}

	*ppwszCommand = NULL;

	if (m_pObj->m_pwszCommand == NULL)
	{
		if (pguidDialect != NULL)
			*pguidDialect = GUID_NULL;

		return DB_E_NOCOMMAND;
	}

	//Allocate memory for result string
	*ppwszCommand = (LPOLESTR)g_pIMalloc->Alloc(g_pIMalloc->GetSize(m_pObj->m_pwszCommand));

	if (*ppwszCommand == NULL)
		return E_OUTOFMEMORY;

	//Copy string
	wcscpy(*ppwszCommand, m_pObj->m_pwszCommand);

	//Check dialect
	if (pguidDialect != NULL)
	{
		if (memcmp(pguidDialect, &DBGUID_DEFAULT, sizeof(GUID)) != 0 &&
			memcmp(pguidDialect, &DBGUID_SQL, sizeof(GUID)) != 0)
		{
			*pguidDialect = DBGUID_DEFAULT;
			return DB_S_DIALECTIGNORED;
		}
	}
	
	return S_OK;

	INTERFACE_METHOD_END();
}

//-----------------------------------------------------------------------------
// CImpICommandText::SetCommandText 
//
// @mfunc Set command text
//
// @rdesc 
//		@flag S_OK | Command text returned
//
STDMETHODIMP CImpICommandText::SetCommandText
	(
	REFGUID		rguidDialect,	// @parm IN  | GUID of syntax
	LPCOLESTR	pwszCommand		// @parm IN  | ptr to command text
	)
{
	INTERFACE_METHOD_START("ICommand::SetCommandText");

	HRESULT hr = m_pObj->Unprepare();
	if (hr != S_OK)
		return hr;

	m_pObj->ClearColumnInfo();

	if (m_pObj->m_pwszCommand != NULL)
	{
		g_pIMalloc->Free(m_pObj->m_pwszCommand);
		m_pObj->m_pwszCommand = NULL;
	}

	//Check if command text is cleared
	if (pwszCommand == NULL || wcscmp(pwszCommand, L"") == 0)
	{
		m_pObj->SetState(STATE_INITIAL);
		return S_OK;
	}

	//Check dialect
	if (memcmp(&rguidDialect, &DBGUID_DEFAULT, sizeof(GUID)) != 0 &&
		memcmp(&rguidDialect, &DBGUID_SQL, sizeof(GUID)) != 0)
	{
		return DB_E_DIALECTNOTSUPPORTED;
	}

	//Allocate memory for result string
	m_pObj->m_pwszCommand = (LPOLESTR)g_pIMalloc->Alloc(sizeof(WCHAR) * (wcslen(pwszCommand) + 1));

	if (m_pObj->m_pwszCommand == NULL)
		return E_OUTOFMEMORY;

	//Copy command text to internal buffer
	wcscpy(m_pObj->m_pwszCommand, pwszCommand);
	m_pObj->SetState(STATE_UNPREPARED);

	return S_OK;

	INTERFACE_METHOD_END();
}



//  ICommandPrepare specific interface methods

//-----------------------------------------------------------------------------
// CImpICommandPrepare::Prepare 
//
// @mfunc Prepare 
//
// @rdesc 
//		@flag S_OK | Command is prepared
//
STDMETHODIMP CImpICommandPrepare::Prepare(ULONG)
{
	INTERFACE_METHOD_START("ICommandPrepare::Prepare");

	HRESULT hr = m_pObj->Prepare(true);
	
	if (hr == S_OK)
		m_pObj->SetState(STATE_PREPARED);
	
	return hr;

	INTERFACE_METHOD_END();
}


//-----------------------------------------------------------------------------
// CImpICommandPrepare::Unprepare 
//
// @mfunc Unprepare 
//
// @rdesc 
//		@flag S_OK | Command is unprepared
//
STDMETHODIMP CImpICommandPrepare::Unprepare()
{
	INTERFACE_METHOD_START("ICommandPrepare::Unprepare");

	HRESULT hr = m_pObj->Unprepare();
	
	if (hr == S_OK)
		m_pObj->SetState(STATE_UNPREPARED);
	
	return hr;

	INTERFACE_METHOD_END();
}




//  ICommandProperties specific interface methods


// ICommandProperties::GetProperties ----------------------------------------------------
//
// @mfunc Returns current settings of all properties in the DBPROPFLAGS_DATASOURCE 
//			property group
//
// @rdesc HRESULT
//      @flag S_OK          | The method succeeded
//      @flag E_INVALIDARG  | pcProperties or prgPropertyInfo was NULL
//      @flag E_OUTOFMEMORY | Out of memory
//
STDMETHODIMP CImpICommandProperties::GetProperties
    (
	    const ULONG			cPropertySets,		//@parm IN | count of restiction guids
		const DBPROPIDSET	rgPropertySets[],	//@parm IN | restriction guids
		ULONG*              pcProperties,		//@parm OUT | count of properties returned
		DBPROPSET**			prgProperties		//@parm OUT | property information returned
    )
{
	INTERFACE_METHOD_START("ICommand::GetProperties");

    assert(m_pObj);
    assert(m_pObj->m_pUtilProp);

	// Check Arguments
	HRESULT hr = m_pObj->m_pUtilProp->GetPropertiesArgChk
										(
										PROPSET_ROWSET,
										cPropertySets, 
										rgPropertySets,
										pcProperties,
										prgProperties
										);
	if (FAILED(hr))
		return hr;

    // Just pass this call on to the utility object that manages our properties
    return m_pObj->m_pUtilProp->GetProperties
										(
										PROPSET_ROWSET,
										cPropertySets, 
										rgPropertySets,
										pcProperties, 
										prgProperties
										);
	INTERFACE_METHOD_END();
}


// CImpICommandProperties::SetProperties  --------------------------------------------
//
// @mfunc Set properties in the DBPROPFLAGS_DATASOURCE property group
//
// @rdesc HRESULT
//      @flag S_OK          | The method succeeded
//      @flag E_INVALIDARG  | cProperties was not equal to 0 and rgProperties was NULL
//
STDMETHODIMP    CImpICommandProperties::SetProperties
    (
	    ULONG		cProperties,
		DBPROPSET	rgProperties[]
	)
{
	INTERFACE_METHOD_START("ICommand::SetProperties");

	assert(m_pObj);
    assert(m_pObj->m_pUtilProp);

	// Quick return if the Count of Properties is 0
	if (cProperties == 0)
		return S_OK;

	// Check Arguments for use by properties
	HRESULT hr = m_pObj->m_pUtilProp->SetPropertiesArgChk(cProperties, rgProperties);
	if (FAILED(hr))
		return hr;

    // just pass this call on to the utility object that manages our properties
    return m_pObj->m_pUtilProp->SetProperties(PROPSET_ROWSET, cProperties, rgProperties);

	INTERFACE_METHOD_END();
}



STDMETHODIMP CImpISupportErrorInfo::InterfaceSupportsErrorInfo(REFIID riid)
{
	INTERFACE_METHOD_START( "ISupportErrorInfo::InterfaceSupportsErrorInfo" );
	
	if (riid == IID_ICommand)
		return S_OK;
	else
		return S_FALSE;

	INTERFACE_METHOD_END();
}


//  ICommandWithParameters specific interface methods


//-----------------------------------------------------------------------------
// CImpICommandWithParameters::GetParameterInfo 
//
// @mfunc  GetParameterInfo
//
// @rdesc 
//		@flag S_OK | Success
//
STDMETHODIMP CImpICommandWithParameters::GetParameterInfo
					(
					ULONG*			pcParams,		//@parm OUT
					DBPARAMINFO**   prgParamInfo,	//@parm OUT
					OLECHAR**       ppNamesBuffer	//@parm OUT
					)
{
	INTERFACE_METHOD_START("ICommandWithParameters::GetParameterInfo");

	//Init out arguments
	*pcParams = 0;

	if (pcParams == NULL || prgParamInfo == NULL)
		return E_INVALIDARG;

	*prgParamInfo = NULL;
	
	if (ppNamesBuffer != NULL)
		*ppNamesBuffer = NULL;

	//We can not derive parameters
	if (m_nCount == 0)
		return DB_E_PARAMUNAVAILABLE;

	//Copy count
	*pcParams = m_nCount;

	//Copy info
	*prgParamInfo = (DBPARAMINFO*) g_pIMalloc->Alloc(m_nCount * sizeof(DBPARAMINFO));
	memcpy(*prgParamInfo, m_pParamInfo, m_nCount * sizeof(DBPARAMINFO));

	//Check if parameters are named
	if (m_pParamInfo[0].pwszName != NULL)
	{
		//Count total number of bytes that is needed for string buffer
		ULONG i;
		WORD cChars;
		DWORD cBytes = 0;

		for (i = 0; i < m_nCount; i++)
		{
			cChars = wcslen(m_pParamInfo[i].pwszName) + 1;
			//Temporary storage for length (old bad style)
			(*prgParamInfo)[i].pwszName = (WCHAR*)cChars;
			//Count bytes
			cBytes += cChars * sizeof(WCHAR);
		}

		//Create string buffer
		*ppNamesBuffer = (WCHAR*) g_pIMalloc->Alloc(cBytes);

		//Fill out the string buffer
		WCHAR* pstrTemp = *ppNamesBuffer;

		for (i = 0; i < m_nCount; i++)
		{
			cChars = (ULONG)(*prgParamInfo)[i].pwszName;
			wcscpy(pstrTemp, m_pParamInfo[i].pwszName);
			(*prgParamInfo)[i].pwszName = pstrTemp;
			pstrTemp += cChars;
		}
	}
	
	return S_OK;

	INTERFACE_METHOD_END();
}


//-----------------------------------------------------------------------------
// CImpICommandWithParameters::MapParameterNames 
//
// @mfunc  MapParameterNames
//
// @rdesc 
//		@flag S_OK | Success
//
STDMETHODIMP CImpICommandWithParameters::MapParameterNames
					(
					ULONG			cParamNames,		//@parm IN
					const OLECHAR*  rgParamNames[],		//@parm IN
					LONG			rgParamOrdinals[]	//@parm OUT
					)
{
	INTERFACE_METHOD_START("ICommandWithParameters::MapParameterNames");

	bool bOneFound = false;
	bool bOneNotFound = false;

	//Check arguments
	if (cParamNames == 0)
		return S_OK;

	if (rgParamNames == NULL || rgParamOrdinals == NULL)
		return E_INVALIDARG;

	for (ULONG i = 0; i < cParamNames; i++)
	{
		bool bFound = false;
		for (ULONG j = 0; j < m_nCount; j++)
		{
			if (0 == _wcsicmp(rgParamNames[i], m_pParamInfo[j].pwszName))
			{
				bFound = true;
				break;
			}
		}

		if (bFound)
		{
			rgParamOrdinals[i] = j + 1;
			bOneFound = true;
		}
		else
		{
			rgParamOrdinals[i] = 0;
			bOneNotFound = true;
		}

	}

	if (bOneFound && !bOneNotFound)
		return S_OK;

	if (bOneFound && bOneNotFound)
		return DB_S_ERRORSOCCURRED;

	return DB_E_ERRORSOCCURRED;

	INTERFACE_METHOD_END();
}


//-----------------------------------------------------------------------------
// CImpICommandWithParameters::SetParameterInfo 
//
// @mfunc  SetParameterInfo
//
// @rdesc 
//		@flag S_OK | Success
//
STDMETHODIMP CImpICommandWithParameters::SetParameterInfo
					(
					ULONG					cParams,			//@parm IN
					const ULONG				rgParamOrdinals[],	//@parm IN
					const DBPARAMBINDINFO   rgParamBindInfo[]	//@parm IN
					)
{
	INTERFACE_METHOD_START("ICommandWithParameters::SetParameterInfo");

	ULONG i;
	DBPARAMINFO* pParamInfo = NULL;
	HRESULT hr				= S_OK;
	ULONG nCount			= cParams;

	//Check arguments
	if (cParams != 0 && rgParamOrdinals == NULL)
		return E_INVALIDARG;

	if (cParams == 0)
		goto Success;

	//Calculate max count of parameters (old + new)
	if (m_pParamInfo != NULL)
		nCount += m_nCount;

	//Create new buffer for parameters info
	pParamInfo = (DBPARAMINFO*) g_pIMalloc->Alloc(nCount * sizeof(DBPARAMINFO));

	//Copy old info (if any)
	if (m_pParamInfo != NULL)
	{
		memcpy(pParamInfo, m_pParamInfo, m_nCount * sizeof(DBPARAMINFO));

		for (i = 0; i < m_nCount; i++)
			pParamInfo[i].pwszName = SysAllocString(m_pParamInfo[i].pwszName);
	}

	//Add new info to buffer
	ULONG j;
	nCount = m_nCount;
	for (i = 0; i < cParams; i++)
	{
		//Check ordinal
		if (rgParamOrdinals[i] == 0)
		{
			hr = E_INVALIDARG;
			goto ErrorExit;
		}

		//Find ordinal (if already set)
		bool bFound = false;
		for (j = 0; j < nCount; j++)
		{
			if (pParamInfo[j].iOrdinal == rgParamOrdinals[i])
			{
				bFound = true;
				break;
			}
		}

		if (!bFound)
		{
			//if not found, use new slot for param info
			j = nCount;
			nCount++;
		}
		else
		{
			//Free old name
			if (pParamInfo[j].pwszName != NULL)
				SysFreeString(pParamInfo[j].pwszName);
			
			//Info is overridden
			hr = DB_S_TYPEINFOOVERRIDDEN; 
		}


		//Set information
		if (rgParamBindInfo[i].pwszName == NULL || *rgParamBindInfo[i].pwszName == 0)
			pParamInfo[j].pwszName  = NULL;
		else
			pParamInfo[j].pwszName	= SysAllocString(rgParamBindInfo[i].pwszName);

		pParamInfo[j].dwFlags		= rgParamBindInfo[i].dwFlags;
		pParamInfo[j].iOrdinal		= rgParamOrdinals[i];
		pParamInfo[j].pTypeInfo		= NULL;
		pParamInfo[j].ulParamSize	= rgParamBindInfo[i].ulParamSize;
		pParamInfo[j].wType			= GetOledbTypeFromName(rgParamBindInfo[i].pwszDataSourceType);
		pParamInfo[j].bPrecision	= rgParamBindInfo[i].bPrecision;
		pParamInfo[j].bScale		= rgParamBindInfo[i].bScale;
	}

	//Check if all infos contain names (or all of them are without names)
	for (i = 1; i < nCount; i++)
	{
		if ((pParamInfo[0].pwszName == NULL && pParamInfo[i].pwszName != NULL) ||
			(pParamInfo[0].pwszName != NULL && pParamInfo[i].pwszName == NULL)) 
		{
			hr = DB_E_BADPARAMETERNAME;
			goto ErrorExit;
		}
	}

Success:
	//delete old infos
	if (m_pParamInfo != NULL)
	{
		for (i = 0; i < m_nCount; i++)
			SysFreeString(m_pParamInfo[i].pwszName);

		g_pIMalloc->Free(m_pParamInfo);
	}

	//set new infos
	m_nCount = nCount;
	m_pParamInfo = pParamInfo;

	return hr;

ErrorExit:
	//delete new infos
	for (i = 0; i < nCount; i++)
		SysFreeString(pParamInfo[i].pwszName);

	g_pIMalloc->Free(pParamInfo);

	return hr;

	INTERFACE_METHOD_END();
}

