/*
 * Copyright (c)  2000
 * SWsoft  company
 *
 * This material is provided "as is", with absolutely no warranty expressed
 * or implied. Any use is at your own risk.
 *
 * Permission to use or copy this software for any purpose is hereby granted 
 * without fee, provided the above notices are retained on all copies.
 * Permission to modify the code and to distribute modified code is granted,
 * provided the above notices are retained, and a notice that the code was
 * modified is included with the above copyright notice.
 *
 */

//--------------------------------------------------------------------
// MySQL OLE DB Provider 
// Functionality: minimum
// Release: 0.1
//
// @doc
//
// @module OPNROWST.CPP | IOpenRowset interface implementation
//

// Includes ------------------------------------------------------------------

#include "hfiles.h"
#include "headers.h"
#include	"mysqlmeta.h"

#include "mysql.h"


#define		SWST_NOINDEX	MT_WORD(-1)

enum EnumIndexStringTypes
{ 
	eERRORINDEXTYPE,
	eUNDEFINED,
	eSTRING,
	eNUMBER,
	eNUMBERASSTRING,
	ePRIMARYKEY
};



// Code ----------------------------------------------------------------------
extern WCHAR INFORMATION_SCHEMA[];

// Get pointer to first symbol of texem in table name (. is delimiter)
static WCHAR* parseTableName( WCHAR* first, WCHAR* last )
{
	if( last == NULL )	// Invalid call
		return NULL;

	if( last <= first )	// no more lexems
		return NULL;

	if( *last == L'\0' )
		last--;
	else if( *(last-1) == L'.' )
		last -= 2;
	else
		return NULL;

	while( last > first && *last != '.' ) last--;

	if( last == first && *last != '.' )
		return last;
	else if( *last == '.' )
		return last + 1;
	else
		return NULL;
}


// Divide table name to parts (catalog, owner, table names)
HRESULT splitToParts( LPWSTR& lpwszTable, WCHAR lpwszCatalog[], BOOL& bIsCatalog, WCHAR lpwszOwner[], BOOL& bIsOwner )
{
	// Init args
	bIsCatalog = bIsOwner = FALSE;

    // Divide name by parts
	WCHAR * lpwszTablePart = parseTableName( lpwszTable, lpwszTable + wcslen( lpwszTable ) );
	WCHAR * lpwszOwnerPart = parseTableName( lpwszTable, lpwszTablePart );
	WCHAR * lpwszCatalogPart = parseTableName( lpwszTable, lpwszOwnerPart);

	// Check if we are provided with table name
	if( lpwszTablePart == NULL || *lpwszTablePart == L'\0' )
		return DB_E_NOTABLE;
	
	// Get Owner name (if present and valid). Owner (Schema) name overrides owner name from properties (if present)
	if( lpwszOwner )
	{
		bIsOwner = TRUE;
		*lpwszOwner = L'\0';
		
		if( lpwszOwnerPart && wcsncmp( lpwszOwnerPart, INFORMATION_SCHEMA, wcslen(INFORMATION_SCHEMA) ) )
		{
			if( lpwszTablePart - lpwszOwnerPart > 1/*delimiter*/ + MAXOWNERNAME )
				return DB_E_BADTABLEID;
			else
				wcscpy0( lpwszOwner, lpwszOwnerPart, lpwszTablePart - lpwszOwnerPart - 1 );
		}
	}
	
	// Get Catalog name (if present and valid). The Catalog name overwrites current catalog
	if( lpwszCatalog )
	{
		bIsCatalog = TRUE;
		*lpwszCatalog = L'\0';

		if( lpwszCatalogPart && lpwszOwnerPart )
		{
			if( lpwszOwnerPart - lpwszCatalogPart > 1/*delimiter*/ + MAXDATASOURCENAME )
				return DB_E_BADTABLEID;
			else
				wcscpy0( lpwszCatalog, lpwszCatalogPart, lpwszOwnerPart - lpwszCatalogPart - 1 );
		}
	}

	// Table name
	lpwszTable = lpwszTablePart;

	// Done!
	return S_OK;
}


HRESULT getIndexStringType( 
	EnumIndexStringTypes& enumIndexType,	
	DBID* pTableID,							DBID*		pIndexID,
	char* szIndexName,						MT_WORD&	wIndexInString,
	WCHAR* const wszOwnerTable,				WCHAR* const wszCatalogTable,
	BOOL bIsCatalogTable,					BOOL		bIsOwnerTable,
	WCHAR* wszOwnerIndex,					WCHAR*		wszCatalogIndex,
	BOOL& bIsCatalogIndex,					BOOL&		bIsOwnerIndex	 )
{
	// No index?
	if( pIndexID == NULL )
		return S_FALSE;

	// Check kind of index name
	// String?
	bool bString = pIndexID->eKind == DBKIND_NAME &&
				   pIndexID->uName.pwszName != NULL &&
				   wcslen(pIndexID->uName.pwszName) != 0;
	// Number?
	bool bNumber = pIndexID->eKind == DBKIND_PROPID;

	if( bString )
	{
		// Split table to catalog + owner + table names
		WCHAR* lpwszIndexPart = pIndexID->uName.pwszName;
		
		HRESULT hr = splitToParts( lpwszIndexPart, wszCatalogIndex, bIsCatalogIndex, wszOwnerIndex, bIsOwnerIndex );
		if( FAILED( hr ) )
			return DB_E_NOINDEX;

		// Catalog and owner settings must comply with table settings if provided
		if( pTableID != NULL )
		{
			// Check catalogs
			if( bIsCatalogTable && bIsCatalogIndex && 
				wcscmp(wszCatalogTable, wszCatalogIndex) )
					return DB_E_NOINDEX;
			
			// Check owners
			if( bIsOwnerTable && bIsOwnerIndex && 
				wcscmp(wszOwnerTable, wszOwnerIndex) )
					return DB_E_NOINDEX;
		}

		// Determine index type
		if( !wcsicmp( lpwszIndexPart, L"Primary key" ) )
			enumIndexType = ePRIMARYKEY;
		else if( lpwszIndexPart[0] == L'#' )
		{
			enumIndexType = eNUMBERASSTRING;
			wIndexInString = _wtoi( lpwszIndexPart + 1 );
		}
		else
			enumIndexType = eSTRING;

		// Convert index name
		if( !W2A( lpwszIndexPart, szIndexName, _MAX_FNAME - 1 ) )
			return DB_E_NOINDEX;

		// Someone may pass something like "IndexName (#N)"
		char* lpszSpace = strchr( szIndexName, ' ' );
		if( lpszSpace != NULL )
			*lpszSpace = '\0';
	}
	else if( bNumber )
	{
		// Consumer must provide table name with such index
		if( pTableID == NULL )
			return DB_E_NOINDEX;

		// Determine index type
		enumIndexType = eNUMBER;
	}
	else // !bString && !bNumber
		return DB_E_NOINDEX;

	return (enumIndexType != eUNDEFINED) ? S_OK : S_FALSE;
}


HRESULT getIndexNumber( 
	EnumIndexStringTypes enumIndexType, CSwstMeta* const pSwstMeta,
	DBID* const pTableID,				DBID* const pIndexID,
	MT_WORD&	wIndexNumber,			MT_WORD	wIndexInString,
    char* const szTableName,			char* const szIndexName )
{
	// Both index and table are provided
	if( pTableID != NULL && pIndexID != NULL )
	{
		// Get table description
		SWSTFILE* pSwstFile = pSwstMeta->FindFile( szTableName );
		if( pSwstFile == NULL )
			return DB_E_NOTABLE;

		// Index must belong to the table
		for( int i = 0; i < pSwstFile->m_wIndexes && wIndexNumber == SWST_NOINDEX; i++ )
		{
			switch( enumIndexType )
			{
			// Index passed by number
			case eNUMBER:
				if( pSwstFile->m_pIndexes[ i ]->m_wNumber == pIndexID->uName.ulPropid )
					wIndexNumber = pSwstFile->m_pIndexes[ i ]->m_wNumber;
				break; // switch
			
			// Index passed by name
			case eSTRING:
				if( pSwstFile->m_pIndexes[ i ]->m_pSwstFieldName &&
					!stricmp( szIndexName, pSwstFile->m_pIndexes[ i ]->m_pSwstFieldName->m_szName ) )
						wIndexNumber = pSwstFile->m_pIndexes[ i ]->m_wNumber;
				else if( pSwstFile->m_pIndexes[ i ]->m_pSwstRelateName &&
					!stricmp( szIndexName, pSwstFile->m_pIndexes[ i ]->m_pSwstRelateName->m_szName ) )
						wIndexNumber = pSwstFile->m_pIndexes[ i ]->m_wNumber;
				break;

			// Index passed by number saved in string
			case eNUMBERASSTRING:
				if( pSwstFile->m_pIndexes[ i ]->m_wNumber == wIndexInString )
					wIndexNumber = pSwstFile->m_pIndexes[ i ]->m_wNumber;
				break; // switch

			// Primary key is queried
			case ePRIMARYKEY:
				if( pSwstFile->m_pIndexes[ i ]->IsPrimaryKey() )
					wIndexNumber = pSwstFile->m_pIndexes[ i ]->m_wNumber;
				break; // switch
			}
		}

		// No matches
		if( wIndexNumber == SWST_NOINDEX )
			return DB_E_NOINDEX;
	}
	// No table is provided
	else if( pTableID == NULL && pIndexID != NULL )
	{
		if( enumIndexType != eSTRING )
			return DB_E_NOINDEX; // Non-unique one!
		
		// In this case, we have named index (see above)
		for( int i = 0; i < pSwstMeta->m_siIndexes && wIndexNumber == SWST_NOINDEX; i++ )
		{
			if( pSwstMeta->m_pIndexes[ i ].m_pSwstFieldName &&
				!stricmp( szIndexName, pSwstMeta->m_pIndexes[ i ].m_pSwstFieldName->m_szName ) )
			{
				strcpy0( szTableName, pSwstMeta->m_pIndexes[ i ].m_pSwstField->m_pSwstFile->m_szName, _MAX_FNAME-1 );
				wIndexNumber = pSwstMeta->m_pIndexes[ i ].m_wNumber;
			}
			else if( pSwstMeta->m_pIndexes[ i ].m_pSwstRelateName &&
				!stricmp( szIndexName, pSwstMeta->m_pIndexes[ i ].m_pSwstRelateName->m_szName ) )
			{
				strcpy0( szTableName, pSwstMeta->m_pIndexes[ i ].m_pSwstField->m_pSwstFile->m_szName, _MAX_FNAME-1 );
				wIndexNumber = pSwstMeta->m_pIndexes[ i ].m_wNumber;
			}
		}

		// No matches
		if( wIndexNumber == SWST_NOINDEX )
			return DB_E_NOINDEX;
	}
	
	return S_OK;
}

// CImpIOpenRowset::OpenRowset ------------------------------------------------
//
// @mfunc Opens and returns a rowset that includes all rows from a single base table
//
// @rdesc HRESULT
//      @flag S_OK                  | The method succeeded
//      @flag DB_S_ASYNCHRONOUS     | The method has initiated asynchronous creation of the rowset.  
//      @flag DB_S_ERRORSOCCURRED   | The rowset was opened but one or more properties were not set.   
//      @flag DB_S_STOPLIMITREACHED | A resource limit has been reached. 
//      @flag E_INVALIDARG          | pTableID was NULL
//      @flag E_FAIL                | Provider-specific error
//      @flag DB_E_NOTABLE          | Specified table does not exist in current Data
//                                  | Data Source object
//      @flag DB_E_ERRORSOCCURRED   | One or more required properties were not set. 
//      @flag DB_E_ABORTLIMITREACHED| A resource limit has been reached
//      @flag DB_E_NOAGGREGATION	| pUnkOuter was not a null pointer and the rowset being created 
//									| does not support aggregation.
//      @flag DB_E_NOINDEX			| The specified index does not exist in the current data source 
//									| or did not apply to the specified table
//      @flag DB_E_NOTABLE			| The specified table does not exist in the current data source
//      @flag DB_E_PARAMNOTOPTIONAL	| 
//      @flag DB_SEC_E_PERMISSIONDENIED| 
//      @flag DB_E_OBJECTOPEN		| 
//      @flag E_OUTOFMEMORY         | Out of memory
//      @flag E_NOINTERFACE         | The requested interface was not available
//

STDMETHODIMP CImpIOpenRowset::OpenRowset
    (
    IUnknown*                   pUnkOuter,          //@parm IN    | Controlling unknown, if any
    DBID*                       pTableID,           //@parm IN    | table to open
	DBID*						pIndexID,			//@parm IN	  | DBID of the index
    REFIID                      riid,               //@parm IN    | interface to return
    ULONG                       cProperties,        //@parm IN    | count of properties
    DBPROPSET					rgProperties[],     //@parm INOUT | array of property values
    IUnknown**                  ppRowset            //@parm OUT   | where to return interface
    )
{
	TRACE2("IOpenRowset::OpenRowset: %S", pTableID ? pTableID->uName.pwszName : L"(by index)" );

    CRowset*    pRowset = NULL;
    HRESULT     hr;				
	HRESULT		hrProp	= S_OK;	

	ULONG		ulRowset;

	/* Start of block of variables */
	MT_WORD	wIndexNumber = SWST_NOINDEX;
	MT_WORD	wIndexInString = SWST_NOINDEX;
	WCHAR wszOwnerTable[ MAXOWNERNAME + 1 ]; *wszOwnerTable = 0;
	WCHAR wszCatalogTable[ MAXDATASOURCENAME + 1 ]; *wszCatalogTable = 0;
    char szTableName[_MAX_FNAME]; *szTableName = 0;
	BOOL bIsCatalogTable = FALSE, bIsOwnerTable = FALSE;
	WCHAR wszOwnerIndex[ MAXOWNERNAME + 1 ]; *wszOwnerIndex = 0;
	WCHAR wszCatalogIndex[ MAXDATASOURCENAME + 1 ]; *wszCatalogIndex = 0;
    char szIndexName[_MAX_FNAME]; *szIndexName = 0;
	BOOL bIsCatalogIndex = FALSE, bIsOwnerIndex = FALSE;
	WCHAR *lpwszTablePart = NULL;
	EnumIndexStringTypes enumIndexType = eUNDEFINED;
	/* End of block of variables */

	// NULL out-params in case of error
    if( ppRowset )
	    *ppRowset = NULL;

	//Check arguments
	if (pTableID == NULL && pIndexID == NULL)
		return E_INVALIDARG;

	if (cProperties != 0 && rgProperties == NULL)
		return E_INVALIDARG;

	// If the eKind is not known to use, basically it
	// means we have no table identifier
	if (pTableID != NULL )
	{
		// String?
		bool bString = pTableID->eKind == DBKIND_NAME &&
					   pTableID->uName.pwszName != NULL &&
					   wcslen(pTableID->uName.pwszName) != 0;
		if( !bString )
			return DB_E_NOTABLE;

		// Split table to catalog + owner + table names
		lpwszTablePart = pTableID->uName.pwszName;
		
		hr = splitToParts( lpwszTablePart, wszCatalogTable, bIsCatalogTable, wszOwnerTable, bIsOwnerTable );
		if( FAILED( hr ) )
			return hr;

		// Convert table name
		if( !W2Asz( lpwszTablePart, szTableName ) )
			return DB_E_NOTABLE;
	}

	// Get index type
	hr = getIndexStringType( 
			enumIndexType,	
			pTableID,		pIndexID,
			szIndexName,	wIndexInString,
			wszCatalogTable,wszOwnerTable,	
			bIsCatalogTable,bIsOwnerTable,
			wszCatalogIndex,wszOwnerIndex,	
			bIsCatalogIndex,bIsOwnerIndex );
	if( FAILED( hr ) )
		return hr;

	// Check interface
	if (riid == IID_NULL)
        return E_NOINTERFACE;
	
	//Check aggregating
	if (pUnkOuter != NULL && riid != IID_IUnknown)
		return DB_E_NOAGGREGATION;

	assert( m_pObj->m_pUtilProp );

	// Check Arguments for use by properties
	hr = m_pObj->m_pUtilProp->SetPropertiesArgChk(cProperties, rgProperties);
	if( FAILED(hr) )
		return hr;

    assert( m_pObj );

	// set the properties
	if ( cProperties )
		hr = m_pObj->m_pUtilProp->SetProperties(PROPSET_ROWSET, cProperties, rgProperties);
	
	if( (hr == DB_E_ERRORSOCCURRED) || 
		(hr == DB_S_ERRORSOCCURRED) )
	{
		// If all the properties set were SETIFCHEAP then we can set 
		// our status to DB_S_ERRORSOCCURRED and continue.
		for(ULONG ul=0;ul<cProperties; ul++)
		{
			for(ULONG ul2=0;ul2<rgProperties[ul].cProperties; ul2++)
			{
				// Check for a required property that failed, if found, we must return
				// DB_E_ERRORSOCCURRED
				if( (rgProperties[ul].rgProperties[ul2].dwStatus != DBPROPSTATUS_OK) &&
					(rgProperties[ul].rgProperties[ul2].dwOptions != DBPROPOPTIONS_SETIFCHEAP) )
					return DB_E_ERRORSOCCURRED;
			}
		}
		hrProp = DB_S_ERRORSOCCURRED;
	}

	if( FAILED(hr) && (hrProp==S_OK) )
		return hr;

	// if ppRowset NULL return
	if ( !ppRowset )
		return S_OK;
    
	//Get meta data: 1) Get current catalogue; 2) Get meta with current catalogue
	ULONG dsn;
	CSwstMetaHolder *pSwstMetaHolder = NULL;
	CSwstMeta *pSwstMeta;

	// 1) Get current catalogue if it is not in table name
	LPOLESTR lpwszCatalog;
	if( bIsCatalogTable )
		lpwszCatalog = wszCatalogTable;
	else if( bIsCatalogIndex )
		lpwszCatalog = wszCatalogIndex;
	else if( !m_pObj->m_pCDataSource->m_pUtilProp->GetPropIndex(DBPROP_CURRENTCATALOG, &dsn ) )
		return E_FAIL;
	else
		lpwszCatalog = m_pObj->m_pCDataSource->m_pUtilProp->m_rgproperties[dsn].pwstrVal;

	// 2) Get owner name if it is not in table name
	LPOLESTR lpwszOwner;
	if( bIsOwnerTable )
		lpwszOwner = wszOwnerTable;
	else if( bIsOwnerIndex )
		lpwszOwner = wszOwnerIndex;
	else if( !m_pObj->m_pCDataSource->m_pUtilProp->GetPropIndex(DBPROP_SWSTTABLEOWNER, &dsn ) )
		return E_FAIL;
	else
		lpwszOwner = m_pObj->m_pCDataSource->m_pUtilProp->m_rgproperties[dsn].pwstrVal;
	
	// 3) Load catalog+schema information if not loaded
	hr = m_pObj->m_pCDataSource->CheckAndTrackOneDatabase( lpwszCatalog, lpwszOwner );
    if (FAILED(hr))
        return hr;
	
	// 4a) Get MetaHolder
	hr = m_pObj->m_pCDataSource->GetSwstMeta(&pSwstMetaHolder);
    if (FAILED(hr))
        return hr;
	
	// 4b) Get meta with current catalogue and 
	// S_OK means that [out] pSwstMeta != NULL
	if( pSwstMetaHolder->At( lpwszCatalog, lpwszOwner, &pSwstMeta ) != S_OK )
		return E_FAIL;

	// Get index number (wIndexNumber)
	hr = getIndexNumber( 
		enumIndexType, pSwstMeta,
		pTableID,	   pIndexID,
		wIndexNumber,  wIndexInString,
		szTableName,   szIndexName );
	if( FAILED( hr ) )
		return hr;

	// open and initialize a file object
	CData* pData;
	//if( !m_pObj->m_pCDataSource->m_pUtilProp->MySqlMode() )
	switch( m_pObj->m_pCDataSource->m_pUtilProp->InternalSqlSupport() )
	{
	case ISS_WRONG:
		return E_UNEXPECTED;

	default:
		{
		}

	case ISS_MYSQL:
		{
		// Check that the table exists / should exist
		SWSTFILE* pSwstFile = pSwstMeta->FindFile( szTableName );
		if( pSwstFile == NULL )
			return DB_E_NOTABLE;

		// Generate simple query
		const char szSelectFrom[] = "select * from ";
		const char szOrderBy[] = " order by ", szComma[] = ", ";
		const char szDesc[] = " desc", szAsc[] = " asc";

		// 1) Calculate necessary length
		int len = 1 + MAXSTR(szSelectFrom) + strlen(szTableName);
		if( wIndexNumber != SWST_NOINDEX )
		{
			const char *pszAppend = szOrderBy;
			for( int i = 0; i < pSwstFile->m_wIndexes; i++ )
			{
				if( pSwstFile->m_pIndexes[ i ]->m_wNumber == wIndexNumber )
				{
					len += (pszAppend == szOrderBy) ? MAXSTR(szOrderBy) : MAXSTR(szComma);
					len += strlen( pSwstFile->m_pIndexes[ i ]->m_pSwstField->m_szName );
					len += pSwstFile->m_pIndexes[ i ]->IsDescending() ? MAXSTR(szDesc) : MAXSTR(szAsc);
					pszAppend = szComma;
				}
				else if( pSwstFile->m_pIndexes[ i ]->m_wNumber > wIndexNumber )
					break;
			}
		}

		// 2) Allocate memory
		char *pszQuery = new char[ len ];
		if( !pszQuery )
			return E_OUTOFMEMORY;

		// 3) Compose query
		strcpy( pszQuery, szSelectFrom ); 
		strcat( pszQuery, szTableName ); 
		
		if( wIndexNumber != SWST_NOINDEX )
		{
			const char *pszAppend = szOrderBy;
			for( int i = 0; i < pSwstFile->m_wIndexes; i++ )
			{
				if( pSwstFile->m_pIndexes[ i ]->m_wNumber == wIndexNumber )
				{
					strcat( pszQuery, pszAppend );
					strcat( pszQuery, pSwstFile->m_pIndexes[ i ]->m_pSwstField->m_szName );
					strcat( pszQuery, pSwstFile->m_pIndexes[ i ]->IsDescending() ? szDesc : szAsc );
					pszAppend = szComma;
				}
				else if( pSwstFile->m_pIndexes[ i ]->m_wNumber > wIndexNumber )
					break;
			}
		}

		// Our sweety object
		CMySql *pMySql = new CMySql();
		if (!pMySql)
			return E_OUTOFMEMORY;

		//Initialize data object
		DWORD dummy;
		hr = pMySql->Init(m_pObj->m_pCDataSource, lpwszCatalog, pszQuery, &dummy );
		
		delete pszQuery;
		
		if (FAILED(hr))
		{
			delete pMySql;
			return hr;
		}

		pData = pMySql;

		break;
		}
	}

    // open and initialize a rowset\cursor object
	// Create Rowset:
	// Get number of first inactive rowset
	hr = m_pObj->GetFirstInactiveRowset(&ulRowset);
	if (hr == S_FALSE)
	{
        delete  pData;
		return DB_E_OBJECTCREATIONLIMITREACHED;	
	}

	// shortcut for (m_pObj->m_pRowsets)[ulRowset]
	pRowset = (m_pObj->m_pRowsets)[ulRowset];
	// some cautions
	assert(ulRowset < NUM_SUPPORTED_ROWSETS_PER_SESSION);
	assert(pRowset == NULL); //it's clear yet

	// allocate memory for new rowset
    pRowset = new CRowset( pUnkOuter );
    if (!pRowset)
	{
        delete  pData;
		return E_OUTOFMEMORY;
	}

	// mark rowset # ulRowset as active
	hr = m_pObj->SetActiveRowset(ulRowset);
	if (hr == S_FALSE)
	{
        delete  pData;
		return 	E_FAIL;	
	}

	// refresh the number of active sessions
	hr = m_pObj->CountActiveRowsets();
	if (hr != S_OK)
	{
        delete  pData;
		return 	E_FAIL;	
	}

	// Initialize session here: needed to set up properties
	pRowset->m_pSession = m_pObj;     // moved from below
	pRowset->m_pSession->AddRef();	  // moved from below

	// Initialize new rowset
    if (!pRowset->Init(ulRowset, pData, ROWSET_ALL))
	{
        delete  pData;
        delete  pRowset;
		return  DB_E_NOTABLE;
	}

	// At this point we have handed off the pData pointer to the provider. 
	// Do not delete it below!

    // get requested interface pointer on rowset\cursor
    hr = pRowset->QueryInterface( riid, (void **) ppRowset );
    if (FAILED( hr ))
    {
        delete pRowset;
		return hr;
    }

	//Assign creator pointer. Used for IRowsetInfo::GetSpecification
    pRowset->m_pCreator = m_pObj;     
	pRowset->m_pCreator->AddRef();

    //Assign session pointer
//	pRowset->m_pSession = m_pObj;     // moved above
//	pRowset->m_pSession->AddRef();	  // moved above	
    m_pObj->m_fRowsetCreated = TRUE;

	TRACE3( "  Passed with 0x%x, 0x%x", hr, hrProp );
	
	return (hr == S_OK) ? hrProp : hr;
}
