/*
 * Mover.java	98/1/05
 * 
 * Copyright (c) 1998 Ichiro Satoh All Rights Reserved.
 * 
 * Permission to use, copy, modify, and distribute this 
 * software and its documentation for NON-COMMERCIAL purposes
 * and without fee is hereby granted provided that this 
 * copyright notice appears in all copies. 
 */

import agentspace.*;
import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;
import java.net.*;

public class Mover extends Agent {
  private TextArea editArea;
  private Panel statusPanel;
  private TextField statusField;
  private TextField transferField;
  private String fileName = null;
  private String buffer = null;
  private CheckboxMenuItem fontItems[];
  private CheckboxMenuItem styleItems[];
  private CheckboxMenuItem sizeItems[];

  private String clipboard;
  private MenuBar menuBar = new MenuBar();
  private Menu fileMenu = new Menu("File");
  private Menu editMenu = new Menu("Edit");
  private Menu helpMenu = new Menu("Help");
  private int width  = 600;
  private int height = 400;

  public Mover() {}
  public void init() {}
  public void create() {
    initWindow();
  }
  public void arrive() {
    editArea.setText(buffer);
    setMenuBar(menuBar);
    pack();
    resize(width, height);
    show();
  }
  public void dispatch(URL url) {
    width  = size().width; 
    height = size().height;
    buffer = editArea.getText();
    dispose();
  }
  public void resume() {
    editArea.setText(buffer);
    setMenuBar(menuBar);
    pack();
    resize(width, height);
    show();
  }
  public void suspend() {
    buffer = editArea.getText();
    dispose();
  }
  public void destroy() {
    dispose();
  }
  public void duplicate(){
    buffer = editArea.getText();
    dispose();
  }
  public void child(AgentIdentifier paid) {
    editArea.setText(buffer);
    setMenuBar(menuBar);
    pack();
    resize(width, height);
    show();
  }
  public void parent(AgentIdentifier caid) {
    editArea.setText(buffer);
    setMenuBar(menuBar);
    pack();
    resize(width, height);
    show();
  }
  public boolean handleEvent(Event evt) {
    if (evt.id == Event.WINDOW_DESTROY) {
      dispose();
      AgentContext ac = getAgentContext();
      ac.send("destroy");
    }
    return super.handleEvent(evt);
  }
  
  private void initWindow() {
    setTitle("TinyEditor");
    setBackground(Color.lightGray);
    fileMenu.add("Open");
    fileMenu.add("Save");
    fileMenu.add("Destroy");
    menuBar.add(fileMenu);

    editMenu.add("Copy");
    editMenu.add("Paste");
    editMenu.add("Cut");
    editMenu.add("Select All");
    menuBar.add(editMenu);

    helpMenu.add("Version");
    menuBar.add(helpMenu);
    setMenuBar(menuBar);

    editArea = new TextArea(25, 80);
    editArea.setText(buffer);

    Font f = new Font("Dialog", Font.PLAIN, 14);
    editArea.setFont(f);


    transferField = new TextField(80);
    statusField.setText(
	"Tiny Java Editor (ver. 0.1) Copyright (C) 1997 Ichiro Satoh.");
    statusField.setEditable(false);

    statusPanel = new Panel();
    statusPanel.add(new Label("Move To:"));
    transferField = new TextField(32);
    transferField.setText("133.65.66.189");
    statusPanel.add(transferField);
    add("North",statusPanel);
    add("South",statusField);
    add("Center",editArea);
    resize(width, height);
    show();
  }
	
  private String getFileName(boolean isOpen) {
    FileDialog dialog = 
      new FileDialog(this,
		     isOpen ? "Open File" : "Save File",
		     isOpen ? FileDialog.LOAD : FileDialog.SAVE);
    dialog.show();
    statusField.setText("File: "+
			dialog.getDirectory() + "/" + dialog.getFile());
    if (dialog.getDirectory() != null && dialog.getFile() != null) {
      return dialog.getDirectory() + dialog.getFile();
    }
    else {
      return null;
    }
  }
	
  private void loadFile(String fileName) {
    statusField.setText("Open File");
    FileReader reader;
    try { 
      reader = new FileReader(fileName); 
    }
    catch(Exception e) { 
      statusField.setText("Run-Time Error: "+e.toString()); 
      return; 
    }
    statusField.setText(reader.getEncoding().toString());
    StringBuffer text = new StringBuffer();
    try {
      char [] buf = new char[64];
      while (reader.ready()) {
	int len = reader.read(buf, 0, 64);
	if (len == -1) {
	  break;
	}
	text.append(buf, 0, len);
      }
      reader.close();
      setTitle(fileName);
      statusField.setText("Successful: File Open");
    }
    catch(Exception e) { 
      statusField.setText("Run-Time Error: "+e.toString()); 
    }
    editArea.setText(text.toString());
  }
	
  private void saveFile(String fileName) {
    statusField.setText("Save File");
    FileWriter writer;
    try { 
      writer = new FileWriter(fileName); 
    }
    catch(Exception e) { 
      statusField.setText("Run-Time Error: "+e.toString()); 
      return; 
    }
    statusField.setText(writer.getEncoding().toString());
    String text = editArea.getText();
    try	{
      writer.write(text, 0, text.length());
      writer.close();
      statusField.setText("Successful: File Save");
    }
    catch(Exception e) { 
      statusField.setText("Run-Time Error: "+e.toString()); }
  }

  public boolean action(Event evt, Object what) { 
    statusField.setText(what.toString());
    if ("Open".equals(what)) {
      String newname = getFileName(true);
      if (newname != null) {
	loadFile(newname);
	fileName = newname;
      }
      return true;
    }      
    else if ("Save".equals(what)) {
      if (fileName != null) {
	saveFile(fileName);
      }
      return true;
    }
    else if ("Save As".equals(what)) {
      String newname = getFileName(false);
      if (newname != null) {
	saveFile(newname);
	fileName = newname;
      }
      return true;
    }
    else if ("Exit".equals(what)) {
      this.dispose();
      AgentContext ac = getAgentContext();
      ac.send("destroy");
      return true;
    }
    else if ("Version".equals(what)) {
      statusField.setText(
	"Tiny Java Editor (ver. 0.1) Copyright (C) 1997 Ichiro Satoh.");
      return true;
    }
    else if ("Copy".equals(what)) {
      clipboard = editArea.getSelectedText();
      statusField.setText("Copy");
      return true;
    }
    else if ("Paste".equals(what)) {
      int start = editArea.getSelectionStart();
      int end = editArea.getSelectionEnd();
      if (start == end) {
        editArea.insertText(clipboard, start);
      }
      else {
        editArea.replaceText(clipboard, start, end);
      }
      statusField.setText("Paste");
      return true;
    }
    else if ("Cut".equals(what)) {
      clipboard = editArea.getSelectedText();
      editArea.replaceText("", editArea.getSelectionStart(), 
			   editArea.getSelectionEnd());
      return true;
    }
    else if ("Select All".equals(what)) {
      editArea.selectAll();
      return true;
    }
    else if ("Compile".equals(what)) {
      try {
	saveFile(fileName);
	Process ps  = Runtime.getRuntime().exec("javac " + fileName);
/*
	DataInputStream commandResult =
	  new DataInputStream(new BufferedInputStream(ps.getInputStream()));
	String result = "";
	String s = "";
	while ((s = commandResult.readLine()) != null) {
	  result = result + s;
	  System.out.println("Compile: "+s);
	}
*/
      }
      catch (Exception e) {
	statusField.setText("Unkonwn Error: Java Compilation");
      }
      return true;
    }

    return false;
  }
  
  public static void main(String args[]) {
    Mover ed = new Mover();
    ed.init();
    ed.create();
  }
}

/*
class ReplaceDialog extends Dialog {
  private TextField from, to;
  private TextArea ta;

  public ReplaceDialog(Frame parentFrame, TextArea ta) {
    super(parentFrame, "Replace", false);
    this.ta = ta;
    initWindow();
  }

  private void initWindow() {
    setBackground(Color.lightGray);
    Panel p = new Panel();
    p.setLayout(new FlowLayout());
    p.add(new Button("Replace"));
    from = new TextField(10);
    p.add(from);
    p.add(new Label("with"));
    to = new TextField(10);
    p.add(to);
    add("Center", p);
    pack();
    show();
  }

  public boolean handleEvent(Event evt) {
    switch(evt.id) {
      case Event.ACTION_EVENT:
      {
	if("Replace".equals(evt.arg)) {
	  String f = from.getText();
	  int n = ta.getText().indexOf(f);
	  if (n >= 0 && f.length() > 0) {
	    ta.replaceText(to.getText(), n, n + f.length());
	  }
	  dispose ();
	  return true;
	}
      }
    }
    return false;
  }
}
*/

