/*
 * Copyright (C) 1997, 1998 Ichiro Satoh
 * All Rights Reserved.
 * written by Ichiro Satoh
 *
 * This program was developed by Ichiro Satoh at Ochanomizu University 
 * as a part of a mobile object system, titled "AgentSpace". All the 
 * programs of the system is developed by Ichiro Satoh and all rights 
 * are reserved by Ichiro Satoh. 
 *
 * Permission to use, copy, modify and distribute this software and
 * its documentation is hereby granted, provided that both the
 * copyright notice and this permission notice appear in all copies of
 * the software, derivative works or modified versions, and any
 * portions thereof, and that both notices appear in supporting 
 * documentation.
 *
 * FREE USE OF THIS SOFTWARE IS ALLOWED IN ITS "AS IS" CONDITION.
 * BUT WITHOUT WARRANTY. THE AUTHORS DISCLAIM ANY LIABILITY OF ANY 
 * KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS 
 * SOFTWARE.
 */

package agentspace;

import java.io.*;
import java.net.*;
import java.util.*;
import java.lang.*;
import java.lang.reflect.*;

/**
 * The <code>AgentContext</code> class is the execution context for 
 * agents and provides various services which are necessary to running 
 * agents. 
 *
 * @author  Ichiro Satoh
 * @version 0.1 9/25/97
 * @see     agentspace.Agent
 */
public class AgentContext {
  private AgentRuntime theRuntime;
  private AgentList theList;
  private AgentInfo theInfo;
  private AgentIdentifier theIdentifier;

  public AgentContext(AgentList alist, AgentInfo ainfo) {
    theList = alist;
    theInfo = ainfo;
    theRuntime = theList.getAgentRuntime(getIdentifier());
  }

  private AgentRuntime getAgentRuntime(AgentIdentifier aid) {
    AgentInfo ainfo = theList.getAgentInfo(aid);
    return ainfo.getAgentRuntime();
  }

  /**
   * Gets the agent identifier.
   * @return the identifier of the agent. 
   */
  public AgentIdentifier getIdentifier() {
    return theInfo.getIdentifier();
  }

  /**
   * Sets the agent name. 
   */
  public void setAgentName(String name) {
    theInfo.setName(name);
  }

  /**
   * Gets the agent name. 
   * @return the name of the agent. 
   */
  public String getAgentName() {
    return theInfo.getName();
  }

  /**
   * Gets the name of an agent specified by aid. 
   * @return the name of the specifed agent. 
   */
  public String getAgentName(AgentIdentifier aid) {
    AgentInfo ainfo = theList.getAgentInfo(aid);
    return ainfo.getName();
  }

  /**
   * Gets the url in which the agent is currently executing. 
   * @return the url of the current execution host 
   */
  public URL getCurrentHost() {
    URL url = null;
    try {
      url = new URL("http://"+AgentServer.theHost 
			       + ":" + AgentServer.theReceivePort);
    }
    catch (MalformedURLException e) {}
    return url;
  }

  /**
   * Gets the url in which the agent was executing previously. 
   * @return the url of the previous execution host 
   */
  public URL getPreviousHost() {
    return theInfo.getPreviousHost();
  }

  /**
   * Returns the sibling agents in the current server. 
   * @return an enumeration of the identifiers of the agents. 
   */
  public Enumeration getAgents() {
    Vector list = new Vector();
    for (int i = 0 ; i < theList.agentInfoSize() ; i++ ) {    
      list.addElement((theList.getAgentInfo(i)).getIdentifier());
    }
    return list.elements();
  }

  public Enumeration getAgents(String name) {
    Vector list = new Vector();
    for (int i = 0 ; i < theList.agentInfoSize() ; i++ ) {    
      AgentInfo ainfo = theList.getAgentInfo(i);
      if (name.equals(ainfo.getName())) {
	list.addElement(ainfo.getIdentifier());
      }
    }
    return list.elements();
  }

  /**
   * Sends a one-way message <code>Message</code> asynchronously. 
   * The sender can continue its following program even if the receiver 
   * does not handling the message yet. 
   * @param  msg a message to send synchronously. 
   * @exception NoSuchAgentException if the receiver agent specified in msg 
   * does not exist. 
   * @exception NoSuchMethodException if the name or 
   * argument is invalid in msg. 
   * @see  agentspace.AgentContext#call(Message msg) 
   * @see  agentspace.AgentContext#future(Message msg) 
   */
  public Object send(Message msg) throws 
  NoSuchMethodException, 
  NoSuchAgentException {
    Future futureObj = new Future();
    msg.setFuture(futureObj);
    msg.setSender(getIdentifier());
    AgentIdentifier aid = msg.getTarget();
    getAgentRuntime(aid).acceptRequest(msg);
    Object obj = futureObj.getReply();
    Exception ex = msg.getException();
    if (ex != null) {
      if (ex instanceof NoSuchAgentException) {
	throw new NoSuchAgentException();
      }
    }
    return obj;
  }

  /**
   * Sends a message <code>Message</code> synchronously. 
   * The sender has to be blocked until the receiver 
   * finishs handling the message. 
   * @param  msg a message to send synchronously. 
   * @return the result object of the handling in the receiver agent. 
   * @exception NoSuchAgentException if the receiver agent specified in msg 
   * is invalid. 
   * @exception NoSuchMethodException if the name or 
   * argument is invalid in msg. 
   * @exception TimeoutException if the handling goes timeout. 
   * @exception InvocationException if a runtime error in the handling. 
   * @see  agentspace.AgentContext#send(Message msg) 
   * @see  agentspace.AgentContext#future(Message msg) 
   */
  public Object call(Message msg) throws 
  NoSuchMethodException, 
  NoSuchAgentException, 
  TimeoutException, 
  InvocationException {
    Future futureObj = new Future();
    msg.setFuture(futureObj);
    msg.setSender(getIdentifier());
    AgentIdentifier aid = msg.getTarget();
    getAgentRuntime(aid).acceptRequest(msg);
    Object obj = futureObj.getReply();
    Exception ex = msg.getException();
    if (ex != null) {
      if (ex instanceof NoSuchAgentException) {
	throw new NoSuchAgentException();
      }
      else if (ex instanceof NoSuchMethodException) {
	throw new NoSuchMethodException();
      }
      else if (ex instanceof TimeoutException) {
	throw new TimeoutException();
      }
      else if (ex instanceof InvocationTargetException) {
	throw new InvocationException();
      }
      else {
	throw new InvocationException();
      }
    }
    return obj;
  }

  /**
   * Sends a message <code>Message</code> asynchronously. 
   * The sender does not have to be blocked until the receiver 
   * finishs handling the message, but can get the result via 
   * a future object returned. 
   * @param  msg a message to send asynchronously. 
   * @return a future object to give the reply of the message to the sender. 
   * @exception NoSuchAgentException if the receiver agent specified in msg 
   * is invalid. 
   * @exception NoSuchMethodException if the name or 
   * argument is invalid in msg. 
   * @exception TimeoutException if the handling goes timeout. 
   * @exception InvocationException if a runtime error in the handling. 
   * @see  agentspace.AgentContext#send(Message msg) 
   * @see  agentspace.AgentContext#call(Message msg) 
   */
  public Object future(Message msg) throws 
  NoSuchMethodException, 
  NoSuchAgentException, 
  TimeoutException, 
  InvocationException {
    Future futureObj = new Future();
    msg.setFuture(futureObj);
    msg.setSender(getIdentifier());
    AgentIdentifier aid = msg.getTarget();
    getAgentRuntime(aid).acceptRequest(msg);
    Object obj = futureObj.getReply();
    Exception ex = msg.getException();
    if (ex != null) {
      if (ex instanceof NoSuchAgentException) {
	throw new NoSuchAgentException();
      }
      else if (ex instanceof NoSuchMethodException) {
	throw new NoSuchMethodException();
      }
      else if (ex instanceof TimeoutException) {
	throw new TimeoutException();
      }
      else if (ex instanceof InvocationTargetException) {
	throw new InvocationException();
      }
      else {
	throw new InvocationException();
      }
    }
    return obj;
  }


  /*
   * Sends a message synchronously. 
   * The sender does not have to be blocked until finishing the message
   * handing. 
   * @param aid the identifier of the receiver agent 
   * @param msg a message to send
   */
  public Object call(AgentIdentifier aid, Message msg) {
    Future futureObj = new Future();
    msg.setFuture(futureObj);
    msg.setSender(getIdentifier());
    msg.setTarget(aid);
    try {
      getAgentRuntime(aid).acceptRequest(msg);
    }
    catch (NoSuchAgentException e) {}
    catch (NoSuchMethodException e) {}
    return futureObj.getReply();
  }

  /*
   * Sends a message asynchronously. 
   * The sender does not have to be blocked until finishing the message
   * handing and can get the result of the handling. 
   * @param aid the identifier of the receiver agent 
   * @param msg a message to send
   */
  public Future future(AgentIdentifier aid, Message msg) {
    Future futureObj = new Future();
    msg.setFuture(futureObj);
    msg.setSender(getIdentifier());
    msg.setTarget(aid);
    try {
      getAgentRuntime(aid).acceptRequest(msg);
    }
    catch (NoSuchAgentException e) {}
    catch (NoSuchMethodException e) {}
    return futureObj;
  }

  /*
   * Sends a message synchronously. 
   * The sender does not have to be blocked until finishing the message
   * handing.
   * @param aid the identifier of the receiver agent 
   * @param msg a message to send
   */
  public void send(AgentIdentifier aid, Message msg) {
    msg.setFuture(null);
    msg.setSender(getIdentifier());
    msg.setTarget(aid);
    try {
      getAgentRuntime(aid).acceptRequest(msg);
    }
    catch (NoSuchAgentException e) {}
    catch (NoSuchMethodException e) {}
  }

  /*
   * Dispatches the agent to the location specified as the argument name.
   * @param name the ip-address of the destination represented in String. 
   */
  public void dispatch(String address) {
    send("dispatch", address);
  }

  /**
   * Kills the calling agent. 
   */
  public void destroy() {
    send("destroy");
  }

  /**
   * Saves an agent itself as a file named as name. 
   * @param the name of the serialized agent. 
   */
  public void suspend(String name) {
    send("suspend", name);
  }

  /**
   * Suspends an agent specified as the identifier aid and saves 
   * it as a file named as name. 
   * @param aid the identifier of the agent. 
   * @param name the name of the serialized agent. 
   */
  public void suspend(AgentIdentifier aid, String name) {
    send(aid, "suspend", name);
  }

  /**
   * Suspends an agent specified as the identifier aid. 
   * @param aid the identifier of the agent. 
   * @return the serialized data of the suspended agent. 
   */
  public byte[] suspend(AgentIdentifier aid) {
    return (byte[])call(aid, "suspend");
  }

  /**
   * Duplicates an agent specified as the identifier aid. 
   * After calling method <code>duplicate()</code>, 
   * method <code>parent(AgentIdentifier child_identifier)</code> 
   * is called in the parent agent to tell the identifier of the 
   * child agent, where in the child agent 
   * the method <code>child()</code> is called to tell 
   * the identifier of its parent agent. 
   * @return the identifier of the newly instantiated agent. 
   */
  public AgentIdentifier duplicate() {
    return theInfo.getAgentRuntime().duplicateRequest();
  }

  /**
   * Creates an instance of the agent specified as name. 
   * @param name the name of the agent. 
   * @return the identifier of the newly instantiated agent. 
   */
  public AgentIdentifier create(String name) {
    return theInfo.getAgentRuntime().createRequest(name);
  }

  /**
   * Creates an instance of the agent serialized as data. 
   * @param data the serialized data of the agent. 
   * @return the identifier of the newly instantiated agent. 
   */
  public AgentIdentifier create(byte[] data) {
    return theInfo.getAgentRuntime().createRequest(data);
  }

  /** 
   * Sends a message to an agent speciifed by aid. 
   * The sender is blocked until the receiver finish handling the message. 
   * the name of the message is <code>name</code> and 
   * the arguments consists of obj1, obj2, obj3, obj4, and obj5. 
   * @param obj1 the first arguments. 
   * @param obj2 the second arguments. 
   * @param obj3 the third  arguments. 
   * @param obj4 the fourth arguments. 
   * @param obj5 the fifth arguments. 
   * @return the results of the handling. 
   */
  public Object call(AgentIdentifier aid, String name, 
     Object obj1, Object obj2, Object obj3, Object obj4, Object obj5) {
    Message msg = new Message(name);
    msg.setArg(obj1);
    msg.setArg(obj2);
    msg.setArg(obj3);
    msg.setArg(obj4);
    msg.setArg(obj5);
    return call(aid, msg);
  }

  public Object call(AgentIdentifier aid, String name, 
		     Object obj1, Object obj2, Object obj3, Object obj4) {
    Message msg = new Message(name);
    msg.setArg(obj1);
    msg.setArg(obj2);
    msg.setArg(obj3);
    msg.setArg(obj4);
    return call(aid, msg);
  }

  public Object call(AgentIdentifier aid, String name, 
		     Object obj1, Object obj2, Object obj3) {
    Message msg = new Message(name);
    msg.setArg(obj1);
    msg.setArg(obj2);
    msg.setArg(obj3);
    return call(aid, msg);
  }

  public Object call(AgentIdentifier aid, String name, 
		     Object obj1, Object obj2) {
    Message msg = new Message(name);
    msg.setArg(obj1);
    msg.setArg(obj2);
    return call(aid, msg);
  }

  public Object call(AgentIdentifier aid, String name, 
		     Object obj1) {
    Message msg = new Message(name);
    msg.setArg(obj1);
    return call(aid, msg);
  }

  public Object call(AgentIdentifier aid, String name) {
    Message msg = new Message(name);
    return call(aid, msg);
  }

  public Future future(AgentIdentifier aid, String name, 
       Object obj1, Object obj2, Object obj3, Object obj4, Object obj5) {
    Message msg = new Message(name);
    msg.setArg(obj1);
    msg.setArg(obj2);
    msg.setArg(obj3);
    msg.setArg(obj4);
    msg.setArg(obj5);
    return future(aid, msg);
  }

  public Future future(AgentIdentifier aid, String name, 
		       Object obj1, Object obj2, Object obj3, Object obj4) {
    Message msg = new Message(name);
    msg.setArg(obj1);
    msg.setArg(obj2);
    msg.setArg(obj3);
    msg.setArg(obj4);
    return future(aid, msg);
  }

  public Future future(AgentIdentifier aid, String name, 
		       Object obj1, Object obj2, Object obj3) {
    Message msg = new Message(name);
    msg.setArg(obj1);
    msg.setArg(obj2);
    msg.setArg(obj3);
    return future(aid, msg);
  }

  public Future future(AgentIdentifier aid, String name, 
		       Object obj1, Object obj2) {
    Message msg = new Message(name);
    msg.setArg(obj1);
    msg.setArg(obj2);
    return future(aid, msg);
  }

  public Future future(AgentIdentifier aid, String name, 
		       Object obj1) {
    Message msg = new Message(name);
    msg.setArg(obj1);
    return future(aid, msg);
  }

  public Future future(AgentIdentifier aid, String name) {
    Message msg = new Message(name);
    return future(aid, msg);
  }

  public void send(AgentIdentifier aid, String name, 
	   Object obj1, Object obj2, Object obj3, Object obj4,  Object obj5) {
    Message msg = new Message(name);
    msg.setArg(obj1);
    msg.setArg(obj2);
    msg.setArg(obj3);
    msg.setArg(obj4);
    msg.setArg(obj5);
    send(aid, msg);
  }

  public void send(AgentIdentifier aid, String name, 
		   Object obj1, Object obj2, Object obj3, Object obj4) {
    Message msg = new Message(name);
    msg.setArg(obj1);
    msg.setArg(obj2);
    msg.setArg(obj3);
    msg.setArg(obj4);
    send(aid, msg);
  }

  public void send(AgentIdentifier aid, String name, 
		   Object obj1, Object obj2, Object obj3) {
    Message msg = new Message(name);
    msg.setArg(obj1);
    msg.setArg(obj2);
    msg.setArg(obj3);
    send(aid, msg);
  }

  public void send(AgentIdentifier aid, String name, 
		   Object obj1, Object obj2) {
    Message msg = new Message(name);
    msg.setArg(obj1);
    msg.setArg(obj2);
    send(aid, msg);
  }

  public void send(AgentIdentifier aid, String name, 
		   Object obj1) {
    Message msg = new Message(name);
    msg.setArg(obj1);
    send(aid, msg);
  }

  public void send(AgentIdentifier aid, String name) {
    Message msg = new Message(name);
    send(aid, msg);
  }

  public void send(String name, 
		   Object obj1, Object obj2, Object obj3, Object obj4) {
    send(getIdentifier(), name, obj1, obj2, obj3, obj4);
  }

  public void send(String name, 
		   Object obj1, Object obj2, Object obj3) {
    send(getIdentifier(), name, obj1, obj2, obj3);
  }

  public void send(String name, 
		   Object obj1, Object obj2) {
    send(getIdentifier(), name, obj1, obj2);
  }

  public void send(String name, 
		   Object obj1) {
    send(getIdentifier(), name, obj1);
  }

  public void send(String name) {
    send(getIdentifier(), name);
  }
}




