/*
 * Copyright (C) 1997 Ichiro Satoh
 * All Rights Reserved.
 * written by Ichiro Satoh
 *
 * This program was developed by Ichiro Satoh at Ochanomizu University 
 * as a part of a mobile object system, titled "AgentSpace". All the 
 * programs of the system is developed by Ichiro Satoh and all rights 
 * are reserved by Ichiro Satoh. 
 *
 * Permission to use, copy, modify and distribute this software and
 * its documentation is hereby granted, provided that both the
 * copyright notice and this permission notice appear in all copies of
 * the software, derivative works or modified versions, and any
 * portions thereof, and that both notices appear in supporting 
 * documentation.
 *
 * FREE USE OF THIS SOFTWARE IS ALLOWED IN ITS "AS IS" CONDITION.
 * BUT WITHOUT WARRANTY. THE AUTHORS DISCLAIM ANY LIABILITY OF ANY 
 * KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS 
 * SOFTWARE.
 */

package agentspace;

import java.io.*;
import java.net.*;
import java.util.*;
import java.lang.*;
import java.lang.reflect.*;
import java.util.zip.*;

public class AgentInputStream {
  DataInputStream dis;
  AgentInfo ainfo;

  public AgentInputStream(InputStream is){
    dis = new DataInputStream(is);
    ainfo = new AgentInfo();
  }

  public AgentInfo read() {
    try {
      int datalen = dis.readInt();
      int codelen = dis.readInt();
      byte data[] = new byte[datalen];
      byte code[] = new byte[codelen];
      dis.read(data, 0, datalen);
      dis.read(code, 0, codelen);
      dis.close();
      ainfo = readEntry(code, data);
      if (ainfo == null) {
	System.err.println("ERROR: illugal agent packet.");
      }
    }
    catch (IOException e) {
      System.err.println(e);
      ainfo = null;
    }      
    return ainfo;
  }

  private AgentInfo readEntry(byte[] code, byte[] data) {
    AgentIdentifier aid = null;
    URL originalHost = null;
    URL previousHost = null;
    URL codeBaseHost = null;
    Date birthDay = null;
    String status = null;
    String aName = null;
    Agent anAgent = null;
    AgentLoader al = null;
    Vector parentList = null;

    try {
      ByteArrayInputStream byteInStream = new ByteArrayInputStream(data);
      ZipInputStream zipInStream = new ZipInputStream(byteInStream);
      ZipEntry zipEnt;
      while ((zipEnt = zipInStream.getNextEntry()) != null) {
	String entryName = zipEnt.getName();
	if (entryName.startsWith("META-INF")) {
	  continue;
	}
	else if (entryName.startsWith("AGID")) {
	  byte[] agid = toByteArray(zipInStream);
	  long idnumber = 
	    (new Long(AgentByteCode.byteArrayToString(agid))).longValue();
	  aid = new AgentIdentifier(idnumber);
	}
	else if (entryName.startsWith("CDBS")) {
	  byte[] cdbs = toByteArray(zipInStream);
	  try {
	    codeBaseHost = new URL(AgentByteCode.byteArrayToString(cdbs));
	  }
	  catch (MalformedURLException e) {}
	}
	else if (entryName.startsWith("OGHT")) {
	  byte[] oght = toByteArray(zipInStream);
	  try {
	    originalHost = new URL(AgentByteCode.byteArrayToString(oght));
	  }
	  catch (MalformedURLException e) {}
	}
	else if (entryName.startsWith("PVHT")) {
	  byte[] pvht = toByteArray(zipInStream);
	  try {
	    previousHost = new URL(AgentByteCode.byteArrayToString(pvht));
	  }
	  catch (MalformedURLException e) {}
	}
	else if (entryName.startsWith("STAT")) {
	  byte[] stat = toByteArray(zipInStream);
	  status = AgentByteCode.byteArrayToString(stat);
	}
	else if (entryName.startsWith("PRNT")) {
	  byte[] prnt = toByteArray(zipInStream);
	  parentList  = (Vector)AgentByteCode.byteArrayToObject(prnt);
	}
	else if (entryName.startsWith("NAME")) {
	  byte[] name = toByteArray(zipInStream);
	  aName = AgentByteCode.byteArrayToString(name);
	  if (aName == null) {
	    String newName = (anAgent.getClass()).toString();
	    if (newName.startsWith("class ")) {
	      newName = newName.substring(6,newName.length());
	    }
	    for ( int i = newName.length() ; i < 12 ; i++ ) {
	      newName = newName + " ";
	    }
	    aName = newName;
	  }
	}
	else if (entryName.startsWith("CNTX")) {
	  byte[] context = toByteArray(zipInStream);
	  al = new AgentLoader();
	  Object obj = al.loadAgent(context, code, codeBaseHost);
	  if (obj != null && obj instanceof Agent) {
	    anAgent = (Agent)obj;
	  }
	}
      }
      ainfo.setAgent(anAgent);
      ainfo.setAgentThread(Thread.currentThread());
      ainfo.setAgentLoader(al);
      ainfo.setOriginalHost(originalHost);
      ainfo.setPreviousHost(previousHost);
      ainfo.setCodeBaseHost(codeBaseHost);
      ainfo.setAgentStatus(status);
      ainfo.setAgentCode(code);
      ainfo.setName(aName);
      if (ainfo.getIdentifier() == null) {
	ainfo.setIdentifier(aid);
      }
      if (parentList == null) {
	ainfo.addParentList(AgentIdentifier.GeneralManager());
      }
      else {
	ainfo.setParentList(parentList);
      }
      System.out.println("Read:  AGID = "+aid.toString()
			 +",  NAME = "+ainfo.getName()
			 +",  STAT = "+ainfo.getAgentStatus());

      zipInStream.close();
      byteInStream.close();
    } catch (IOException e) { 	 
      System.err.println(e); 	 
    } 
    return ainfo;
  }

  public byte[] toByteArray(ZipInputStream zipInStream) {
    byte[] data = null;
    try {
      ByteArrayOutputStream byteOutStream = new ByteArrayOutputStream();
      byte[] buffer = new byte[1024];
      int count;
      while ((count = zipInStream.read(buffer)) != -1) {
	byteOutStream.write(buffer, 0, count);
      }
      data = byteOutStream.toByteArray();
      byteOutStream.close();
    } catch (IOException e) { 	 
      System.err.println(e); 	 
    } 
    return data;
  }
}
