/*
 * Copyright (C) 1997 Ichiro Satoh
 * All Rights Reserved.
 * written by Ichiro Satoh
 *
 * This program was developed by Ichiro Satoh at Ochanomizu University 
 * as a part of a mobile object system, titled "AgentSpace". All the 
 * programs of the system is developed by Ichiro Satoh and all rights 
 * are reserved by Ichiro Satoh. 
 *
 * Permission to use, copy, modify and distribute this software and
 * its documentation is hereby granted, provided that both the
 * copyright notice and this permission notice appear in all copies of
 * the software, derivative works or modified versions, and any
 * portions thereof, and that both notices appear in supporting 
 * documentation.
 *
 * FREE USE OF THIS SOFTWARE IS ALLOWED IN ITS "AS IS" CONDITION.
 * BUT WITHOUT WARRANTY. THE AUTHORS DISCLAIM ANY LIABILITY OF ANY 
 * KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS 
 * SOFTWARE.
 */

package agentspace;

import java.io.*;
import java.util.*;
import java.lang.*;
import java.net.*;

class AgentManager {
  private AgentServer theAgentServer;
  private AgentMonitor theAgentMonitor;
  private AgentList localAgentList;
  private ThreadGroup theThreadGroup = null;

  public AgentManager(AgentServer as, AgentList al) {
    localAgentList = al;
    theAgentServer = as;
    theThreadGroup = new ThreadGroup("MANAGER");
//  (new Messenger()).initialize(localAgentList, this);
    localAgentList.setAgentManager(this);
  }

  public void quit() {
    theAgentServer.quit();
  }

  public AgentList getAgentList() {
    return localAgentList;
  }

  public void unregistry(AgentInfo ainfo) {
    localAgentList.removeAgentInfo(ainfo);
  }

  public synchronized AgentInfo registry(AgentInfo ainfo) {
    if (ainfo == null) {
      return null;
    }
    AgentRuntime aruntime = new AgentRuntime(this, ainfo);
    AgentContext acontext = new AgentContext(localAgentList, ainfo);
    (ainfo.getAgent()).setAgentContext(acontext);
    if (!(ainfo.getAgentStatus().equals(AgentStatus.ONTRANSMIT))) {
      AgentIdentifier aid = new AgentIdentifier();
      ainfo.setIdentifier(aid);
    }
    ainfo.setAgentContext(acontext);
    ainfo.setAgentRuntime(aruntime);

    /* The thread of each agent is started here */
    aruntime.startRuntime();
    localAgentList.addAgentInfo(ainfo);
    return ainfo;
  }

  public AgentIdentifier createAgent(String name, AgentIdentifier aid){
    AgentInfo ainfo = localAgentList.getAgentInfo(aid);
    AgentInfo newinfo = loadAgent(name);
    newinfo.setParentList(ainfo.getParentList());
    newinfo.addParentList(ainfo.getIdentifier());
    return newinfo.getIdentifier();
  }

  public AgentIdentifier createAgent(byte[] data, AgentIdentifier aid) {
    AgentInfo ainfo = localAgentList.getAgentInfo(aid);
    AgentInfo newinfo = loadAgent(data);
    newinfo.setParentList(ainfo.getParentList());
    newinfo.addParentList(ainfo.getIdentifier());
    return newinfo.getIdentifier();
  }

  public void destroyAgent(AgentIdentifier aid) {
    AgentInfo ainfo = localAgentList.getAgentInfo(aid);
    AgentRuntime art = ainfo.getAgentRuntime();
    art.destroyForce();
    localAgentList.removeAgentInfo(ainfo);
  }

  public void killAgent(AgentIdentifier aid) {
    AgentInfo ainfo = localAgentList.getAgentInfo(aid);
    AgentRuntime art = ainfo.getAgentRuntime();
    art.killForce();
    localAgentList.removeAgentInfo(ainfo);
  }

  public void dispatchAgent(AgentIdentifier aid, URL url) {
    AgentInfo ainfo = localAgentList.getAgentInfo(aid);
    AgentRuntime art = ainfo.getAgentRuntime();
    localAgentList.removeFromAgentInfo(ainfo);
    art.dispatchForce(url);
    ((Agent)ainfo.getAgent()).setAgentContext(null);
    int port;
    if (url.getPort() >= 0) {
      port = url.getPort();
    }
    else {
      port = AgentServer.theSendPort;
    }
    AgentSender as = new AgentSender(url.getHost(), port, this);
    as.transfer(ainfo);
  }

  public void dispatchAgent(AgentIdentifier aid, String address) {
    AgentInfo ainfo = localAgentList.getAgentInfo(aid);
    AgentRuntime art = ainfo.getAgentRuntime();
    art.dispatchForce(address);
    ((Agent)ainfo.getAgent()).setAgentContext(null);
    AgentSender as = new AgentSender(address, AgentServer.theSendPort, this);
    as.transfer(ainfo);
    localAgentList.removeFromAgentInfo(ainfo);
  }

  public void dispatchFailure(AgentInfo ainfo) {
    System.out.println("AgentManager: dispatchFailure()");
    ainfo.setAgentStatus(AgentStatus.ERROR);
    ainfo.setAgentError(AgentError.ONSENDING);
    registry(ainfo);
  }

  public AgentIdentifier duplicateAgent(AgentIdentifier aid) {
    AgentInfo parentinfo = localAgentList.getAgentInfo(aid);
    (parentinfo.getAgentRuntime()).duplicateForce();
    AgentInfo childinfo = (new AgentClone(this)).cloneAgent(parentinfo);
    AgentIdentifier childaid = childinfo.getIdentifier();
    (parentinfo.getAgentRuntime()).parentRequest(childaid);
    return childaid;
  }

  public AgentInfo loadAgent(String name) {
    AgentFile af = new AgentFile(this);
    return af.load(name);
  }

  public AgentInfo loadAgent(byte[] data) {
    AgentFile af = new AgentFile(this);
    return af.load(data);
  }

  public byte[] suspendAgent(AgentIdentifier aid) {
    byte[] data;
    AgentInfo ainfo = localAgentList.getAgentInfo(aid);
    AgentRuntime art = ainfo.getAgentRuntime();
    art.suspendForce();
    ((Agent)ainfo.getAgent()).setAgentContext(null);
    AgentFile af = new AgentFile(this);
    data = af.save(ainfo);
    localAgentList.removeAgentInfo(ainfo);
    return data;
  }

  public void suspendAgent(AgentIdentifier aid, String fileName) {
    AgentInfo ainfo = localAgentList.getAgentInfo(aid);
    AgentRuntime art = ainfo.getAgentRuntime();
    art.suspendForce();
    ((Agent)ainfo.getAgent()).setAgentContext(null);
    AgentFile af = new AgentFile(this);
    af.save(fileName, ainfo);
    localAgentList.removeAgentInfo(ainfo);
  }

  public void toFrontAgent(AgentIdentifier aid) {
    AgentInfo ainfo = localAgentList.getAgentInfo(aid);
    AgentRuntime art = ainfo.getAgentRuntime();
    art.toFrontRequest();
  }

  public AgentInfo loadAgentFile(String dirName, String fileName) {
    if (fileName == null) {
      return null;
    }
    if (dirName == null) {
      dirName = ".";
    }
    AgentFile af = new AgentFile(this);
    return af.load(dirName + fileName);
  }

  protected Vector listeners = new Vector();

  public void addAgentListener(AgentListener listener) {
    listeners.addElement(listener);
  }

  public void removeAgentListener(AgentListener listener) {
    listeners.removeElement(listener);
  }

  protected void dispatchAgentChanged() {
    AgentEvent e = new AgentEvent(this);
    Vector listeners = (Vector)this.listeners.clone();
    for( int i = 0 ; i < listeners.size() ; i++ ) {
      ((AgentListener)listeners.elementAt(i)).agentChanged(e);
    }
  }

  public void update() {
    dispatchAgentChanged();
  }

  public ThreadGroup getThreadGroup() {
    return theThreadGroup;
  }
}
