/*
 * Copyright (C) 1997, 1998 Ichiro Satoh
 * All Rights Reserved.
 * written by Ichiro Satoh
 *
 * This program was developed by Ichiro Satoh at Ochanomizu University 
 * as a part of a mobile object system, titled "AgentSpace". All the 
 * programs of the system is developed by Ichiro Satoh and all rights 
 * are reserved by Ichiro Satoh. 
 *
 * Permission to use, copy, modify and distribute this software and
 * its documentation is hereby granted, provided that both the
 * copyright notice and this permission notice appear in all copies of
 * the software, derivative works or modified versions, and any
 * portions thereof, and that both notices appear in supporting 
 * documentation.
 *
 * FREE USE OF THIS SOFTWARE IS ALLOWED IN ITS "AS IS" CONDITION.
 * BUT WITHOUT WARRANTY. THE AUTHORS DISCLAIM ANY LIABILITY OF ANY 
 * KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS 
 * SOFTWARE.
 */


package agentspace;

import java.io.*;
import java.net.*;
import java.util.*;
import java.lang.*;
import java.lang.reflect.*;

public class AgentRuntime extends Thread {
  private Agent theAgent;
  private AgentInfo theInfo;
  private AgentManager theManager = null;
  private AgentMessageQueue queue;
  private ThreadGroup theThreadGroup = null;
  private Thread theThread = null;

  public AgentRuntime(AgentManager am, AgentInfo ainfo) {
    theManager = am;
    theInfo = ainfo;
    queue = new AgentMessageQueue();
  }

  public void run() {
    theAgent = theInfo.getAgent();
    if (theAgent == null) {
//    System.err.println("Agent is not valid.");
      return;
    }
    if (theInfo.getAgentStatus().equals(AgentStatus.PREINIT)) {
      theInfo.setAgentStatus(AgentStatus.ONCREATE);
      theAgent.create();
      theInfo.setAgentStatus(AgentStatus.NORMAL);
    }
    else if (theInfo.getAgentStatus().equals(AgentStatus.ONTRANSMIT)) {
      theInfo.setAgentStatus(AgentStatus.ONARRIVE);
      theAgent.arrive();
      theInfo.setAgentStatus(AgentStatus.NORMAL);
    }
    else if (theInfo.getAgentStatus().equals(AgentStatus.PERSISTENT)) {
      theInfo.setAgentStatus(AgentStatus.ONRESUME);
      theAgent.resume();
      theInfo.setAgentStatus(AgentStatus.NORMAL);
    }
    else if (theInfo.getAgentStatus().equals(AgentStatus.CHILD)) {
      theInfo.setAgentStatus(AgentStatus.ONCLONE);
      theAgent.child(theInfo.getParentIdentifier());
      theInfo.setAgentStatus(AgentStatus.NORMAL);
    }
    else if (theInfo.getAgentStatus().equals(AgentStatus.ERROR)) {
      theInfo.setAgentStatus(AgentStatus.ONEXCEPT);
      if (! (theAgent.except(theInfo.getAgentError()))) {
	theInfo.setAgentStatus(AgentStatus.DEATH);
	theInfo.setAgentError(null);
      	return;
      }
      theInfo.setAgentError(null);
      theInfo.setAgentStatus(AgentStatus.NORMAL);
    }
    else {
      System.out.println("unknown status: "+theInfo.getAgentStatus());
    }
    while(true) {
      theManager.update();
      if (theInfo.getAgentStatus() == AgentStatus.DEATH) {
	break;
      }
      if (acceptMessage()) {
	break;
      }
    }
    theManager.update();
    theAgent = null;
    theInfo = null;
    queue.clear();
    stopRuntime();
  }

  /* Each agent has its own message queue. Incoming messages are 
     stored in the queue and basically handled at the arriving order, 
     called FCFS. However, the messages which are sent by the receving 
     agents are first handled. */
  private boolean acceptMessage() {
    Message msg = queue.getMessage();
    System.out.println("Message \""+msg.getName()+"\" handled at "
		       +theInfo.getIdentifier());

//  AgentTimeout am = new AgentTimeout(this, theThreadGroup);
//    return am.invokeMethod(msg);
    return handleMessage(msg);
  }


  /* When the result of this method is true, the activitiy of the agent
     becomes inactive, otherwise it active. */
  public boolean handleMessage(Message msg) {
    Future reply = msg.getFuture();
    if ((msg.getName()).equals("dispatch") && msg.getArgCount() == 1 
	&& msg.getArg(0) instanceof URL) {
      if ( ! (theInfo.hasCapability(msg.getSender()))) {
	throw new SecurityException("Unknown Agent trys to dispatch me");
      }
      theManager.dispatchAgent(theInfo.getIdentifier(), (URL)msg.getArg(0));
      return true;
    }
    if ((msg.getName()).equals("dispatch") && msg.getArgCount() == 1 
	&& msg.getArg(0) instanceof String) {
      if ( ! (theInfo.hasCapability(msg.getSender()))) {
	throw new SecurityException("Unknown Agent trys to dispatch me");
      }
      theManager.dispatchAgent(theInfo.getIdentifier(), (String)msg.getArg(0));
      return true;
    }
    if ((msg.getName()).equals("destroy") && msg.getArgCount() == 0) {
      if ( ! (theInfo.hasCapability(msg.getSender()))) {
	throw new SecurityException("Unknown Agent trys to destroy me");
      }
      theManager.destroyAgent(theInfo.getIdentifier());
      return true;
    }
    if ((msg.getName()).equals("create") && msg.getArgCount() == 1 
	&& msg.getArg(0) instanceof String) {
      AgentIdentifier aid = 
	theManager.createAgent((String)msg.getArg(0),theInfo.getIdentifier());
      if (reply != null) {
	reply.setReply(aid);
      }
      return true;
    }
    if ((msg.getName()).equals("parent") && msg.getArgCount() == 1) {
      AgentIdentifier caid = (AgentIdentifier)msg.getArg(0);
      parentForce(caid);
      if (reply != null) {
	reply.setReply(caid);
      }
      return false;
    }
    if ((msg.getName()).equals("duplicate") && msg.getArgCount() == 0) {
      if ( ! (theInfo.hasCapability(msg.getSender()))) {
	throw new SecurityException("Unknown Agent trys to duplicate me");
      }
      AgentIdentifier aid = theManager.duplicateAgent(theInfo.getIdentifier());
      if (reply != null) {
	reply.setReply(aid);
      }
      return false;
    }
    if ((msg.getName()).equals("suspend")) {
      if (msg.getArgCount() == 0) {
	if ( ! (theInfo.hasCapability(msg.getSender()))) {
	  throw new SecurityException("Unknown Agent trys to suspend me");
	}
	Object result = theManager.suspendAgent(theInfo.getIdentifier());
	if (reply != null) {
	  reply.setReply(result);
	}
	return true;
      }
      if (msg.getArgCount() == 1 && msg.getArg(0) instanceof String) {
	if ( ! (theInfo.hasCapability(msg.getSender()))) {
	  throw new SecurityException("Unknown Agent trys to suspend me");
	}
	theManager.suspendAgent(theInfo.getIdentifier(),(String)msg.getArg(0));
	return true;
      }
    }

    /* When the agent receives the reserved message names, it invoke 
       methods whose both name and parameters are the same to those. */
    else {
      try {
	Object result = AgentMethod.invokeMethod(theInfo, msg);
	/* when future call and sychronous call need to return 
	   their result to the senders. The results are set in "reply". */
	if (reply != null) {
	  reply.setReply(result);
	}
      }
      catch (NoSuchMethodException e) {
	msg.setException(e);
	e.printStackTrace();
	reply.setReply(null);
      }
      catch (IllegalAccessException e) {
	msg.setException(e);
	e.printStackTrace();
	reply.setReply(null);
      }
      catch (IllegalArgumentException e) {
	msg.setException(e);
	e.printStackTrace();
	reply.setReply(null);
      }
      catch (InvocationTargetException e) {
	msg.setException(e);
	e.printStackTrace();
	reply.setReply(null);
      }
      catch (NullPointerException e) {
	e.printStackTrace();
      }
    }
    return false;
  }

  private void forceMessage(Message msg) {
    queue.pushMessage(msg);
  }

  public void acceptRequest(Message msg) throws 
  NoSuchMethodException, 
  NoSuchAgentException {
    if (msg.getTarget() == null || 
	theManager.getAgentList().getAgentInfo(msg.getTarget()) == null) {
//    System.out.println("RUNTIME: NoSuchAgentException "+msg.getName());
      throw new NoSuchAgentException();
    }
    try {
      AgentMethod.isMethod(theInfo, msg);
    }
    catch (IllegalArgumentException e) {
      throw new NoSuchMethodException();
    }
    catch (IllegalAccessException e) {
      throw new NoSuchMethodException();
    }
    System.out.println("Message \""+msg.getName()
		       +"\" coming to  "+theInfo.getIdentifier());
    if (msg.getSender() == theInfo.getIdentifier()) {
      if (((msg.getName()).equals("destroy") ||
	   (msg.getName()).equals("suspend")) && msg.getArgCount() == 0) {
	queue.putMessage(msg);
      }
      else {
	queue.pushMessage(msg);
      }
    }
    else {
      queue.putMessage(msg);
    }
    if(msg.getTimeout() > 0 && msg.getFuture() != null) {
      AgentTimeout am = new AgentTimeout();
      am.enqueueMessage(msg, queue, theThreadGroup);
    }
  }

  public void startRuntime() {
    ThreadGroup managerThreadGroup = theManager.getThreadGroup();
    theThreadGroup = 
      new ThreadGroup(managerThreadGroup, theInfo.getIdentifier().toString());
    theThread = new Thread(theThreadGroup, this, "-RUNTIME");
    theInfo.setAgentThread(theThread);
    theThread.start();
  }

  public void stopRuntime() {
    if (queue != null) {
      while(!(queue.isEmpty())) {
	Message msg = queue.getMessage();
	msg.setException(new NoSuchAgentException());
	Future reply = msg.getFuture();
	reply.setReply(null);
	System.out.println("Message \""+msg.getName()+"\" canceled at "
			   +theInfo.getIdentifier());
      }
    }
    if (!(theThreadGroup.parentOf(Thread.currentThread().getThreadGroup()))) {
      theThreadGroup.stop();
    }
  }

  public void createForce() {
    theManager.update();
  }

  public void duplicateForce() {
    theInfo.setAgentStatus(AgentStatus.ONCLONE);
    if (theAgent != null) {
      theAgent.duplicate();
    }
    theInfo.setAgentStatus(AgentStatus.NORMAL);
    theManager.update();
  }

  public void parentForce(AgentIdentifier caid) {
    theInfo.setAgentStatus(AgentStatus.ONPARENT);
    if (theAgent != null) {
      theAgent.parent(caid);
    }
    theInfo.setAgentStatus(AgentStatus.NORMAL);
    theManager.update();
  }

  public void dispatchForce(String address) {
    URL url = null;
    try {
      url = new URL("http://"+address);
    }
    catch (MalformedURLException e) {
      System.err.println(e);
    }
    dispatchForce(url);
  }

  public void dispatchForce(URL url) {
    theInfo.setAgentStatus(AgentStatus.ONDISPATCH);
    if (theAgent != null) {
      theAgent.dispatch(url);
    }
    theInfo.setAgentStatus(AgentStatus.ONTRANSMIT);
    Thread agentThread = theInfo.getAgentThread();
    if (Thread.currentThread() != agentThread) {
      stopRuntime();
    }
    theManager.update();
  }

  public void suspendForce() {
    theInfo.setAgentStatus(AgentStatus.ONSUSPEND);
    if (theAgent != null) {
      theAgent.suspend();
    }
    theInfo.setAgentStatus(AgentStatus.PERSISTENT);
    Thread agentThread = theInfo.getAgentThread();
    if (Thread.currentThread() != agentThread) {
      stopRuntime();
    }
    theManager.update();
  }

  public void destroyForce() {
    theInfo.setAgentStatus(AgentStatus.ONDESTROY);
    if (theAgent != null) {
      theAgent.destroy();
    }
    theInfo.setAgentStatus(AgentStatus.DEATH);
    Thread agentThread = theInfo.getAgentThread();
    if (Thread.currentThread() != agentThread) {
      stopRuntime();
    }
    theManager.update();
  }

  public void killForce() {
    Thread agentThread = theInfo.getAgentThread();
    if (Thread.currentThread() != agentThread) {
      stopRuntime();
    }
    theManager.update();
  }

  public void killRequest() {
    Thread agentThread = theInfo.getAgentThread();
    if (Thread.currentThread() != agentThread) {
      stopRuntime();
    }
  }

  public AgentIdentifier createRequest(String name) {
    AgentIdentifier aid = theManager.createAgent(name,theInfo.getIdentifier());
    return aid;
  }

  public AgentIdentifier createRequest(byte[] data) {
    AgentIdentifier aid = theManager.createAgent(data,theInfo.getIdentifier());
    return aid;
  }

  public void parentRequest(AgentIdentifier caid) {
    Message msg = new Message("parent");
    msg.setFuture(null);
    msg.setSender(theInfo.getIdentifier());
    msg.setTarget(theInfo.getIdentifier());
    msg.setArg(caid);
    forceMessage(msg);
  }

  public AgentIdentifier duplicateRequest() {
    Future futureObj = new Future();
    Message msg = new Message("duplicate");
    msg.setFuture(futureObj);
    msg.setSender(theInfo.getIdentifier());
    msg.setTarget(theInfo.getIdentifier());
    forceMessage(msg);
    Object obj = futureObj.getReply();
    if (obj instanceof AgentIdentifier) {
      return (AgentIdentifier)obj;
    }
    return null;
  }

  public synchronized void dispatchRequest(String address) {
    while(theInfo.getAgentStatus() != AgentStatus.NORMAL 
	  && !(queue.isEmpty())){
      try {
	wait();
      }
      catch (InterruptedException e) {}
    }
    Message msg = new Message("dispatch");
    msg.setFuture(null);
    msg.setSender(theInfo.getIdentifier());
    msg.setTarget(theInfo.getIdentifier());
    msg.setArg(address);
    forceMessage(msg);
  }

  public synchronized void dispatchRequest(URL url) {
    while(theInfo.getAgentStatus() != AgentStatus.NORMAL 
	  && !(queue.isEmpty())){
      try {
	wait();
      }
      catch (InterruptedException e) {}
    }
    Message msg = new Message("dispatch");
    msg.setFuture(null);
    msg.setSender(theInfo.getIdentifier());
    msg.setTarget(theInfo.getIdentifier());
    msg.setArg(url);
    forceMessage(msg);
  }

  public synchronized void destroyRequest() {
    while(theInfo.getAgentStatus() != AgentStatus.NORMAL 
	  && !(queue.isEmpty())){
      try {
	wait();
      }
      catch (InterruptedException e) {}
    }
    Message msg = new Message("destroy");
    msg.setFuture(null);
    msg.setSender(theInfo.getIdentifier());
    msg.setTarget(theInfo.getIdentifier());
    forceMessage(msg);
  }

  public synchronized void suspendRequest(String name) {
    while(theInfo.getAgentStatus() != AgentStatus.NORMAL 
	  && !(queue.isEmpty())){
      try {
	wait();
      }
      catch (InterruptedException e) {}
    }
    Message msg = new Message("suspend");
    msg.setFuture(null);
    msg.setSender(theInfo.getIdentifier());
    msg.setTarget(theInfo.getIdentifier());
    msg.setArg(name);
    forceMessage(msg);
  }

  public synchronized void suspendRequest() {
    while(theInfo.getAgentStatus() != AgentStatus.NORMAL 
	  && !(queue.isEmpty())){
      try {
	wait();
      }
      catch (InterruptedException e) {}
    }
    Message msg = new Message("suspend");
    msg.setFuture(null);
    msg.setSender(theInfo.getIdentifier());
    msg.setTarget(theInfo.getIdentifier());
    forceMessage(msg);
  }

  public void toFrontRequest() {
    if (theAgent != null) {
      theAgent.toFront();
    }
  }
}

