/*
 *
 * CongaRequest オブジェクト
 *
 *   XmlHttpRequest を使ってリクエスト送信を行い，
 *   受信したXMLデータを handleResponse メソッドによって解釈し，
 *   ドキュメントの更新を行なう。
 *   handleResponse 関数は，new CongaRequest().parse() メソッドを呼び出す。
 *
 *  コンストラクタ：
 *     CongaRequest
 *      @param method リクエスト時の method（POST または GET）を指定する。
 *                    指定がない場合は POST が設定される。
 *      @param requestForm リクエストデータ構築のための form を指定する。
 *                         指定がない場合は document.forms[0] が設定される。
 *
 *  メソッド:
 *     send(url, requestData) リクエスト送信を行なう。
 *      @param url         リクエスト送信先 URL。指定必須。
 *      @param requestData リクエスト対象データ。リクエスト対象データ文字列
 *                         を返す function かリクエスト対象データ文字列のど
 *                         ちらかが指定できる。
 *                         この引数を指定しなかった場合，コンストラクタの 
 *                         requestForm 引数で指定した form の内容が送信対象
 *                         データとして使用される。
 *
 *  プロパティ ( [ ]内はデフォルト値 ) :
 *     method リクエスト時のメソッド [コンストラクタ引数 method で指定された値]
 *
 *     contentType リクエスト時の ContentType [application/x-www-form-urlencoded;charset=UTF-8]
 *
 *     async 非同期送信するかどうか [false]
 *
 *     requestForm [コンストラクタ引数 requestForm で指定された値]
 *
 */


/**
 * CongaRequest を作成する。
 *
 * @param method リクエスト時の method（POST または GET）を指定する。
 *               指定がない場合は POST が設定される。
 * @param requestForm リクエストデータ構築のための form を指定する。
 *                    指定がない場合は document.forms[0] が設定される。
 */
function CongaRequest(method, requestForm) {
	//// ローカル変数定義
	var handler = this;


	//// プロパティ定義

	/**
	 * リクエスト時の method。（[POST] / GET）
	 */
	this.method = (method)? method : "POST";

	/**
	 * リクエストデータ構築のための form。
	 */
	this.requestForm = (requestForm)? requestForm : document.forms[0];

	/**
	 * ContentType ["application/x-www-form-urlencoded;charset=UTF-8"]
	 */
	this.contentType = "application/x-www-form-urlencoded;charset=UTF-8";

	/**
	 * 非同期リクエストを行なうかどうか [false]
	 */
	this.async = false;


	//// public メソッド定義

	/**
	 * リクエスト送信を実行する。
	 * @param url リクエスト送信先 URL。指定必須。
	 * @param requestData リクエスト対象データ。リクエスト対象データ文字列
	 *                    を返す function かリクエスト対象データ文字列のど
	 *                    ちらかが指定できる。
	 *                    この引数を指定しなかった場合，コンストラクタの 
	 *                    requestForm 引数で指定した form の内容が送信対象
	 *                    データとして使用される。
	 * @return 通信が正常終了した場合は true。それ以外は false。
	 *         非同期モードで通信した場合も false。
	 */
	this.send = function (url, requestData) {
		if(!requestData) {
			requestData = null;
		}

		var data = null;
		if(requestData instanceof Function) {
			data = requestData();
		}
		else if(requestData) {
			data = requestData;
		}
		else if(handler.requestForm) {
			data = getFormData(handler.requestForm);
		}

		data = "_nga.async=" + ((handler.async)?"true":"false") + "&" + data;

		var xmlreq = createXMLHttpRequest();
		var handleResponse = handler.handleResponse;

		if(handler.async) {
			xmlreq.onreadystatechange =function () {
				if(xmlreq.readyState == 4 ){
					handleResponse(xmlreq);
				}
			}
		}

		xmlreq.open(handler.method, url, handler.async);
		xmlreq.setRequestHeader('Content-Type',handler.contentType);
		xmlreq.send(data);

		if(!handler.async) {
			return handleResponse(xmlreq);
		}

		return false;

		/**
		 * XmlHttpRequest オブジェクトを作成する。
		 * @return XmlHttpRequest オブジェクト。
		 */
		function createXMLHttpRequest() {
			if(window.ActiveXObject) {
				try {
					return new ActiveXObject("Msxml2.XMLHTTP");
				}
				catch(e) {
					try {
						return new ActiveXObject("Microsoft.XMLHTTP");
					}
					catch (e2) {
						return null;
					}
				}
			}
			else if(window.XMLHttpRequest) {
				var xmlhttp = new XMLHttpRequest();
				return xmlhttp;

			}
			else {
				return null;
			}
		}

		/**
		 * 引数で指定された form オブジェクトから送信対象データ文字列を作成する。
		 * @param form 送信対象 form。
		 * @return 送信対象データ文字列。
		 */
		function getFormData(form) {
			var arr = new Array();
			var index = 0;
			var length = form.elements.length;
			for(var i = 0; i < length; i++) {
				var elem = form.elements[i];
				if(!elem.name) {
					continue;
				}

				if(elem.checked!=undefined) {
					if(elem.type=="radio" && !elem.checked) {
						continue;
					}
				}

				if(index > 0) {
					arr[index++] = "&";
				}
				arr[index++] = elem.name;
				arr[index++] = "=";
				arr[index++] = encodeURIComponent(elem.value);
			}
			return arr.join("");
		}
	}


	/**
	 * 受信データを表示する。
	 * @param xmlreq XMLHttpRequest オブジェクト。
	 * @return 通信が正常終了した場合は true。それ以外は false。
	 */
	this.handleResponse = function(xmlreq) {
		if(xmlreq.status!=200 || xmlreq.getResponseHeader("Content-Type").indexOf("text/xml")!=0) {
			w = window.open("", "", "width=600,height=400,top=0,left=0,resizable=yes,scrollbars=yes,status=no");
			w.document.write(xmlreq.responseText);
			w.document.close();
			return false;
		}

		var doc = xmlreq.responseXML.getElementsByTagName("c").item(0);
		new CongaResponse().parse(doc);
		return true;
	}

}
