/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * http://aipostyle.com/
 * 
 * Copyright(C) 2010 avanza Co.,Ltd. All rights reserved.
 * http://www.avnz.co.jp/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.aimluck.eip.intramail;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.services.upload.TurbineUpload;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.cayenne.om.portlet.AvzTIntramail;
import com.aimluck.eip.cayenne.om.portlet.AvzTIntramailFile;
import com.aimluck.eip.cayenne.om.portlet.AvzTIntramailMap;
import com.aimluck.eip.cayenne.om.portlet.auto._AvzTIntramail;
import com.aimluck.eip.cayenne.om.portlet.auto._AvzTIntramailFile;
import com.aimluck.eip.cayenne.om.security.TurbineUser;
import com.aimluck.eip.cayenne.om.security.auto._TurbineUser;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipConstants;
import com.aimluck.eip.common.ALEipManager;
import com.aimluck.eip.common.ALEipPost;
import com.aimluck.eip.common.ALEipUser;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.fileupload.util.FileuploadUtils;
import com.aimluck.eip.intramail.beans.FileuploadLiteBean;
import com.aimluck.eip.intramail.exceptions.InsertFileDataDeledateException;
import com.aimluck.eip.intramail.util.IntraMailUtils;
import com.aimluck.eip.intramail.util.IntraMailWhatsNewUtils;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.DatabaseOrmService;
import com.aimluck.eip.orm.query.ResultList;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.util.ALEipUtils;

/**
 * IntraMailのフォームデータを管理するクラスです。 <BR>
 * 
 */
public class IntraMailFormData extends ALAbstractFormData {

  /** logger */
  private static final JetspeedLogger logger =
    JetspeedLogFactoryService.getLogger(IntraMailFormData.class.getName());

  /** ログインユーザーのID * */
  private int login_user_id;

  /** ACL用の変数 * */
  private String aclPortletFeature;

  /** 回覧先一覧 */
  private List<ALEipUser> memberList;

  /** 件名 */
  private ALStringField subject = null;

  /** 本文 */
  private ALStringField body = null;

  /** 送信:T 下書き:F */
  private ALStringField send_flg;

  /** 添付ファイルリスト */
  private List<FileuploadLiteBean> fileuploadList = null;

  private String folderName = null;

  private String org_id;

  private ALStringField mail_type;

  private ALStringField org_mail_id;

  /** 現在日時 */
  private Date nowDate;

  /** 下書きトレイ or 受信／送信トレイかを判別するフラグ 下書き：true */
  private boolean isDraftEdit = false;

  /**
   * init
   * 
   * @param action
   * @param rundata
   * @param context
   * @see com.aimluck.eip.common.ALAbstractFormData#init(com.aimluck.eip.modules.actions.common.ALAction,
   *      org.apache.turbine.util.RunData, org.apache.velocity.context.Context)
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);

    login_user_id = ALEipUtils.getUserId(rundata);

    folderName = rundata.getParameters().getString("folderName");

    org_id = DatabaseOrmService.getInstance().getOrgId(rundata);

    nowDate = Calendar.getInstance().getTime();

    String s = rundata.getParameters().getString("draft_edit");
    if (s != null && IntraMailUtils.DRAFT_EDIT_FORM.equals(s)) {
      isDraftEdit = true;
    }
  }

  /**
   * 各フィールドを初期化します。 <BR>
   * 
   * @see com.aimluck.eip.common.ALData#initField()
   */
  public void initField() {

    // Subject
    subject = new ALStringField();
    subject.setFieldName("件名");
    subject.setTrim(true);

    // Body
    body = new ALStringField();
    body.setFieldName("本文");
    body.setTrim(false);

    send_flg = new ALStringField();

    mail_type = new ALStringField();

    org_mail_id = new ALStringField();

    // 回覧先一覧
    memberList = new ArrayList<ALEipUser>();

    fileuploadList = new ArrayList();
  }

  /**
   * の各フィールドに対する制約条件を設定します。 <BR>
   * 
   * @see com.aimluck.eip.common.ALAbstractFormData#setValidator()
   */
  @Override
  protected void setValidator() {

    // 件名の文字数制限
    subject.limitMaxLength(IntraMailUtils.FIELD_SUBJECT_MAX_LEN);

    // 本文の文字数制限
    body.limitMaxLength(IntraMailUtils.FIELD_BODY_MAX_LEN);

  }

  /**
   * のフォームに入力されたデータの妥当性検証を行います。 <BR>
   * 
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#validate(java.util.ArrayList)
   */
  @Override
  protected boolean validate(List<String> msgList) {

    try {
      subject.validate(msgList);
      body.validate(msgList);

      if (!isDraftEdit()) {
        if (memberList == null || memberList.size() <= 0) {
          msgList.add("『 <span class='em'>宛先</span> 』を指定してください。");
        }
      }
    } catch (Exception e) {
      logger.error("Exception", e);
    }
    return (msgList.size() == 0);
  }

  /**
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return
   * @see com.aimluck.eip.common.ALAbstractFormData#setFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean setFormData(RunData rundata, Context context,
      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {
    boolean res = super.setFormData(rundata, context, msgList);
    if (res) {
      try {
        String userNames[] = rundata.getParameters().getStrings("members");
        if (userNames != null && userNames.length > 0) {

          SelectQuery<TurbineUser> query = Database.query(TurbineUser.class);
          Expression exp1 =
            ExpressionFactory
              .inExp(_TurbineUser.LOGIN_NAME_PROPERTY, userNames);
          Expression exp2 =
            ExpressionFactory.matchExp(_TurbineUser.DISABLED_PROPERTY, "F");
          query.setQualifier(exp1);
          query.andQualifier(exp2);

          ResultList<TurbineUser> list = query.getResultList();

          TurbineUser record = null;
          int length = userNames.length;
          for (int i = 0; i < length; i++) {
            record = getEipUserRecord(list, userNames[i]);
            ALEipUser user = new ALEipUser();
            user.initField();
            user.setUserId(record.getUserId().intValue());
            user.setName(record.getLoginName());
            user.setAliasName(record.getFirstName(), record.getLastName());
            memberList.add(user);
          }
        }

        boolean migrate = false;
        if (ALEipConstants.MODE_INSERT.equals(getMode())) {
          // 送信 or 保存ボタン押下時処理
          if (!"".equals(mail_type.getValue())) {

            // 添付ファイル付きメールで、「返信する」「全員へ返信する」「転送する」のいずれかを
            // 行なう場合、添付ファイルを元のメールから新しいメールへコピーする必要があります。

            String[] fileids =
              rundata.getParameters().getStrings("attachments");
            if (fileids != null && !"".equals(fileids[0])) {

              File saveFolder =
                new File(FileuploadUtils.getRootFolder(org_id, login_user_id)
                  + File.separator
                  + folderName);
              if (!saveFolder.exists()) {
                saveFolder.mkdirs();
              }

              for (String fileid : fileids) {

                if (!fileid.trim().startsWith("s")) {
                  // 先頭が"s"でなければ、新規追加ファイル
                  continue;
                }
                // 元メールの登録済みのファイルを、一時ファイル置き場にコピーします。
                // fileid = fileid.trim().substring(1);

                AvzTIntramailFile file =
                  IntraMailUtils.getAvzTIntramailFile(Integer.parseInt(fileid
                    .trim()
                    .substring(1)));
                if (file == null) {
                  continue;
                }
                String org_file_path =
                  IntraMailUtils.getSaveDirPath(DatabaseOrmService
                    .getInstance()
                    .getOrgId(rundata), file.getOwnerId().intValue())
                    + file.getFilePath();
                // System.out.println("★★ファイル移行元" + org_file_path);

                String dest_file_path = saveFolder + File.separator + fileid;
                // System.out.println("★★ファイル移行先" + dest_file_path);

                // IntraMailUtils.copyFileBinaly(new File(org_file_path), new
                // File(dest_file_path));
                FileuploadUtils.copyFile(new File(org_file_path), new File(
                  dest_file_path));

                // 一時添付ファイル名の保存("1.txt"など)
                createTempNameFile(dest_file_path, file);

                migrate = true;
              }
            }
          }
        }

        // fileuploadList = FileuploadUtils.getFileuploadList(rundata);
        fileuploadList = IntraMailUtils.getFileuploadList(rundata, migrate);

      } catch (Exception ex) {
        deleteTempFile();
        logger.error("Exception", ex);
      }
    }
    return res;
  }

  /**
   * 既存メールから取得した一時ファイルを削除します（sX、sX.txt）。
   */
  private void deleteTempFile() {

    File dest_file_path =
      new File(FileuploadUtils.getRootFolder(org_id, login_user_id)
        + File.separator
        + folderName);
    if (!dest_file_path.exists()) {
      return;
    }

    if (fileuploadList == null) {
      return;
    }

    for (FileuploadLiteBean file : fileuploadList) {
      if (!file.getFileId().startsWith("s")) {
        continue;
      }
      File df = new File(dest_file_path + File.separator + file.getFileId());
      if (df.exists() && df.isFile()) {
        df.delete();
      }

      File nf =
        new File(dest_file_path
          + File.separator
          + file.getFileId()
          + FileuploadUtils.EXT_FILENAME);
      if (nf.exists() && nf.isFile()) {
        nf.delete();
      }
    }
  }

  /**
   * ファイル名を保存するファイルを作成します（例 s1.txt）
   * 
   * @param dest_file_path
   * @param file
   * @throws UnsupportedEncodingException
   * @throws FileNotFoundException
   */
  private void createTempNameFile(String dest_file_path, AvzTIntramailFile file)
      throws UnsupportedEncodingException, FileNotFoundException {
    PrintWriter writer = null;
    try {
      // 一時添付ファイル名の保存
      writer =
        new PrintWriter(new OutputStreamWriter(
          new FileOutputStream(dest_file_path + FileuploadUtils.EXT_FILENAME),
          FileuploadUtils.FILE_ENCODING));
      writer.println(file.getFileName());
      writer.flush();
    } catch (UnsupportedEncodingException e) {
      e.printStackTrace();
      throw e;
    } catch (FileNotFoundException e) {
      e.printStackTrace();
      throw e;
    } finally {
      if (writer != null) {
        writer.close();
      }
    }
  }

  /**
   * をデータベースから読み出します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#loadFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context)
   */
  @Override
  protected boolean loadFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      // オブジェクトモデルを取得
      AvzTIntramail mail = IntraMailUtils.getAvzTIntramail(rundata, context);
      if (mail == null) {
        return false;
      }

      if (isDraftEdit()) {
        // 下書き編集時
        subject.setValue(mail.getSubject());
        body.setValue(mail.getBody());
        // List<AvzTIntramailMap> list = mail.getAvzTIntramailMapArray();
        List<AvzTIntramailMap> list =
          IntraMailUtils.getAvzTIntramailMap(
            mail.getIntramailId(),
            IntraMailUtils.TRANSMISSION_CLS_RECEIVE);
        if (list != null) {
          for (AvzTIntramailMap m : list) {
            // if( "S".equals(m.getTransmissionCls()) ) {
            // continue;
            // }
            memberList.add(ALEipUtils.getALEipUser(m.getUserId()));
          }
        }
      } else {
        mail_type.setValue(rundata.getParameters().getString("mailtype"));
        if (IntraMailUtils.MAIL_TYPE_REPLY.equals(mail_type.getValue())) {
          // 返信する
          subject.setValue(IntraMailUtils.PREFIX_REPLY + mail.getSubject());
          memberList.add(ALEipUtils.getALEipUser(mail.getOwnerId()));
        } else if (IntraMailUtils.MAIL_TYPE_ALL_REPLY.equals(mail_type
          .getValue())) {
          // 全員へ返信する
          subject.setValue(IntraMailUtils.PREFIX_REPLY + mail.getSubject());
          List<AvzTIntramailMap> list = mail.getAvzTIntramailMapArray();
          for (AvzTIntramailMap m : list) {
            // if( "S".equals(m.getTransmissionCls()) ) {
            // continue;
            // }
            boolean exist = false;
            for (ALEipUser u : memberList) {
              if (m.getUserId().intValue() == (int) u.getUserId().getValue()) {
                exist = true;
                break;
              }
            }
            if (!exist) {
              memberList.add(ALEipUtils.getALEipUser(m.getUserId()));
            }
          }
        } else {
          // 転送する
          subject.setValue(IntraMailUtils.PREFIX_FORWARD + mail.getSubject());
        }
        body.setValue(IntraMailUtils.HEADER_ORIGINAL_MESSAGE + mail.getBody());

        // 「返信する」「全員へ返信する」「転送する」で使用します。
        org_mail_id.setValue(mail.getIntramailId().toString());
      }

      // ファイル
      SelectQuery<AvzTIntramailFile> query =
        Database.query(AvzTIntramailFile.class);

      query.andQualifier(ExpressionFactory.matchDbExp(
        _AvzTIntramailFile.TO_AVZ_TINTRAMAIL_PROPERTY
          + "."
          + _AvzTIntramail.INTRAMAIL_ID_PK_COLUMN,
        mail.getIntramailId()));

      List<AvzTIntramailFile> tmpList = query.getResultList();
      for (AvzTIntramailFile file : tmpList) {
        FileuploadLiteBean fbean = new FileuploadLiteBean();
        fbean.initField();
        fbean.setFileId(String.valueOf(file.getFileId()));
        fbean.setFileName(file.getFileName());
        fileuploadList.add(fbean);
      }

    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * 送信 or 保存ボタン押下時処理
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#insertFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean insertFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      // 新規オブジェクトモデル
      AvzTIntramail mail = Database.create(AvzTIntramail.class);

      // リクエスト名
      mail.setOwnerId(login_user_id);

      if ("".equals(subject.getValue())) {
        subject.setValue(IntraMailUtils.NO_TITLE);
      }
      mail.setSubject(subject.getValue());

      mail.setBody(body.getValue());

      mail.setSendFlg(send_flg.getValue());

      mail.setDisabled(IntraMailUtils.AVAILABLE);

      mail.setMailVolume(0);

      // 作成日
      mail.setCreateDate(nowDate);
      // 更新日
      mail.setUpdateDate(nowDate);

      // 送信者レコード追加
      insertAvzTIntramailMap(
        mail,
        login_user_id,
        IntraMailUtils.TRANSMISSION_CLS_SEND,
        IntraMailUtils.READ_FLG_READ,
        send_flg.getValue());

      // 受信者レコード追加
      int size = memberList.size();
      for (int i = 0; i < size; i++) {
        insertAvzTIntramailMap(
          mail,
          (int) memberList.get(i).getUserId().getValue(),
          IntraMailUtils.TRANSMISSION_CLS_RECEIVE,
          IntraMailUtils.READ_FLG_UNREAD,
          send_flg.getValue());
      }

      // 添付ファイル処理
      if (!IntraMailUtils.insertFileDataDelegate(
        rundata,
        context,
        mail,
        fileuploadList,
        folderName,
        msgList)) {
        deleteTempFile();
        throw new InsertFileDataDeledateException();
      }

      // リクエストを登録
      Database.commit();

      // 添付ファイル保存先のフォルダを削除
      File folder =
        FileuploadUtils.getFolder(org_id, login_user_id, folderName);
      FileuploadUtils.deleteFolder(folder);

      // 新着ポートレット登録
      if (IntraMailUtils.VIEW_FLG_VISIBLE.equals(send_flg.getValue())) {
        int target_size = memberList.size();
        for (int i = 0; i < target_size; i++) {
          ALEipUser user = memberList.get(i);
          if (user.getUserId().getValue() != login_user_id) {
            IntraMailWhatsNewUtils.insertWhatsNew(
              IntraMailWhatsNewUtils.WHATS_NEW_TYPE_INTRAMAIL,
              mail.getIntramailId(),
              (int) memberList.get(i).getUserId().getValue());
          }
        }
        // リクエストを登録
        Database.commit();
      }

    } catch (InsertFileDataDeledateException ex) {
      Database.rollback();
      logger.debug("Exception", ex);
      return false;
    } catch (Exception ex) {
      Database.rollback();
      ex.printStackTrace();
      logger.error("Exception", ex);
      return false;
    } finally {
      // 添付ファイル保存先のフォルダを削除
      // File folder = FileuploadUtils.getFolder(org_id, login_user_id,
      // folderName);
      // FileuploadUtils.deleteFolder(folder);
    }
    return true;
  }

  /**
   * AvzTIntramailMapの登録処理
   * 
   * @param request
   * @param user
   * @param status
   */
  private void insertAvzTIntramailMap(AvzTIntramail request, int userid,
      String transmissionCls, String read_flg, String view_flg) {
    AvzTIntramailMap map = Database.create(AvzTIntramailMap.class);
    map.setToAvzTIntramail(request);
    map.setUserId(Integer.valueOf(userid));
    map.setTransmissionCls(transmissionCls);
    map.setReadFlg(read_flg);
    map.setViewFlg(view_flg);
    map.setDisabled(IntraMailUtils.AVAILABLE);
    map.setCreateDate(nowDate);
    map.setUpdateDate(nowDate);
  }

  private void deleteAvzTIntramailMap(AvzTIntramail mail) {
    List<AvzTIntramailMap> list = mail.getAvzTIntramailMapArray();
    int listsize = list.size();
    for (int i = 0; i < listsize; i++) {
      AvzTIntramailMap map = list.get(i);
      Database.delete(map);
    }
  }

  /**
   * データベースに格納されているを更新します。 <BR>
   * 下書きを開いて、送信 or 保存する処理
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#updateFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean updateFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      AvzTIntramail mail = IntraMailUtils.getAvzTIntramail(rundata, context);
      if (mail == null) {
        return false;
      }

      if ("".equals(subject.getValue())) {
        subject.setValue(IntraMailUtils.NO_TITLE);
      }
      mail.setSubject(subject.getValue());

      mail.setBody(body.getValue());

      mail.setSendFlg(send_flg.getValue());

      // 更新日
      mail.setUpdateDate(nowDate);

      deleteAvzTIntramailMap(mail);

      // 送信者レコード追加
      insertAvzTIntramailMap(
        mail,
        login_user_id,
        IntraMailUtils.TRANSMISSION_CLS_SEND,
        IntraMailUtils.READ_FLG_READ,
        send_flg.getValue());

      // 受信者レコード追加
      int size = memberList.size();
      for (int i = 0; i < size; i++) {
        insertAvzTIntramailMap(
          mail,
          (int) memberList.get(i).getUserId().getValue(),
          IntraMailUtils.TRANSMISSION_CLS_RECEIVE,
          IntraMailUtils.READ_FLG_UNREAD,
          send_flg.getValue());
      }

      // 添付ファイル処理
      if (!IntraMailUtils.insertFileDataDelegate(
        rundata,
        context,
        mail,
        fileuploadList,
        folderName,
        msgList)) {
        deleteTempFile();
        throw new InsertFileDataDeledateException();
      }
      // リクエストを登録
      Database.commit();

      // 添付ファイル保存先のフォルダを削除
      File folder =
        FileuploadUtils.getFolder(org_id, login_user_id, folderName);
      FileuploadUtils.deleteFolder(folder);

      // 新着ポートレット登録
      if (IntraMailUtils.VIEW_FLG_VISIBLE.equals(send_flg.getValue())) {
        int target_size = memberList.size();
        for (int i = 0; i < target_size; i++) {
          ALEipUser user = memberList.get(i);
          if (user.getUserId().getValue() != login_user_id) {
            IntraMailWhatsNewUtils.insertWhatsNew(
              IntraMailWhatsNewUtils.WHATS_NEW_TYPE_INTRAMAIL,
              mail.getIntramailId(),
              (int) memberList.get(i).getUserId().getValue());
          }
        }
        // リクエストを登録
        Database.commit();
      }

    } catch (ALPageNotFoundException ex) {
      Database.rollback();
      ex.printStackTrace();
      logger.error("Exception", ex);
      return false;
    } catch (InsertFileDataDeledateException ex) {
      Database.rollback();
      logger.debug("Exception", ex);
      return false;
    } catch (Exception ex) {
      Database.rollback();
      ex.printStackTrace();
      logger.error("Exception", ex);
      return false;
    } finally {
      // 添付ファイル保存先のフォルダを削除
      // File folder = FileuploadUtils.getFolder(org_id, login_user_id,
      // folderName);
      // FileuploadUtils.deleteFolder(folder);
    }
    return true;
  }

  /**
   * リクエストをデータベースから削除します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#deleteFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context)
   */
  @Override
  protected boolean deleteFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      if (isDraftEdit()) {
        // 下書きの削除の場合は、全て削除
        // オブジェクトモデルを取得
        AvzTIntramail entity =
          IntraMailUtils.getAvzTIntramail(rundata, context);
        if (entity == null) {
          return false;
        }
        IntraMailUtils.deleteAvzTIntramail(rundata, context, entity);
      } else {
        // 受信・送信レコードの削除の場合
        // 最後のレコードを判定して、物理削除を行なうか迷いどころです。

        AvzTIntramailMap entity =
          IntraMailUtils.getAvzTIntramailMap(rundata, context);
        if (entity == null) {
          return false;
        }
        /*
         * entity.setDisabled(IntraMailUtils.DISABLED);
         */

        if (IntraMailUtils.isLastRecord(entity)) {
          // 最後のAvzTIntramailMapの場合は、親レコード（AvzTIntramail）
          // 共々物理削除
          IntraMailUtils.deleteAvzTIntramail(rundata, context, entity
            .getToAvzTIntramail());
        } else {
          // 最後でないなら論理削除
          entity.setDisabled(IntraMailUtils.DISABLED);
        }

      }
      Database.commit();
    } catch (Exception ex) {
      Database.rollback();
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * 指定したユーザ名のオブジェクトを取得する．
   * 
   * @param userList
   * @param userName
   * @return
   */
  private TurbineUser getEipUserRecord(List<TurbineUser> userList,
      String userName) {
    int size = userList.size();
    for (int i = 0; i < size; i++) {
      TurbineUser record = userList.get(i);
      if (record.getLoginName().equals(userName)) {
        return record;
      }
    }
    return null;
  }

  /**
   * 本文取得処理
   * 
   * @return
   */
  public ALStringField getBody() {
    return body;
  }

  /**
   * 件名取得処理
   * 
   * @return
   */
  public ALStringField getSubject() {
    return subject;
  }

  /**
   * グループメンバーを取得します。 <BR>
   * 
   * @return
   */
  public List<ALEipUser> getMemberList() {
    return memberList;
  }

  /**
   * グループ名に紐づくユーザーリスト取得
   * 
   * @param groupname
   * @return
   */
  public List<ALEipUser> getUsers(String groupname) {
    return ALEipUtils.getUsers(groupname);
  }

  /**
   * 部署リスト取得
   * 
   * @return
   */
  public Map<Integer, ALEipPost> getPostMap() {
    return ALEipManager.getInstance().getPostMap();
  }

  /**
   * 添付ファイルリスト取得
   * 
   * @return
   */
  public List<FileuploadLiteBean> getAttachmentFileNameList() {
    return fileuploadList;
  }

  /**
   * アクセス権限チェック用メソッド。<br />
   * アクセス権限の機能名を返します。
   * 
   * @return
   */
  @Override
  public String getAclPortletFeature() {
    return aclPortletFeature;
  }

  /**
   * 下書きトレイか？
   * 
   * @return
   */
  public boolean isDraftEdit() {
    return isDraftEdit;
  }

  public void setDraftEdit(boolean isDraftEdit) {
    this.isDraftEdit = isDraftEdit;
  }

  /**
   * アップロードファイルの最大容量（合計）
   * 
   * @return
   */
  public int getMaxMailVolume() {
    return TurbineUpload.getSizeMax() / 1024 / 1024;
  }

  /**
   * メールタイプ（返信、全員へ返信、転送）
   * 
   * @return
   */
  public String getMailType() {
    return mail_type.getValue();
  }

  /**
   * 元メールＩＤ（返信、全員へ返信、転送で使用）
   * 
   * @return
   */
  public String getOrgMailId() {
    return org_mail_id.getValue();
  }

}
