﻿using System;
using System.Collections.Generic;
using System.Text;
using IrrlichtNetSwig;

namespace Collision
{
    class Program
    {
        static void Main(string[] args)
        {
            // let user select driver type

            E_DRIVER_TYPE driverType;

            System.Console.WriteLine(
                @"Please select the driver you want for this example:
(a) Direct3D 9.0c 
(b) Direct3D 8.1
(c) OpenGL 1.5
(d) Software Renderer
(e) Burning's Software Renderer
(f) NullDevice
(otherKey) exit");

            ConsoleKeyInfo consoleKeyInfo = Console.ReadKey(true);

            switch (consoleKeyInfo.KeyChar)
            {
                case 'a':
                    driverType = E_DRIVER_TYPE.EDT_DIRECT3D9;
                    break;
                case 'b':
                    driverType = E_DRIVER_TYPE.EDT_DIRECT3D8;
                    break;
                case 'c':
                    driverType = E_DRIVER_TYPE.EDT_OPENGL;
                    break;
                case 'd':
                    driverType = E_DRIVER_TYPE.EDT_SOFTWARE;
                    break;
                case 'e':
                    driverType = E_DRIVER_TYPE.EDT_BURNINGSVIDEO;
                    break;
                case 'f':
                    driverType = E_DRIVER_TYPE.EDT_NULL;
                    break;
                default:
                    return;
            }

            // create device

            IrrlichtDevice device =
                IrrlichtNet.createDevice(driverType, new dimension2di(640, 480), 16, false);

            if (device == null)
                return; // could not create selected driver.

            IVideoDriver driver = device.getVideoDriver();
            ISceneManager smgr = device.getSceneManager();

            device.getFileSystem().addZipFileArchive("../../media/map-20kdm2.pk3");

            IAnimatedMesh q3levelmesh = smgr.getMesh("20kdm2.bsp");
            ISceneNode q3node = null;

            if (q3levelmesh != null)
                q3node = smgr.addOctTreeSceneNode(q3levelmesh.getMesh(0));

            /*
            So far so good, we've loaded the quake 3 level like in tutorial 2. Now,
            here comes something different: We create a triangle selector. A
            triangle selector is a class which can fetch the triangles from scene
            nodes for doing different things with them, for example collision
            detection. There are different triangle selectors, and all can be
            created with the ISceneManager. In this example, we create an
            OctTreeTriangleSelector, which optimizes the triangle output a little
            bit by reducing it like an octree. This is very useful for huge meshes
            like quake 3 levels. After we created the triangle selector, we attach
            it to the q3node. This is not necessary, but in this way, we do not
            need to care for the selector, for example dropping it after we do not
            need it anymore.
            */

            ITriangleSelector selector = null;

            if (q3node != null)
            {
                q3node.setPosition(new vector3df(-1350, -130, -1400));

                selector = smgr.createOctTreeTriangleSelector(
                    q3levelmesh.getMesh(0), q3node, 128);
                q3node.setTriangleSelector(selector);
            }


            /*
            We add a first person shooter camera to the scene for being able to
            move in the quake 3 level like in tutorial 2. But this, time, we add a
            special animator to the camera: A Collision Response animator. This
            animator modifies the scene node to which it is attached to in order to
            prevent it moving through walls, and to add gravity to it. The
            only thing we have to tell the animator is how the world looks like,
            how big the scene node is, how much gravity to apply and so on. After the 
            collision response animator is attached to the camera, we do not have to do
            anything more for collision detection, anything is done automatically,
            all other collision detection code below is for picking. And please
            note another cool feature: The collision response animator can be
            attached also to all other scene nodes, not only to cameras. And it can
            be mixed with other scene node animators. In this way, collision
            detection and response in the Irrlicht engine is really, really easy.

            Now we'll take a closer look on the parameters of
            createCollisionResponseAnimator(). The first parameter is the
            TriangleSelector, which specifies how the world, against collision
            detection is done looks like. The second parameter is the scene node,
            which is the object, which is affected by collision detection, in our
            case it is the camera. The third defines how big the object is, it is
            the radius of an ellipsoid. Try it out and change the radius to smaller
            values, the camera will be able to move closer to walls after this. The
            next parameter is the direction and speed of gravity.  We'll set it to 
            (0, -10, 0), which approximates to realistic gravity, assuming that our
            units are metres. You could set it to (0,0,0) to disable gravity. And the 
            last value is just a translation: Without this, the ellipsoid with which 
            collision detection is done would be around the camera, and the camera would 
            be in the middle of the ellipsoid. But as human beings, we are used to have our
            eyes on top of the body, with which we collide with our world, not in
            the middle of it. So we place the scene node 50 units over the center
            of the ellipsoid with this parameter. And that's it, collision
            detection works now.
            */

            // Set a jump speed of 3 units per second, which gives a fairly realistic jump
            // when used with the gravity of (0, -10, 0) in the collision response animator.
            ICameraSceneNode camera =
                smgr.addCameraSceneNodeFPS(null, 100.0f, .3f, -1, null, 0, true, 3.0f);
            camera.setPosition(new vector3df(-100, 50, -150));

            if (selector != null)
            {
                ISceneNodeAnimator anim = smgr.createCollisionResponseAnimator(
                    selector, camera, new vector3df(30, 50, 30),
                    new vector3df(0, -10, 0),
                    new vector3df(0, 50, 0));
                camera.addAnimator(anim);
            }

            /*
            Because collision detection is no big deal in irrlicht, I'll describe how to
            do two different types of picking in the next section. But before this,
            I'll prepare the scene a little. I need three animated characters which we
            could pick later, a dynamic light for lighting them,
            a billboard for drawing where we found an intersection,	and, yes, I need to
            get rid of this mouse cursor. :)
            */

            // disable mouse cursor

            device.getCursorControl().setVisible(false);

            // add billboard

            IBillboardSceneNode bill = smgr.addBillboardSceneNode();
            bill.setMaterialType(E_MATERIAL_TYPE.EMT_TRANSPARENT_ADD_COLOR);
            bill.setMaterialTexture(0, driver.getTexture("../../media/particle.bmp"));
            bill.setMaterialFlag(E_MATERIAL_FLAG.EMF_LIGHTING, false);
            bill.setMaterialFlag(E_MATERIAL_FLAG.EMF_ZBUFFER, false);
            bill.setSize(new dimension2df(20.0f, 20.0f));

            // add 3 animated faeries.

            SMaterial material = new SMaterial();
            material.setTexture(0, driver.getTexture("../../media/faerie2.bmp"));
            material.Lighting = true;

            IAnimatedMeshSceneNode node = null;
            IAnimatedMesh faerie = smgr.getMesh("../../media/faerie.md2");

            if (faerie != null)
            {
                node = smgr.addAnimatedMeshSceneNode(faerie);
                node.setPosition(new vector3df(-70, 0, -90));
                node.setMD2Animation(EMD2_ANIMATION_TYPE.EMAT_RUN);
                node.getMaterial(0).CopyFrom(material);

                node = smgr.addAnimatedMeshSceneNode(faerie);
                node.setPosition(new vector3df(-70, 0, -30));
                node.setMD2Animation(EMD2_ANIMATION_TYPE.EMAT_SALUTE);
                node.getMaterial(0).CopyFrom(material);

                node = smgr.addAnimatedMeshSceneNode(faerie);
                node.setPosition(new vector3df(-70, 0, -60));
                node.setMD2Animation(EMD2_ANIMATION_TYPE.EMAT_JUMP);
                node.getMaterial(0).CopyFrom(material);
            }

            material.setTexture(0, null);
            material.Lighting = false;

            // Add a light

            smgr.addLightSceneNode(null, new vector3df(-60, 100, 400),
                                   new SColorf(1.0f, 1.0f, 1.0f, 1.0f),
                                   600.0f);


            /*
            For not making it to complicated, I'm doing picking inside the drawing
            loop. We take two pointers for storing the current and the last
            selected scene node and start the loop.
            */


            ISceneNode selectedSceneNode = null;
            ISceneNode lastSelectedSceneNode = null;


            int lastFPS = -1;

            while (device.run())
                if (device.isWindowActive())
                {
                    driver.beginScene(true, true, new SColor());

                    smgr.drawAll();

                    /*
                    After we've drawn the whole scene with smgr.drawAll(), we'll
                    do the first picking: We want to know which triangle of the
                    world we are looking at. In addition, we want the exact point
                    of the quake 3 level we are looking at. For this, we create a
                    3d line starting at the position of the camera and going
                    through the lookAt-target of it. Then we ask the collision
                    manager if this line collides with a triangle of the world
                    stored in the triangle selector. If yes, we draw the 3d
                    triangle and set the position of the billboard to the
                    intersection point.
                    */

                    line3df line = new line3df();
                    line.start = camera.getPosition();
                    line.end = line.start + (camera.getTarget() - line.start).normalize()*1000.0f;

                    vector3df intersection = new vector3df();
                    triangle3df tri = new triangle3df();

                    if (smgr.getSceneCollisionManager().getCollisionPoint(
                        line, selector, intersection, tri))
                    {
                        bill.setPosition(intersection);

                        driver.setTransform(E_TRANSFORMATION_STATE.ETS_WORLD, new CMatrix4f());
                        driver.setMaterial(material);
                        driver.draw3DTriangle(tri, new SColor(0, 255, 0, 0));
                    }


                    /*
                    Another type of picking supported by the Irrlicht Engine is
                    scene node picking based on bounding boxes. Every scene node has
                    got a bounding box, and because of that, it's very fast for
                    example to get the scene node which the camera looks at. Again,
                    we ask the collision manager for this, and if we've got a scene
                    node, we highlight it by disabling Lighting in its material, if
                    it is not the billboard or the quake 3 level.
                    */

                    selectedSceneNode =
                        smgr.getSceneCollisionManager().getSceneNodeFromCameraBB(camera);

                    if (lastSelectedSceneNode != null)
                        lastSelectedSceneNode.setMaterialFlag(E_MATERIAL_FLAG.EMF_LIGHTING, true);

                    if (selectedSceneNode == q3node || selectedSceneNode == bill)
                        selectedSceneNode = null;

                    if (selectedSceneNode != null)
                        selectedSceneNode.setMaterialFlag(E_MATERIAL_FLAG.EMF_LIGHTING, false);

                    lastSelectedSceneNode = selectedSceneNode;


                    /*
                    That's it, we just have to finish drawing.
                    */

                    driver.endScene();

                    int fps = driver.getFPS();

                    if (lastFPS != fps)
                    {
                        string str = "Collision detection example - Irrlicht Engine [";
                        str += driver.getName();
                        str += "] FPS:";
                        str += fps.ToString();

                        device.setWindowCaption(str);
                        lastFPS = fps;
                    }
                }


            return;
        }
    }
}
