package jp.sourceforge.jetspeedjapan.portlet.weblog;

/* ====================================================================================
 * Copyright (c) 2003, jetspeed-japan.sourceforge.jp
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *     * Redistributions of source code must retain the above copyright notice,
 *     * this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *     * notice, this list of conditions and the following disclaimer in the
 *     * documentation and/or other materials provided with the distribution.
 *     * Neither the name of the jetspeed-japan.sourceforge.jp nor the names 
 *       of its contributors
 *     * may be used to endorse or promote products derived from this software
 *     * without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================================
 */

import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;

import jp.sourceforge.jetspeedjapan.portlet.weblog.om.Weblog;
import jp.sourceforge.jetspeedjapan.portlet.weblog.om.WeblogMessage;
import jp.sourceforge.jetspeedjapan.portlet.weblog.om.WeblogMessagePeer;
import jp.sourceforge.jetspeedjapan.portlet.weblog.om.WeblogPeer;
import jp.sourceforge.jetspeedjapan.portlet.weblog.om.utils.PageList;
import jp.sourceforge.jetspeedjapan.portlet.weblog.util.CalendarUtil;
import jp.sourceforge.jetspeedjapan.portlet.weblog.util.WebLogUtil;

import org.apache.jetspeed.portal.portlets.VelocityPortlet;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.jetspeed.util.PortletConfigState;
import org.apache.torque.Torque;
import org.apache.torque.TorqueException;
import org.apache.torque.util.Criteria;
import org.apache.turbine.util.DynamicURI;
import org.apache.turbine.util.ParameterParser;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

/**
 * This class is a controller of WebLogEditorPortletAction class.
 * 
 * @author <a href="mailto:shinsuke@users.sourceforge.jp">Shinsuke Sugaya</a>
 */
public class WebLogManager
{
    /**
     * Static initialization of the logger for this class
     */
    private static final JetspeedLogger logger = JetspeedLogFactoryService.getLogger(WebLogManager.class.getName());

    public static String DEFAULT_FRAME_BACKGROUND_COLOR = "#ffcc00";
    public static String DEFAULT_FRAME_TEXT_COLOR = "#000000";
    public static String DEFAULT_CONTENT_BACKGROUND_COLOR = "#ffffff";
    public static String DEFAULT_CONTENT_TEXT_COLOR = "#000000";
    public static String DEFAULT_TARGET_DAY_BACKGROUND_COLOR = "#ff3399";
    public static String DEFAULT_TARGET_DAY_TEXT_COLOR = "#ffffff";
    public static String DEFAULT_DAY_BACKGROUND_COLOR = "#ffffff";
    public static String DEFAULT_DAY_TEXT_COLOR = "#000000";
    public static String DEFAULT_SUNDAY_BACKGROUND_COLOR = "#ffccff";
    public static String DEFAULT_SUNDAY_TEXT_COLOR = "#ff0000";
    public static String DEFAULT_SATURDAY_BACKGROUND_COLOR = "#ccffff";
    public static String DEFAULT_SATURDAY_TEXT_COLOR = "#0000ff";

    private Context context;
    private RunData rundata;
    private ParameterParser params;

    private String userName;
    private int year;
    private int month;
    private int day;
    private String title;
    private String message;
    private String formatType;
    private int weblogMessageId;
    private String currentUrl;
    private int page;
    private int size;
    private int topicSize;

    private String type;

    private ArrayList infoList;
    private ArrayList errorList;

    private String frameBackgroundColor;
    private String frameTextColor;
    private String contentBackgroundColor;
    private String contentTextColor;
    private String targetdayBackgroundColor;
    private String targetdayTextColor;
    private String dayBackgroundColor;
    private String dayTextColor;
    private String sundayBackgroundColor;
    private String sundayTextColor;
    private String saturdayBackgroundColor;
    private String saturdayTextColor;

    /**
     * Default Constractor for WebLogManager class.
     * 
     * @param context
     * @param rundata
     * @param username
     */
    public WebLogManager(Context context, RunData rundata, String username)
    {
        this.context = context;
        this.rundata = rundata;
        this.params = rundata.getParameters();
        infoList = new ArrayList();
        errorList = new ArrayList();

        setUserName(username);
        setYear(params.getInt(WebLogUtil.YEAR, -1));
        setMonth(params.getInt(WebLogUtil.MONTH, -1));
        setDay(params.getInt(WebLogUtil.DAY, -1));
        setTitle(params.getString(WebLogUtil.TITLE, null));
        setMessage(params.getString(WebLogUtil.MESSAGE, null));
        setFormatType(params.getString(WebLogUtil.FORMATTYPE, null));
        setWeblogMessageId(params.getInt(WebLogUtil.WEBLOGMESSAGEID, 0));
        setPage(params.getInt(WebLogUtil.WEBLOG_PAGE, 1));
        setSize(params.getInt(WebLogUtil.WEBLOG_SIZE, 5));

        // Setup URL
        String url = DynamicURI.toString(rundata);
        int pos = url.indexOf("?");
        if (pos > -1)
        {
            setCurrentUrl(url.substring(0, pos));
        }
        else
        {
            setCurrentUrl(url);
        }

        // Current date
        Calendar cal = Calendar.getInstance();

        if (this.getYear() == -1 || this.getMonth() == -1 || this.getDay() == -1)
        {
            this.setYear(cal.get(Calendar.YEAR));
            this.setMonth(cal.get(Calendar.MONTH) + 1);
            this.setDay(cal.get(Calendar.DAY_OF_MONTH));
        }
        else
        {
            cal.set(this.getYear(), this.getMonth() - 1, this.getDay());
            setYear(cal.get(Calendar.YEAR));
            setMonth(cal.get(Calendar.MONTH) + 1);
            setDay(cal.get(Calendar.DAY_OF_MONTH));
        }

        // If username is none, change to null
        if (getUserName() != null && getUserName().equals(""))
        {
            setUserName(null);
        }

        VelocityPortlet portlet = (VelocityPortlet) context.get("portlet");
        if (portlet != null)
        {
            // Frame Color
            this.setFrameBackgroundColor(
                (String) PortletConfigState.getParameter(
                    portlet,
                    rundata,
                    WebLogUtil.FRAME_BACKGROUND_COLOR,
                    DEFAULT_FRAME_BACKGROUND_COLOR));
            context.put(WebLogUtil.FRAME_BACKGROUND_COLOR, this.getFrameBackgroundColor());
            this.setFrameTextColor(
                (String) PortletConfigState.getParameter(
                    portlet,
                    rundata,
                    WebLogUtil.FRAME_TEXT_COLOR,
                    DEFAULT_FRAME_TEXT_COLOR));
            context.put(WebLogUtil.FRAME_TEXT_COLOR, this.getFrameTextColor());
            // Contents Color
            this.setContentBackgroundColor(
                (String) PortletConfigState.getParameter(
                    portlet,
                    rundata,
                    WebLogUtil.CONTENT_BACKGROUND_COLOR,
                    DEFAULT_CONTENT_BACKGROUND_COLOR));
            context.put(WebLogUtil.CONTENT_BACKGROUND_COLOR, this.getContentBackgroundColor());
            this.setContentTextColor(
                (String) PortletConfigState.getParameter(
                    portlet,
                    rundata,
                    WebLogUtil.CONTENT_TEXT_COLOR,
                    DEFAULT_CONTENT_TEXT_COLOR));
            context.put(WebLogUtil.CONTENT_TEXT_COLOR, this.getContentTextColor());
            // Day Color
            this.setDayBackgroundColor(
                (String) PortletConfigState.getParameter(
                    portlet,
                    rundata,
                    WebLogUtil.DAY_BACKGROUND_COLOR,
                    DEFAULT_DAY_BACKGROUND_COLOR));
            context.put(WebLogUtil.DAY_BACKGROUND_COLOR, this.getDayBackgroundColor());
            this.setDayTextColor(
                (String) PortletConfigState.getParameter(
                    portlet,
                    rundata,
                    WebLogUtil.DAY_TEXT_COLOR,
                    DEFAULT_DAY_TEXT_COLOR));
            context.put(WebLogUtil.DAY_TEXT_COLOR, this.getDayTextColor());
            // Targetted Day Color
            this.setTargetdayBackgroundColor(
                (String) PortletConfigState.getParameter(
                    portlet,
                    rundata,
                    WebLogUtil.TARGET_DAY_BACKGROUND_COLOR,
                    DEFAULT_TARGET_DAY_BACKGROUND_COLOR));
            context.put(WebLogUtil.TARGET_DAY_BACKGROUND_COLOR, this.getTargetdayBackgroundColor());
            this.setTargetdayTextColor(
                (String) PortletConfigState.getParameter(
                    portlet,
                    rundata,
                    WebLogUtil.TARGET_DAY_TEXT_COLOR,
                    DEFAULT_TARGET_DAY_TEXT_COLOR));
            context.put(WebLogUtil.TARGET_DAY_TEXT_COLOR, this.getTargetdayTextColor());
            // SunDay Color
            this.setSundayBackgroundColor(
                (String) PortletConfigState.getParameter(
                    portlet,
                    rundata,
                    WebLogUtil.SUNDAY_BACKGROUND_COLOR,
                    DEFAULT_SUNDAY_BACKGROUND_COLOR));
            context.put(WebLogUtil.SUNDAY_BACKGROUND_COLOR, this.getSundayBackgroundColor());
            this.setSundayTextColor(
                (String) PortletConfigState.getParameter(
                    portlet,
                    rundata,
                    WebLogUtil.SUNDAY_TEXT_COLOR,
                    DEFAULT_SUNDAY_TEXT_COLOR));
            context.put(WebLogUtil.SUNDAY_TEXT_COLOR, this.getSundayTextColor());
            // SaturDay Color
            this.setSaturdayBackgroundColor(
                (String) PortletConfigState.getParameter(
                    portlet,
                    rundata,
                    WebLogUtil.SATURDAY_BACKGROUND_COLOR,
                    DEFAULT_SATURDAY_BACKGROUND_COLOR));
            context.put(WebLogUtil.SATURDAY_BACKGROUND_COLOR, this.getSaturdayBackgroundColor());
            this.setSaturdayTextColor(
                (String) PortletConfigState.getParameter(
                    portlet,
                    rundata,
                    WebLogUtil.SATURDAY_TEXT_COLOR,
                    DEFAULT_SATURDAY_TEXT_COLOR));
            context.put(WebLogUtil.SATURDAY_TEXT_COLOR, this.getSaturdayTextColor());
        }
    }

    /**
     * Puts all value to the context.
     */
    //    public void putDate()
    //    {
    //        context.put(WebLogUtil.CURRENT_URL, getCurrentUrl());
    //
    //        if (getUserName() != null)
    //        {
    //            context.put(WebLogUtil.USER_NAME, getUserName());
    //        }
    //        if (getTitle() != null)
    //        {
    //            context.put(WebLogUtil.TITLE, getTitle());
    //        }
    //        context.put(WebLogUtil.YEAR, Integer.toString(getYear()));
    //        context.put(WebLogUtil.MONTH, Integer.toString(getMonth()));
    //        context.put(WebLogUtil.DAY, Integer.toString(getDay()));
    //        context.put(WebLogUtil.WEBLOG_PAGE, Integer.toString(getPage()));
    //        context.put(WebLogUtil.WEBLOG_SIZE, Integer.toString(getSize()));
    //
    //        if (!infoList.isEmpty())
    //        {
    //            context.put(WebLogUtil.INFO_MESSAGE, infoList);
    //        }
    //        if (!errorList.isEmpty())
    //        {
    //            context.put(WebLogUtil.ERROR_MESSAGE, errorList);
    //        }
    //    }

    public void putDate()
    {
        context.put(WebLogUtil.YEAR, Integer.toString(getYear()));
        context.put(WebLogUtil.MONTH, Integer.toString(getMonth()));
        context.put(WebLogUtil.DAY, Integer.toString(getDay()));
    }

    public void putInfo()
    {
        context.put(WebLogUtil.CURRENT_URL, getCurrentUrl());

        if (getUserName() != null)
        {
            context.put(WebLogUtil.USER_NAME, getUserName());
        }

        if (!infoList.isEmpty())
        {
            context.put(WebLogUtil.INFO_MESSAGE, infoList);
        }
        if (!errorList.isEmpty())
        {
            context.put(WebLogUtil.ERROR_MESSAGE, errorList);
        }
    }

    public void putPageInfo()
    {
        context.put(WebLogUtil.WEBLOG_PAGE, Integer.toString(getPage()));
        context.put(WebLogUtil.WEBLOG_SIZE, Integer.toString(getSize()));

    }

    /**
     * Gets Web Log message for the given date and put it to context.
     */
    public void putWeblog()
    {
        Connection con = null;

        try
        {
            con = Torque.getConnection(WeblogPeer.DATABASE_NAME);
            con.setAutoCommit(false);
            Weblog wlog = WeblogPeer.getInstance(con, getUserName(), getYear(), getMonth(), getDay());
            if (wlog != null)
            {
                wlog.setCachedWeblogMessages(con);
            }
            con.commit();
            context.put(WebLogUtil.WEBLOG, wlog);
        }
        catch (WebLogException e)
        {
            addErrorMessage(e);
            rollback(con);
        }
        catch (TorqueException e)
        {
            addErrorMessage(e);
            rollback(con);
        }
        catch (SQLException e)
        {
            addErrorMessage(e);
            rollback(con);
        }
        finally
        {
            Torque.closeConnection(con);
        }
    }

    /**
     * Gets Web Log message for the given date and put it to context.
     */
    public void putWeblogs()
    {
        Connection con = null;
        PageList pageList = null;
        try
        {
            con = Torque.getConnection(WeblogPeer.DATABASE_NAME);
            con.setAutoCommit(false);
            pageList =
                WeblogPeer.getWeblogs(
                    con,
                    this.getUserName(),
                    this.getYear(),
                    this.getMonth(),
                    this.getDay(),
                    this.getPage(),
                    this.getSize());
            con.commit();
        }
        catch (WebLogException e)
        {
            addErrorMessage(e);
            rollback(con);
            return;
        }
        catch (TorqueException e)
        {
            addErrorMessage(e);
            rollback(con);
            return;
        }
        catch (SQLException e)
        {
            addErrorMessage(e);
            rollback(con);
            return;
        }
        finally
        {
            Torque.closeConnection(con);
        }

        if (!pageList.isEmpty())
        {
            Vector v = new Vector();
            Iterator i = pageList.iterator();
            Criteria c = new Criteria();
            c.addAscendingOrderByColumn(WeblogMessagePeer.POSTED_DATE);
            while (i.hasNext())
            {
                Object o = i.next();
                if (o instanceof Weblog)
                {
                    ((Weblog) o).setCachedWeblogMessages(c);
                    v.add((Weblog) o);
                }
                else
                {
                    logger.warn("selectMessageByDate: Object is not Weblog!");
                }
            }
            context.put(WebLogUtil.WEBLOGS, v);
            context.put(WebLogUtil.WEBLOG_NEXTPAGE, Boolean.toString(pageList.hasNextPage()));
        }
        else
        {
            addInfoMessage("Weblog for this day does not exist.");
        }
    }

    /**
     * Gets Web Log message for the given date and put it to context.
     */
    public void putWeblogMessages()
    {
        try
        {
            List l =
                WeblogMessagePeer.getWeblogMessages(this.getUserName(), this.getYear(), this.getMonth(), this.getDay());

            if (!l.isEmpty())
            {
                Vector v = new Vector();
                Iterator i = l.iterator();
                while (i.hasNext())
                {
                    Object o = i.next();
                    if (o instanceof WeblogMessage)
                    {
                        v.add((WeblogMessage) o);
                    }
                    else
                    {
                        logger.warn("getWeblogMessage: Object is not WeblogMessage!");
                    }
                }
                context.put(WebLogUtil.WEBLOG_MESSAGES, v);
            }
        }
        catch (WebLogException e)
        {
            addErrorMessage(e);
        }
    }

    /**
     * Add Web Log message.
     */
    public void addWeblogMessage()
    {
        Connection con = null;

        try
        {
            con = Torque.getConnection(WeblogPeer.DATABASE_NAME);
            con.setAutoCommit(false);

            Weblog wlog = WeblogPeer.getInstance(con, getUserName(), getYear(), getMonth(), getDay());

            Calendar cal = Calendar.getInstance();
            if (wlog != null)
            {
                // Insert WeblogMessage
                WeblogMessagePeer.add(
                    con,
                    wlog.getWeblogId(),
                    this.getTitle() != null ? WebLogUtil.encodeForDatabase(this.getTitle()) : "",
                    this.getMessage() != null ? WebLogUtil.encodeForDatabase(this.getMessage()) : "",
                    this.getFormatType() == null ? WebLogUtil.PLAIN : this.getFormatType(),
                    cal.getTime());
                con.commit();
                addInfoMessage("Added New Message.");
            }
            else
            {
                // New Weblog Item
                wlog = WeblogPeer.add(con, this.getUserName(), this.getYear(), this.getMonth(), this.getDay());
                WeblogMessagePeer.add(
                    con,
                    wlog.getWeblogId(),
                    this.getTitle() != null ? WebLogUtil.encodeForDatabase(this.getTitle()) : "",
                    this.getMessage() != null ? WebLogUtil.encodeForDatabase(this.getMessage()) : "",
                    this.getFormatType() == null ? WebLogUtil.PLAIN : this.getFormatType(),
                    cal.getTime());
                con.commit();
                addInfoMessage("Created New Message.");
            }
        }
        catch (WebLogException e)
        {
            addErrorMessage(e);
            rollback(con);
        }
        catch (TorqueException e)
        {
            addErrorMessage(e);
            rollback(con);
        }
        catch (SQLException e)
        {
            addErrorMessage(e);
            rollback(con);
        }
        finally
        {
            Torque.closeConnection(con);
        }
    }

    /**
     * Update Web Log message.
     */
    public void updateWeblogMessage()
    {
        Connection con = null;

        try
        {
            con = Torque.getConnection(WeblogPeer.DATABASE_NAME);
            con.setAutoCommit(false);

            WeblogMessagePeer.update(
                con,
                this.getUserName(),
                this.getWeblogMessageId(),
                -1,
                this.getTitle() != null ? WebLogUtil.encodeForDatabase(this.getTitle()) : "",
                this.getMessage() != null ? WebLogUtil.encodeForDatabase(this.getMessage()) : "",
                this.getFormatType() == null ? WebLogUtil.PLAIN : this.getFormatType(),
                null);
            con.commit();

            addInfoMessage("Updated WeblogMessage data.");
        }
        catch (WebLogException e)
        {
            addErrorMessage(e);
            rollback(con);
        }
        catch (TorqueException e)
        {
            addErrorMessage(e);
            rollback(con);
        }
        catch (SQLException e)
        {
            addErrorMessage(e);
            rollback(con);
        }
        finally
        {
            Torque.closeConnection(con);
        }

    }

    /**
     * Delete Web Log message.
     */
    public void deleteWeblogMessage()
    {
        Connection con = null;

        try
        {
            con = Torque.getConnection(WeblogPeer.DATABASE_NAME);
            con.setAutoCommit(false);

            WeblogMessagePeer.delete(con, this.getUserName(), this.getWeblogMessageId());
            con.commit();

            addInfoMessage("Deleted WeblogMessage data.");
        }
        catch (WebLogException e)
        {
            addErrorMessage(e);
            rollback(con);
        }
        catch (TorqueException e)
        {
            addErrorMessage(e);
            rollback(con);
        }
        catch (SQLException e)
        {
            addErrorMessage(e);
            rollback(con);
        }
        finally
        {
            Torque.closeConnection(con);
        }

    }

    /**
     * Update the title for Weblog data.
     */
    public void updateWeblog()
    {
        Connection con = null;

        try
        {
            con = Torque.getConnection(WeblogPeer.DATABASE_NAME);
            con.setAutoCommit(false);

            Weblog wlog = WeblogPeer.getInstance(con, getUserName(), getYear(), getMonth(), getDay());

            wlog.setTitle(getTitle());
            wlog.save(con);
            con.commit();

            addInfoMessage("Updated Weblog data.");
        }
        catch (WebLogException e)
        {
            addErrorMessage(e);
            rollback(con);
        }
        catch (TorqueException e)
        {
            addErrorMessage(e);
            rollback(con);
        }
        catch (SQLException e)
        {
            addErrorMessage(e);
            rollback(con);
        }
        finally
        {
            Torque.closeConnection(con);
        }
    }

    /**
     * Update the title for Weblog data.
     */
    public void deleteWeblog()
    {
        Connection con = null;

        try
        {
            con = Torque.getConnection(WeblogPeer.DATABASE_NAME);
            con.setAutoCommit(false);

            Weblog wlog = WeblogPeer.getInstance(con, getUserName(), getYear(), getMonth(), getDay());

            List list = wlog.getWeblogMessages(con);
            if (list != null)
            {
                Iterator i = list.iterator();
                while (i.hasNext())
                {
                    Object obj = i.next();
                    if (obj instanceof WeblogMessage)
                    {
                        WeblogMessagePeer.doDelete((WeblogMessage) obj, con);
                    }
                    else
                    {
                        addErrorMessage("Wrong WeblogMessage type.");
                        rollback(con);
                        return;
                    }
                }
            }

            WeblogPeer.doDelete(wlog, con);
            con.commit();

            addInfoMessage("Deleted Weblog data.");
        }
        catch (WebLogException e)
        {
            addErrorMessage(e);
            rollback(con);
        }
        catch (TorqueException e)
        {
            addErrorMessage(e);
            rollback(con);
        }
        catch (SQLException e)
        {
            addErrorMessage(e);
            rollback(con);
        }
        finally
        {
            Torque.closeConnection(con);
        }
    }

    /**
     * Selects messages by date.
     */
    public void putTopicWeblogs()
    {
        Connection con = null;
        PageList pageList;
        try
        {
            con = Torque.getConnection(WeblogPeer.DATABASE_NAME);
            con.setAutoCommit(false);

            pageList = WeblogPeer.getLatestWeblogs(con, getUserName(), 1, getTopicSize());
            con.commit();
        }
        catch (WebLogException e)
        {
            addErrorMessage(e);
            rollback(con);
            return;
        }
        catch (TorqueException e)
        {
            addErrorMessage(e);
            rollback(con);
            return;
        }
        catch (SQLException e)
        {
            addErrorMessage(e);
            rollback(con);
            return;
        }
        finally
        {
            Torque.closeConnection(con);
        }

        if (!pageList.isEmpty())
        {
            Vector v = new Vector();
            Iterator i = pageList.iterator();
            Criteria c = new Criteria();
            c.addAscendingOrderByColumn(WeblogMessagePeer.POSTED_DATE);
            while (i.hasNext())
            {
                Object o = i.next();
                if (o instanceof Weblog)
                {
                    ((Weblog) o).setCachedWeblogMessages(c);
                    v.add((Weblog) o);
                }
                else
                {
                    logger.warn("selectMessageByDate: Object is not Weblog!");
                }
            }
            context.put(WebLogUtil.WEBLOGS, v);
            context.put(WebLogUtil.WEBLOG_NEXTPAGE, Boolean.toString(pageList.hasNextPage()));
        }
        else
        {
            addErrorMessage("Cannot find messages.");
        }
    }

    public void putCalendar()
    {
        StringBuffer buf = new StringBuffer();
        buf.append(CalendarUtil.printCalendar(this));
        context.put(WebLogUtil.WEBLOG_CALENDAR, buf.toString());
    }

    public void putEditorHeader()
    {
        StringBuffer buf = new StringBuffer();
        buf.append(CalendarUtil.printTextDateNavigation(this));

        context.put(WebLogUtil.WEBLOG_EDITOR_HEADER, buf.toString());
    }

    public void putViewerHeader()
    {
        StringBuffer buf = new StringBuffer();
        buf.append(CalendarUtil.printFormNavigation(this));
        buf.append(CalendarUtil.printTextDateNavigation(this));

        context.put(WebLogUtil.WEBLOG_VIEWER_HEADER, buf.toString());
    }

    public void putCalendarHeader()
    {
        StringBuffer buf = new StringBuffer();
        buf.append(CalendarUtil.printFormNavigation(this));

        context.put(WebLogUtil.WEBLOG_CALENDAR_HEADER, buf.toString());
    }

    public void putEditorFooter()
    {
        StringBuffer buf = new StringBuffer();
        buf.append(CalendarUtil.printFormNavigation(this));

        context.put(WebLogUtil.WEBLOG_EDITOR_FOOTER, buf.toString());
    }

    public void putViewerFooter()
    {
        StringBuffer buf = new StringBuffer();

        context.put(WebLogUtil.WEBLOG_VIEWER_FOOTER, buf.toString());
    }

    public void putCalendarFooter()
    {
        StringBuffer buf = new StringBuffer();
        buf.append(CalendarUtil.printTextMonthNavigation(this));

        context.put(WebLogUtil.WEBLOG_CALENDAR_FOOTER, buf.toString());
    }

    /* getter and setter method */

    /**
     * Returns the user name.
     * 
     * @return
     */
    public String getUserName()
    {
        return userName;
    }

    /**
     * Sets the user name.
     * 
     * @param string    User name
     */
    public void setUserName(String string)
    {
        userName = string;
    }

    /**
     * Returns Day value
     * 
     * @return  Day
     */
    public int getDay()
    {
        return day;
    }

    /**
     * Returns Month value
     * 
     * @return  Month
     */
    public int getMonth()
    {
        return month;
    }

    /**
     * Returns Year value
     * 
     * @return
     */
    public int getYear()
    {
        return year;
    }

    /**
     * Sets Day value
     * 
     * @param i Day
     */
    public void setDay(int i)
    {
        day = i;
    }

    /**
     * Sets Month value
     * 
     * @param i Month
     */
    public void setMonth(int i)
    {
        month = i;
    }

    /**
     * Sets Year value
     * 
     * @param i Year
     */
    public void setYear(int i)
    {
        year = i;
    }

    /**
     * Returns Message
     * 
     * @return  Message
     */
    public String getMessage()
    {
        return message;
    }

    /**
     * Returns Title
     * 
     * @return  Title
     */
    public String getTitle()
    {
        return title;
    }

    /**
     * Sets Message
     * 
     * @param string    Message
     */
    public void setMessage(String string)
    {
        message = string;
    }

    /**
     * Sets Title
     * 
     * @param string    Title
     */
    public void setTitle(String string)
    {
        title = string;
    }

    /**
     * Returns ID of the message
     * 
     * @return  ID
     */
    public int getWeblogMessageId()
    {
        return weblogMessageId;
    }

    /**
     * Returns the format type of the message
     * 
     * @return  Format Type
     */
    public String getFormatType()
    {
        return formatType;
    }

    /**
     * Sets ID of the message
     * 
     * @param i ID
     */
    public void setWeblogMessageId(int i)
    {
        weblogMessageId = i;
    }

    /**
     * Sets the format type of the message
     * 
     * @param string    Format Type
     */
    public void setFormatType(String string)
    {
        formatType = string;
    }

    /**
     * Add the information message
     * 
     * @param str	the information message
     */
    private void addInfoMessage(String str)
    {
        infoList.add(str);
    }

    /**
     * Add the error message
     * 
     * @param str	the error message
     */
    private void addErrorMessage(String str)
    {
        errorList.add(str);
    }

    /**
     * Add the error message
     * 
     * @param str   the error message
     */
    private void addErrorMessage(Exception e)
    {
        logger.warn(e.toString());
        addErrorMessage(e.getMessage());
    }

    /**
     * @return
     */
    public String getCurrentUrl()
    {
        return currentUrl;
    }

    /**
     * @param string
     */
    public void setCurrentUrl(String string)
    {
        currentUrl = string;
    }

    /**
     * @return
     */
    public int getPage()
    {
        return page;
    }

    /**
     * @return
     */
    public int getSize()
    {
        return size;
    }

    /**
     * @param i
     */
    public void setPage(int i)
    {
        page = i;
    }

    /**
     * @param i
     */
    public void setSize(int i)
    {
        size = i;
    }

    /**
     * Roll back the transaction.
     * 
     * @param con
     */
    public void rollback(Connection con)
    {
        try
        {
            con.rollback();
        }
        catch (SQLException e)
        {
            logger.warn("addWeblogMessage: " + e.toString());
        }

    }
    /**
     * Gets the type for WebLogManager. The value is WEBLOG_EDIT or WEBLOG_VIEW.
     * 
     * @return
     */
    public String getType()
    {
        return type;
    }

    /**
     * Sets the type for WebLogManager. The value is WEBLOG_EDIT or WEBLOG_VIEW.
     * 
     * @param string
     */
    public void setType(String string)
    {
        type = string;
    }

    /**
     * @return
     */
    public int getTopicSize()
    {
        return topicSize;
    }

    /**
     * @param i
     */
    public void setTopicSize(int i)
    {
        topicSize = i;
    }

    /**
     * @return
     */
    public String getDayBackgroundColor()
    {
        return dayBackgroundColor;
    }

    /**
     * @return
     */
    public String getDayTextColor()
    {
        return dayTextColor;
    }

    /**
     * @return
     */
    public String getFrameBackgroundColor()
    {
        return frameBackgroundColor;
    }

    /**
     * @return
     */
    public String getFrameTextColor()
    {
        return frameTextColor;
    }

    /**
     * @return
     */
    public String getSaturdayBackgroundColor()
    {
        return saturdayBackgroundColor;
    }

    /**
     * @return
     */
    public String getSaturdayTextColor()
    {
        return saturdayTextColor;
    }

    /**
     * @return
     */
    public String getSundayBackgroundColor()
    {
        return sundayBackgroundColor;
    }

    /**
     * @return
     */
    public String getSundayTextColor()
    {
        return sundayTextColor;
    }

    /**
     * @return
     */
    public String getTargetdayBackgroundColor()
    {
        return targetdayBackgroundColor;
    }

    /**
     * @return
     */
    public String getTargetdayTextColor()
    {
        return targetdayTextColor;
    }

    /**
     * @param string
     */
    public void setDayBackgroundColor(String string)
    {
        dayBackgroundColor = string;
    }

    /**
     * @param string
     */
    public void setDayTextColor(String string)
    {
        dayTextColor = string;
    }

    /**
     * @param string
     */
    public void setFrameBackgroundColor(String string)
    {
        frameBackgroundColor = string;
    }

    /**
     * @param string
     */
    public void setFrameTextColor(String string)
    {
        frameTextColor = string;
    }

    /**
     * @param string
     */
    public void setSaturdayBackgroundColor(String string)
    {
        saturdayBackgroundColor = string;
    }

    /**
     * @param string
     */
    public void setSaturdayTextColor(String string)
    {
        saturdayTextColor = string;
    }

    /**
     * @param string
     */
    public void setSundayBackgroundColor(String string)
    {
        sundayBackgroundColor = string;
    }

    /**
     * @param string
     */
    public void setSundayTextColor(String string)
    {
        sundayTextColor = string;
    }

    /**
     * @param string
     */
    public void setTargetdayBackgroundColor(String string)
    {
        targetdayBackgroundColor = string;
    }

    /**
     * @param string
     */
    public void setTargetdayTextColor(String string)
    {
        targetdayTextColor = string;
    }

    /**
     * @return
     */
    public String getContentBackgroundColor()
    {
        return contentBackgroundColor;
    }

    /**
     * @param string
     */
    public void setContentBackgroundColor(String string)
    {
        contentBackgroundColor = string;
    }

    /**
     * @return
     */
    public String getContentTextColor()
    {
        return contentTextColor;
    }

    /**
     * @param string
     */
    public void setContentTextColor(String string)
    {
        contentTextColor = string;
    }

}
