/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.m4hv.extensions.validator;

import javax.validation.ConstraintValidator;
import javax.validation.ConstraintValidatorContext;

import org.maru.m4hv.extensions.constraints.Hiragana;
import org.maru.m4hv.extensions.constraints.SpaceType;

/**
 * Hiragana validator.<br>
 * This validator validates the value passed as a parameter is Hiragana.
 * Validating the voicing mark and the iteration mark and the hiragana digraph is optional.
 * If you would like to validate those, turn on the those flags to true and you might
 * sometimes include space in your validating value, for this requirement {@code Hiragana}
 * annotation has enableSpace flag to validate the value including space. We have option
 * to validate the types of spaces whick is fullwidth space or hulfwidth space or both.
 *
 */
public class HiraganaValidator implements ConstraintValidator<Hiragana, String> {
    private boolean enableVoicingMarks;

    private boolean enableIterationMarks;

    private boolean enableHiraganaDigraph;

    private boolean enableSpace;

    private SpaceType spaceType;

    public void initialize(Hiragana hiragana) {
        this.enableVoicingMarks = hiragana.enableVoicingMarks();
        this.enableIterationMarks = hiragana.enableIterationMarks();
        this.enableHiraganaDigraph = hiragana.enableHiraganaDigraph();
        this.enableSpace = hiragana.enableSpace();
        this.spaceType = hiragana.spaceType();
    }

    public boolean isValid(String object, ConstraintValidatorContext constraintContext) {
        if (object == null) {
            return true;
        }
        for (int i = 0; i < object.codePointCount(0, object.length()); i++) {
            if (object.codePointAt(i) >= Characters.START_HIRAGANA_LETTER &&
                    object.codePointAt(i) <= Characters.END_HIRAGANA_LETTER) {
                continue;
            }
            if (enableVoicingMarks == true) {
                if(object.codePointAt(i) >= Characters.START_VOICING_MARK &&
                        object.codePointAt(i) <= Characters.END_VOICING_MARK) {
                    continue;
                }
            }
            if (enableIterationMarks == true) {
                if (object.codePointAt(i) == Characters.HIRAGANA_ITERATION_MARK ||
                        object.codePointAt(i) == Characters.HIRAGANA_VOICED_ITERATION_MARK) {
                    continue;
                }
            }
            if (enableHiraganaDigraph == true) {
                if (object.codePointAt(i) == Characters.HIRAGANA_DIGRAPH_YORI) {
                    continue;
                }
            }
            if (enableSpace == true) {
                if (spaceType.equals(SpaceType.BOTH)) {
                    if (object.codePointAt(i) == Characters.FULLWIDTH_SPACE ||
                            object.codePointAt(i) == Characters.HULFWIDTH_SPACE) {
                        continue;
                    }
                } else if (spaceType.equals(SpaceType.FULLWIDTH_SPACE)) {
                    if (object.codePointAt(i) == Characters.FULLWIDTH_SPACE) {
                        continue;
                    }
                } else {
                    // The space type is hulf width space.
                    if (object.codePointAt(i) == Characters.HULFWIDTH_SPACE) {
                        continue;
                    }
                }
            }
            return false;
        }
        return true;
    }
}
