#pragma once
#include <mof/script/Environment.hpp>
#include <mof/ConsoleIO.hpp>
#include "mof/EventScheduler.hpp"
#include "mof/OnceEventCondition.hpp"
#include "mof/KeyPressedEventCondition.hpp"
#include <mof/widgets.hpp>
#include <mof/streams.hpp>
#include <mof/Matrix3D.hpp>
#include <mof/Sprite.hpp>
#include <mof/script/ObjectData.hpp>
#include <mof/utilities.hpp>
#include <boost/bind.hpp>
#include <boost/format.hpp>
#include <boost/algorithm/string.hpp>
#include <mof/sound/SoundDevice.hpp>
#include <vector>
#include <memory>
	
namespace mof
{
namespace script
{
//{{{ Impl
	struct Environment::Impl
	{
		bool waiting_;                  ///< Interpreterにコマンド実行の停止を伝えるための状態変数
		mof::EventScheduler scheduler_; ///< キー入力待機など
     	std::shared_ptr<mof::InputReceiver> input_;
		std::vector<std::shared_ptr<mof::script::MessageData>> message_data_list_;
		std::vector<std::shared_ptr<mof::script::MenuData>> menu_data_list_;
		std::vector<std::shared_ptr<mof::script::SoundData>> sound_data_list_;
		std::vector<mof::script::PictureData::ptr> picture_data_list_;
		std::vector<mof::script::ParticlegenData::ptr> particlegen_data_list_;
		std::map<std::string, int> property_map_;
		
		Impl(std::shared_ptr<mof::InputReceiver> input)
			: waiting_(false), input_(input)
		{
		}

		void setWaiting(bool waiting)
		{
			waiting_ = waiting;
		}

	};
//}}}
//{{{ constructor
	Environment::Environment(std::shared_ptr<mof::InputReceiver> input)
	: impl_(new Impl(input))
	{
		impl_->property_map_["system.menu.move_cursor_sound"] = sound_create(_T("sound/move_cursor.wav"));
	}
//}}}
//{{{ destructor
	Environment::~Environment()
	{
	}
//}}}
//{{{ message_create
	int Environment::message_create(const mof::tstring& title, const GameData::entry_t& style)
	{
		using namespace mof::script;
		using namespace boost;

		DEBUG_PRINT("message_create(" << title << ")");
		std::shared_ptr<MessageData> data = std::shared_ptr<MessageData>(create_message_data(title, style).release());
		impl_->message_data_list_.push_back(data);// TODO 再利用
		return impl_->message_data_list_.size() -1;
	}
//}}}	
//{{{ massage_next
	int Environment::message_next(int id, const tstring& text)
	{
		DEBUG_PRINT("message_next(" << id << "," << text << ")");
		std::shared_ptr<mof::script::MessageData>& data = impl_->message_data_list_.at(id);
		data->message_->addPage(text);
		return data->message_->nextPage();
	}
//}}}
//{{{ menu_create
	int Environment::menu_create
	(
		const mof::tstring& title,
		const std::vector<mof::tstring>& items,
		const GameData::entry_t& style
	)
	{
		using namespace mof::script;
		using namespace boost;

		DEBUG_PRINT("menu_create(" << title << ")");
		std::shared_ptr<MenuData> data = std::shared_ptr<MenuData>(create_menu_data(title, items, style).release());
		impl_->menu_data_list_.push_back(data);
		return impl_->menu_data_list_.size() -1;
	}
//}}}	
//{{{ menu_move_cursor
	int Environment::menu_move_cursor(int id, MoveDirection direction)
	{
		using namespace mof::script;
		using namespace boost;

		DEBUG_PRINT(format("menu_move_cursor(%d, %d)") % id % direction);
		auto& data = impl_->menu_data_list_.at(id);
		auto& sound_data = impl_->sound_data_list_.at(impl_->property_map_["system.menu.move_cursor_sound"]);
		int prev_index = data->menu_->getSelectedIndex();
		mof::FrameNumber period;
		switch (direction) {
			case MOVE_UP : 
				period = data->menu_->up();
				break;
			case MOVE_DOWN : 
				period = data->menu_->down();
				break;
			case MOVE_LEFT : 
				period = data->menu_->left();
				break;
			case MOVE_RIGHT : 
				period = data->menu_->right();
				break;
			default:
				return 0;
		}
		if (data->menu_->getSelectedIndex() != prev_index) {
			sound_data->sound_->stop();
			sound_data->sound_->play(sound_data->streaming_);
		}
		return period;
	}
//}}}	
//{{{ menu_select
	int Environment::menu_select(int id)
	{
		using namespace mof::script;
		using namespace boost;

		DEBUG_PRINT(format("menu_select(%d)") % id);
		auto& data = impl_->menu_data_list_.at(id);
		return data->menu_->performAction();
	}
//}}}	
//{{{ menu_get_current
	int Environment::menu_get_current(int id)
	{
		using namespace mof::script;
		using namespace boost;

		DEBUG_PRINT(format("menu_get_current(%d)") % id);
		auto& data = impl_->menu_data_list_.at(id);
		DEBUG_PRINT(data->menu_->getSelectedIndex());
		return data->menu_->getSelectedIndex();
	}
//}}}	
//{{{ wait_for_key
	void Environment::wait_for_key(mof::InputReceiver::Key key)
	{
		DEBUG_PRINT("wait_for_key(" << key << ")");
		bool& waiting = impl_->waiting_;
        impl_->scheduler_.addEvent
        (
            new mof::OnceEventCondition(new mof::KeyPressedEventCondition(key, impl_->input_)), 
			[&waiting](){waiting = false;}
        );
		impl_->waiting_ = true;
	}
//}}}
//{{{ wait_frame
	void Environment::wait_frame(size_t frame)
	{
		DEBUG_PRINT("wait_frame(" << frame << ")");
		if (frame == 0)return;
		bool& waiting = impl_->waiting_;
		impl_->scheduler_.addEvent(frame, [&waiting](){waiting = false;});// set waiting_ false after 'frame' 
		impl_->waiting_ = true;
	}
//}}}
//{{{ get_last_key
	mof::InputReceiver::Key Environment::get_last_key()
	{
		//TODO 複数同時をtimestampで識別
		DEBUG_PRINT("get_last_key()");
		for (int key = mof::InputReceiver::BEGIN + 1; key < mof::InputReceiver::ANY; ++key) {
			if (impl_->input_->testKeyState(static_cast<mof::InputReceiver::Key>(key))) {
				return static_cast<mof::InputReceiver::Key>(key);// 最初に見つかったものを選ぶ
			}
		}
		return mof::InputReceiver::END;// 何も入力されていないとき
	}
//}}}
//{{{ sound_create
	int Environment::sound_create(const mof::tstring& filepath)
	{
		using namespace mof::script;
		using namespace boost;

		//TODO キャッシュ
		
		DEBUG_PRINT("sound_create(" << filepath << ")");
		auto ptr = mof::sound::SoundDevice::create_static_sound(filepath);
		//TODO 分岐
		auto data = std::make_shared<SoundData>();
		data->sound_ = std::shared_ptr<mof::SoundBuffer>(ptr.release());
		data->streaming_ = false;
		impl_->sound_data_list_.push_back(data);
		return impl_->sound_data_list_.size() -1;
	}
//}}}	
//{{{ sound_play
	void Environment::sound_play(int id)
	{
		using namespace mof::script;
		using namespace boost;

		DEBUG_PRINT("sound_play(" << id << ")");
		impl_->sound_data_list_.at(id)->sound_->play(false);
	}
//}}}	
//{{{ load_game_data
	GameData::ptr Environment::load_game_data(const mof::tstring& resource_path)
	{
		return get_game_data(resource_path);
	}
//}}}	
//{{{ print_debug
	void Environment::print_debug(const mof::tstring& message)
	{
		DEBUG_PRINT(message);
	}
//}}}
//{{{ picture_create
	int Environment::picture_create(const mof::tstring& filepath)
	{
		using namespace mof::script;
		using namespace boost;

		DEBUG_PRINT("picture_create(" << filepath << ")");
		auto data = std::make_shared<PictureData>();
		data->sprite_ = 
			std::make_shared<Sprite>
			(
				std::make_shared<mof::Texture>(filepath),
				mof::Rectangle<mof::real>(0, 0, 1, 1)
			);
		impl_->picture_data_list_.push_back(data);// TODO 再利用
		return impl_->picture_data_list_.size() -1;

	}
//}}}	
//{{{ particlegen_create
	int Environment::particlegen_create()
	{
		using namespace mof::script;
		using namespace boost;

		DEBUG_PRINT("particlegen_create()");
		//DEBUG_PRINT("picture_create(" << filepath << ")");
		auto data = std::make_shared<ParticlegenData>();
		data->particlegen_ = std::make_shared<mof::particlegen>();
		data->particlegen_->world_transform() << data->position_ref_.makeRef(Matrix3D::createIdentity());
		impl_->particlegen_data_list_.push_back(data);// TODO 再利用
		return impl_->particlegen_data_list_.size() -1;

	}
//}}}	
// methods for common objects 
//{{{ show
int Environment::show(int id, const mof::tstring& class_path)
{
		using namespace mof::script;
		using namespace boost;

		if (class_path == "menu") {
			DEBUG_PRINT("menu_show(" << id << ")");
			std::shared_ptr<MenuData>& data = impl_->menu_data_list_.at(id);
			return data->frame_->show();
		}
		else if (class_path == "message") {
			DEBUG_PRINT("message_show(" << id << ")");
			std::shared_ptr<mof::script::MessageData>& data = impl_->message_data_list_.at(id);
			return data->frame_->show();
		}
		else throw std::invalid_argument("unknown class path:" + class_path);
}
//}}}
//{{{ hide
int Environment::hide(int id, const mof::tstring& class_path)
{
		using namespace mof::script;
		using namespace boost;

		if (class_path == "menu") {
			DEBUG_PRINT("menu_hide(" << id << ")");
			std::shared_ptr<MenuData>& data = impl_->menu_data_list_.at(id);
			return data->frame_->hide();
		}
		else if (class_path == "message") {
			DEBUG_PRINT("message_hide(" << id << ")");
			std::shared_ptr<mof::script::MessageData>& data = impl_->message_data_list_.at(id);
			return data->frame_->hide();
		}
		else throw std::invalid_argument("unknown class path:" + class_path);
}
//}}}
//{{{ dispose
void Environment::dispose(int id, const mof::tstring& class_path)
{
		using namespace mof::script;
		using namespace boost;

		if (class_path == "menu") {
			DEBUG_PRINT("menu_dispose(" << id << ")");
			impl_->menu_data_list_.at(id) = std::shared_ptr<MenuData>();// NULLに設定
		}
		else if (class_path == "message") {
			DEBUG_PRINT("message_dispose(" << id << ")");
			impl_->message_data_list_.at(id) = std::shared_ptr<MessageData>();// NULLに設定
		}
		else if (class_path == "picture") {
			DEBUG_PRINT("picture_dispose(" << id << ")");
			impl_->picture_data_list_.at(id) = std::shared_ptr<PictureData>();// NULLに設定
		}
		else if (class_path == "particlegen") {
			DEBUG_PRINT("particlegen_dispose(" << id << ")");
			impl_->particlegen_data_list_.at(id) = std::shared_ptr<ParticlegenData>();// NULLに設定
		}
		else if (class_path == "sound") {
			DEBUG_PRINT("sound_dispose(" << id << ")");
			impl_->sound_data_list_.at(id) = std::shared_ptr<SoundData>();// NULLに設定
		}
		else throw std::invalid_argument("unknown class path:" + class_path);
}
//}}}
//{{{ get_properties
	GameData::ptr Environment::get_properties(int id, const mof::tstring& class_path)
	{
		using namespace mof::script;
		using namespace boost;
		GameData::ptr game_data = std::make_shared<GameData>();
		game_data->data_.resize(1);
		GameData::entry_t& e = game_data->data_.front();

		if (class_path == "menu") {
			DEBUG_PRINT("get_property(" << id << class_path << ")");
			std::shared_ptr<MenuData>& data = impl_->menu_data_list_.at(id);
			Vector2D v = data->frame_->getView()->getPreferredSize();
			e["preferred_size"] = (format("%d,%d") % v.x % v.y).str();
			Vector2D pos = data->frame_->getView()->getPositionStream().value();
			e["position2"] = (format("%d,%d") % pos.x % pos.y).str();
		}
		else if (class_path == "message") {
			DEBUG_PRINT("get_property(" << id << class_path << ")");
			std::shared_ptr<MessageData>& data = impl_->message_data_list_.at(id);
			Vector2D v = data->frame_->getView()->getPreferredSize();
			e["preferred_size"] = (format("%d,%d") % v.x % v.y).str();
			Vector2D pos = data->frame_->getView()->getPositionStream().value();
			e["position2"] = (format("%d,%d") % pos.x % pos.y).str();
		}
		else if (class_path == "picture") {
			DEBUG_PRINT("get_property(" << id << class_path << ")");
			std::shared_ptr<PictureData>& data = impl_->picture_data_list_.at(id);
			Vector2D v = data->sprite_->getPreferredSize();
			e["preferred_size"] = (format("%d,%d") % v.x % v.y).str();
			Vector2D pos = data->sprite_->getPositionStream().value();
			e["position2"] = (format("%d,%d") % pos.x % pos.y).str();
		}

		else throw std::invalid_argument("unknown class path:" + class_path);

		return game_data;
	}
//}}}
//{{{ set_color_behavior
	void Environment::set_color_behavior(int id, const mof::tstring& class_path, const Manipulator<Color4f>::ptr& seq, mof::FrameNumber period)
	{
		using namespace boost::algorithm;
		DEBUG_PRINT("set_color_behavior(" << ")");	

		std::vector<mof::tstring> splited_list;
		split(splited_list, class_path, is_any_of("."));
		if (splited_list.front() == "picture") {
			auto picture = impl_->picture_data_list_.at(id);
			picture->sprite_->setColorStream(ColorStream(seq));
			return;
		}
		else if (splited_list.front() == "menu") {
			if (splited_list.at(1) == "open") {
				auto menu = impl_->menu_data_list_.at(id);
				menu->frame_->setBehaviorOnColor(mof::widget::Frame::FRAME_OPEN, seq, period);
				return;
			}
			else if (splited_list.at(1) == "close") {
				auto menu = impl_->menu_data_list_.at(id);
				menu->frame_->setBehaviorOnColor(mof::widget::Frame::FRAME_CLOSE, seq, period);
				return;
			}

		}
		else if (splited_list.front() == "message") {
			if (splited_list.at(1) == "open") {
				auto message = impl_->message_data_list_.at(id);
				message->frame_->setBehaviorOnColor(mof::widget::Frame::FRAME_OPEN, seq, period);
				return;
			}
			else if (splited_list.at(1) == "close") {
				auto message = impl_->message_data_list_.at(id);
				message->frame_->setBehaviorOnColor(mof::widget::Frame::FRAME_CLOSE, seq, period);
				return;
			}

		}



		throw std::logic_error("unknown class_path:" + class_path);
	}
//}}}
//{{{ set_position_behavior
	void Environment::set_position_behavior(int id, const mof::tstring& class_path, const Manipulator<Vector2D>::ptr& seq, mof::FrameNumber period)
	{
		using namespace boost::algorithm;
		DEBUG_PRINT("set_position_behavior(" << ")");	

		std::vector<mof::tstring> splited_list;
		split(splited_list, class_path, is_any_of("."));
		if (splited_list.front() == "picture") {
			auto picture = impl_->picture_data_list_.at(id);
			picture->sprite_->setPositionStream(Vector2DStream(seq));
			return;
		}
		else if (splited_list.front() == "menu") {
			if (splited_list.at(1) == "open") {
				auto menu = impl_->menu_data_list_.at(id);
				menu->frame_->setBehaviorOnPosition(mof::widget::Frame::FRAME_OPEN, seq, period);
				return;
			}
			else if (splited_list.at(1) == "close") {
				auto menu = impl_->menu_data_list_.at(id);
				menu->frame_->setBehaviorOnPosition(mof::widget::Frame::FRAME_CLOSE, seq, period);
				return;
			}

		}
		else if (splited_list.front() == "message") {
			if (splited_list.at(1) == "open") {
				auto message = impl_->message_data_list_.at(id);
				message->frame_->setBehaviorOnPosition(mof::widget::Frame::FRAME_OPEN, seq, period);
				return;
			}
			else if (splited_list.at(1) == "close") {
				auto message = impl_->message_data_list_.at(id);
				message->frame_->setBehaviorOnPosition(mof::widget::Frame::FRAME_CLOSE, seq, period);
				return;
			}
		}


		throw std::logic_error("unknown class_path:" + class_path);
	}
//}}}
//{{{ set_position_behavior
	void Environment::set_position_behavior(int id, const mof::tstring& class_path, const Manipulator<Vector3D>::ptr& seq, mof::FrameNumber period)
	{
		using namespace boost::algorithm;
		using namespace mof;
		DEBUG_PRINT("set_position_behavior(" << ")");	

		std::vector<mof::tstring> splited_list;
		split(splited_list, class_path, is_any_of("."));
		if (splited_list.front() == "particlegen") {
			auto p = impl_->particlegen_data_list_.at(id);
			p->position_ref_.replace(0, std::make_shared<Translation3D>(seq));
			return;
		}


		throw std::logic_error("unknown class_path:" + class_path);
	}
//}}}
//{{{ set_size_behavior
	void Environment::set_size_behavior(int id, const mof::tstring& class_path, const Manipulator<Vector2D>::ptr& seq, mof::FrameNumber period)
	{
		using namespace boost::algorithm;
		DEBUG_PRINT("set_position_behavior(" << ")");	

		std::vector<mof::tstring> splited_list;
		split(splited_list, class_path, is_any_of("."));
		if (splited_list.front() == "picture") {
			auto picture = impl_->picture_data_list_.at(id);
			picture->sprite_->setSizeStream(Vector2DStream(seq));
			return;
		}
		else if (splited_list.front() == "menu") {
			if (splited_list.at(1) == "open") {
				auto menu = impl_->menu_data_list_.at(id);
				menu->frame_->setBehaviorOnSize(mof::widget::Frame::FRAME_OPEN, seq, period);
				return;
			}
			else if (splited_list.at(1) == "close") {
				auto menu = impl_->menu_data_list_.at(id);
				menu->frame_->setBehaviorOnSize(mof::widget::Frame::FRAME_CLOSE, seq, period);
				return;
			}

		}
		else if (splited_list.front() == "message") {
			if (splited_list.at(1) == "open") {
				auto message = impl_->message_data_list_.at(id);
				message->frame_->setBehaviorOnSize(mof::widget::Frame::FRAME_OPEN, seq, period);
				return;
			}
			else if (splited_list.at(1) == "close") {
				auto message = impl_->message_data_list_.at(id);
				message->frame_->setBehaviorOnSize(mof::widget::Frame::FRAME_CLOSE, seq, period);
				return;
			}
		}

		throw std::logic_error("unknown class_path:" + class_path);
	}
//}}}

//{{{ update
	void Environment::update()
	{
		using namespace mof::script;
		using namespace std;
		impl_->scheduler_.update();
		foreach (auto data, impl_->message_data_list_) {
			if (data.get()) data->frame_->update();
		}
		foreach (auto data, impl_->menu_data_list_) {
			if (data.get()) data->frame_->update(); 
		}
		foreach (auto data, impl_->sound_data_list_) {
			if (data.get()) data->sound_->update(); 
		}

		foreach (auto data, impl_->picture_data_list_) {
			if (data.get()) data->sprite_->update(); 
		}

		foreach (auto data, impl_->particlegen_data_list_) {
			if (data) data->particlegen_->update(); 
		}


	}
//}}}
//{{{ draw
	void Environment::draw() const
	{
		using namespace mof::script;
		using namespace std;
		
		mof::GraphicsDevice::setAlphaBlendingMode(mof::GraphicsDevice::BLENDING_MODE_ADD);
		mof::GraphicsDevice::setZBuffer(false);
		foreach (auto& data, impl_->particlegen_data_list_) {
			if (data) {
				auto& list = data->particlegen_->drawables(); 
				foreach (auto p, list) p->draw();
			}

		}
		mof::GraphicsDevice::setZBuffer(true);
		mof::GraphicsDevice::setAlphaBlendingMode(mof::GraphicsDevice::BLENDING_MODE_ALPHA);

		foreach (auto data, impl_->picture_data_list_) {
			if (data.get()) data->sprite_->draw();
		}


		foreach (auto data,  impl_->message_data_list_) {
			if (data.get()) data->frame_->getView()->draw();
		}
		
		foreach (auto data, impl_->menu_data_list_) {
			if (data.get()) data->frame_->getView()->draw();
		}//TODO ソート
	
		
	}
//}}}
//{{{ isWaiting
	bool Environment::isWaiting() const
	{
		return impl_->waiting_;
	}
//}}}
}
}
