#include <mof/script/lua_state.hpp>
#include <mof/script/CommandSet.hpp>
#include <mof/ConsoleIO.hpp>
#include <mof/streams.hpp>
#include <mof/utilities.hpp>
#include <mof/script/seq_parser.hpp>
#include <boost/algorithm/string.hpp>

namespace mof
{
namespace script
{
	lua_state lua_state::singleton_lua;
//{{{ unpack_style
	GameData::entry_t unpack_style(const mof::tstring& style)
	{
		using namespace boost::algorithm;
		using namespace std;

		GameData::entry_t entry;
		vector<mof::tstring> style_items;
		split(style_items, style, is_any_of(";"));
		vector<mof::tstring> key_value;
		foreach (mof::tstring& item, style_items) {
			key_value.clear();
			split(key_value, item, is_any_of("="));
			if (key_value.size() != 2) continue;
			entry[key_value[0]] = key_value[1];
		}
		return entry;
	}
//}}}
//{{{ binders *
	int message_create(const tstring& title, const tstring& style)
	{ 
		return lua_state::instance().command_set()->message_create(title, unpack_style(style));
	}
	int message_next(int id, const tstring& text){ return lua_state::instance().command_set()->message_next(id, text); }
	void wait_for_key(mof::InputReceiver::Key key){ return lua_state::instance().command_set()->wait_for_key(key); }
	void wait_frame(size_t frame){ return lua_state::instance().command_set()->wait_frame(frame); }
	int get_last_key(){ return lua_state::instance().command_set()->get_last_key(); }
	//{{{ menu_create
	int menu_create(lua_State* l)
	{
        int n = lua_gettop(l);
        if(!lua_isstring(l , 1))throw std::runtime_error("invalid call");
        if(!lua_istable(l , 2))throw std::runtime_error("invalid call");
        if(!lua_isstring(l , 3))throw std::runtime_error("invalid call");
        mof::tstring title = lua_tostring(l , 1);
        mof::tstring style = lua_tostring(l , 3);
    
        std::vector<mof::tstring> items;
        for (int i = 1 ; ; i++) {
            lua_pushnumber(l , i);
            lua_gettable(l , 2);
            if (lua_isnil(l , -1)) break;
            if (!lua_isstring(l , -1)) throw std::runtime_error("invalid call");
            items.push_back(lua_tostring(l , -1));
            lua_pop(l , 1);
        }
        lua_pop(l , n);
		if (items.empty()) items.push_back("aaa");

		int id = lua_state::instance().command_set()->menu_create(title, items, unpack_style(style));
		lua_pushnumber(l, id);
        return 1;
    } 
	//}}}
	int menu_move_cursor(int id, CommandSet::MoveDirection direction){ return lua_state::instance().command_set()->menu_move_cursor(id, direction); }
	int menu_get_current(int id){ return lua_state::instance().command_set()->menu_get_current(id); }
	int menu_select(int id){ return lua_state::instance().command_set()->menu_select(id); }
	int sound_create(const mof::tstring& filepath){ return lua_state::instance().command_set()->sound_create(mof::tstring(filepath)); }
	void sound_play(int id){ return lua_state::instance().command_set()->sound_play(id); }
	//{{{ load_game_data
	int load_game_data(lua_State* l)
	{
        int n = lua_gettop(l);
        if(!lua_isstring(l , 1))throw std::runtime_error("invalid call");
		GameData::ptr p = lua_state::instance().command_set()->load_game_data(lua_tostring(l, 1));
		lua_pop(l, n);
       
   
	    lua_newtable(l);
		int tab_index1 = lua_gettop(l);
        std::vector<mof::tstring> items;
		for (int i = 0; i < p->data_.size(); ++i) {
            lua_pushnumber(l, i+1);// lua配列の添え字は1から
	    	lua_newtable(l);
			int tab_index2 = lua_gettop(l);
			for (auto itr = p->data_[i].begin(); itr != p->data_[i].end(); ++itr) {
            	lua_pushstring(l, itr->first.c_str());
            	lua_pushstring(l, itr->second.c_str());
            	lua_settable(l , tab_index2);
			}
            lua_settable(l , tab_index1);
        }
        return 1;

	}
	//}}}
	void print_debug(const mof::tstring& filepath){ lua_state::instance().command_set()->print_debug(mof::tstring(filepath)); }
	int picture_create(const mof::tstring& filepath){ return lua_state::instance().command_set()->picture_create(filepath); }
	int particlegen_create(){ return lua_state::instance().command_set()->particlegen_create(); }
	//{{{ object_set_behavior
	int object_set_behavior(lua_State* l)
	{
        int n = lua_gettop(l);
        if (!lua_isnumber(l , 1)) throw std::runtime_error("invalid call");
        if (!lua_isstring(l , 2)) throw std::runtime_error("invalid call");
        if (!lua_istable(l , 3)) throw std::runtime_error("invalid call");
        if (!lua_isnumber(l , 4)) throw std::runtime_error("invalid call");
        int id = lua_tonumber(l , 1);
		mof::tstring target = lua_tostring(l , 2);
		int period = lua_tonumber(l, 4);
		
		using namespace boost::algorithm;
		std::vector<mof::tstring> splited_list;
		split(splited_list, target, is_any_of("."));
		if (splited_list.back() == "color") {
			seq_parser<Color4f> parser;
			lua_state::instance().command_set()->set_color_behavior(id, target, parser.parse(l, 3), period);
		}
		else if (splited_list.back() == "position2") {
			seq_parser<Vector2D> parser;
			lua_state::instance().command_set()->set_position_behavior(id, target, parser.parse(l, 3), period);
		}
		else if (splited_list.back() == "position3") {
			seq_parser<Vector3D> parser;
			lua_state::instance().command_set()->set_position_behavior(id, target, parser.parse(l, 3), period);
		}
		else if (splited_list.back() == "size2") {
			seq_parser<Vector2D> parser;
			lua_state::instance().command_set()->set_size_behavior(id, target, parser.parse(l, 3), period);
		}


		else {
			throw std::logic_error("unknown target:" + target);
		}

        return 0;
	}
	//}}}
	int show(int id, const tstring& class_path){ return lua_state::instance().command_set()->show(id, class_path); }
	int hide(int id, const tstring& class_path){ return lua_state::instance().command_set()->hide(id, class_path); }
	void dispose(int id, const tstring& class_path){ lua_state::instance().command_set()->dispose(id, class_path); }
	void get_property(int id, const tstring& class_path){ lua_state::instance().command_set()->dispose(id, class_path); }
	//{{{ get_properties
	int get_properties(lua_State* l)
	{
        int n = lua_gettop(l);
        if (!lua_isnumber(l , 1)) throw std::runtime_error("invalid call");
        if(!lua_isstring(l , 2))throw std::runtime_error("invalid call");
        int id = lua_tonumber(l , 1);
		mof::tstring class_path = lua_tostring(l , 2);

		GameData::ptr p = lua_state::instance().command_set()->get_properties(id, class_path);
		lua_pop(l, n);
   
		assert(p->data_.size() == 1);
	    lua_newtable(l);
		int tab_index1 = lua_gettop(l);
        std::vector<mof::tstring> items;

		for (auto itr = p->data_[0].begin(); itr != p->data_[0].end(); ++itr) {
           	lua_pushstring(l, itr->first.c_str());
           	lua_pushstring(l, itr->second.c_str());
           	lua_settable(l , tab_index1);
		}
        return 1;
	}
	//}}}
//}}}
//{{{ Impl
	struct lua_state::Impl
	{
		lua_State* l_;
		std::shared_ptr<mof::script::CommandSet> commands_;

		Impl()
		: l_(NULL)
		{
		}

		~Impl()
		{
			if(l_)lua_close(l_);
		}
	};
//}}}
//{{{ constructor
	lua_state::lua_state()
	: impl_(new Impl())
	{	
		impl_->l_ = lua_open();
		luaL_openlibs(impl_->l_);
		luabind::open(impl_->l_);
	}
//}}}
//{{{ destructor
	lua_state::~lua_state(){}
//}}} destructor
//{{{ bind *
	void lua_state::bind(std::shared_ptr<mof::script::CommandSet> commands)
	{
		impl_->commands_ = commands;
		luabind::module(impl_->l_)[luabind::def("messageNewImpl", &message_create)];
		luabind::module(impl_->l_)[luabind::def("messageNextImpl", &message_next)];
		luabind::module(impl_->l_)[luabind::def("waitFrameImpl", &wait_frame)];
		luabind::module(impl_->l_)[luabind::def("waitForKeyImpl", &wait_for_key)];
		luabind::module(impl_->l_)[luabind::def("getLastKeyImpl", &get_last_key)];
		lua_register(impl_->l_, "menuNewImpl", &menu_create);
		luabind::module(impl_->l_)[luabind::def("menuMoveCursorImpl", &menu_move_cursor)];
		luabind::module(impl_->l_)[luabind::def("menuGetCurrentImpl", &menu_get_current)];
		luabind::module(impl_->l_)[luabind::def("menuSelectImpl", &menu_select)];
		luabind::module(impl_->l_)[luabind::def("soundCreateImpl", &sound_create)];
		luabind::module(impl_->l_)[luabind::def("soundPlayImpl", &sound_play)];
		lua_register(impl_->l_, "loadGameDataImpl", &load_game_data);
		luabind::module(impl_->l_)[luabind::def("printDebugImpl", &print_debug)];
		luabind::module(impl_->l_)[luabind::def("pictureCreateImpl", &picture_create)];
		luabind::module(impl_->l_)[luabind::def("particlegenCreateImpl", &particlegen_create)];
		lua_register(impl_->l_, "objectSetBehaviorImpl", &object_set_behavior);
		luabind::module(impl_->l_)[luabind::def("showImpl", &show)];
		luabind::module(impl_->l_)[luabind::def("hideImpl", &hide)];
		luabind::module(impl_->l_)[luabind::def("disposeImpl", &dispose)];
		lua_register(impl_->l_, "objectGetPropertiesImpl", &get_properties);
	}
//}}}
//{{{ raw_lua
	lua_State* lua_state::raw_lua() const
	{
		return impl_->l_;
	}
//}}}
//{{{ command_set
	std::shared_ptr<mof::script::CommandSet>& const lua_state::command_set() const
	{
		return impl_->commands_;
	}
//}}}
//{{{ instance
	lua_state& lua_state::instance()
	{
		return singleton_lua;
	}
//}}}

}
}
