package onig4j.java.util.regex;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.MatchResult;
import onig4j.OnigRegex;
import onig4j.OnigRegion;

/**
 *
 * @author calico
 */
public class Matcher implements MatchResult {

    private static class OnigMatchResult implements MatchResult {
        
        private final OnigRegion region;
        private final CharSequence input;
        
        private OnigMatchResult(OnigRegion region, CharSequence input) {
            this.region = region.clone();
            this.input = input;
        }
        
        // START implements
        public int start() {
            return region.begin(0);
        }

        public int start(int group) {
            return region.begin(group);
        }

        public int end() {
            return region.end(0);
        }

        public int end(int group) {
            return region.end(group);
        }

        public String group() {
            return input.subSequence(region.begin(0), region.end(0)).toString();
        }

        public String group(int group) {
            return input.subSequence(region.begin(group), region.end(group)).toString();
        }

        public int groupCount() {
            return (region.count() > 0 ? region.count() - 1 : 0);
        }
        // END implements
    }
    
    // START implements
    public int start() {
        return region.begin(0);
    }

    public int start(int group) {
        return region.begin(group);
    }

    public int end() {
        return region.end(0);
    }

    public int end(int group) {
        return region.end(group);
    }

    public String group() {
        return input.subSequence(region.begin(0), region.end(0)).toString();
    }

    public String group(int group) {
        return input.subSequence(region.begin(group), region.end(group)).toString();
    }

    public int groupCount() {
        return (region.count() > 0 ? region.count() - 1 : 0);
    }
    // END implements
    
    private Pattern pattern;
    private CharSequence input;
    private int end;
    private int start;
    private int range;
    private OnigRegion region;
    
    Matcher(Pattern pattern, CharSequence input) {
        this.pattern = pattern;
        this.input = input;
        this.end = input.length();
        this.range = this.end;
        this.region = new OnigRegion();
    }
    
    public Pattern pattern() {
        return pattern;
    }

    public MatchResult toMatchResult() {
        return new OnigMatchResult(region, input);
    }

    public Matcher usePattern(Pattern newPattern) {
        pattern = newPattern;
        region.clear();
        region.resize(0);
        return this;
    }

    public Matcher reset() {
        start = 0;
        end = input.length();
        range = end;
        region.clear();
        region.resize(0);
        return this;
    }

    public Matcher reset(CharSequence input) {
        this.input = input;
        return reset();
    }

    public boolean matches() {
        return (pattern.regex.match(input, end, 0, region) >= 0);
    }

    public boolean find() {
        final int ret = pattern.regex.search(input, end, start, range, region);
        if (ret >= 0) {
            start = region.end(0);
            return true;
        } else {
            return false;
        }
    }

    public boolean find(int start) {
        this.start = start;
        return find();
    }

    public boolean lookingAt() {
        return (pattern.regex.search(input, end, start, end, region) >= 0);
    }

    public static String quoteReplacement(String s) {
        throw new UnsupportedOperationException("Not supported yet");
    }

    public Matcher appendReplacement(StringBuffer sb, String replacement) {
        sb.append(replace(new StringBuilder(replacement), 1, false));
        return this;
    }

    public StringBuffer appendTail(StringBuffer sb) {
        if (start < end) {
            sb.append(input.subSequence(start, end).toString());                
        }
        return sb;
    }

    public String replace(StringBuilder replacement, int limit, boolean isAppendTail) {
        final List<Object> replace = new ArrayList<Object>();
        final boolean hasPrevReference = parseReplacement(replacement, replace);
        final StringBuilder replaced = new StringBuilder();
        int ret;
        while ((ret = pattern.regex.search(input, end, start, range, region)) >= 0) {
            replaced.append(input.subSequence(start, ret).toString());
            start = region.end(0);
            if (hasPrevReference) {
                for (final Object elm : replace) {
                    if (elm instanceof Integer) {
                        replaced.append(group((Integer)elm));
                    } else {
                        replaced.append(elm);
                    }
                }
            } else {
                replaced.append(replacement);
            }
            --limit;
            if (limit == 0) {
                break;
            }
        }
        if (isAppendTail && start < end) {
            replaced.append(input.subSequence(start, end).toString());                
        }
        return replaced.toString();
    }
    
    private static void escapeLiteralCharacters(StringBuilder literal) {
        int i = 0;
        while ((i = literal.indexOf("\\", i)) != -1) {
            literal.deleteCharAt(i);
            if (i < literal.length()) {
                if (literal.charAt(i) == '\\') {
                    ++i;
                }
            } else {
                break;
            }
        }
    }

    private boolean parseReplacement(StringBuilder replacement, List<Object> replace) {
        final OnigRegex regex = new OnigRegex("\\$[0-9]{1}");
        try {
            final int e = replacement.length();
            int s = 0;
            int ret = regex.search(replacement, e, s, e);
            if (ret < 0) {
                // 置換文字列に前方参照を含まない場合
                escapeLiteralCharacters(replacement);
                return false;

            } else {
                // 置換文字列に前方参照を含んでいる場合
                boolean hasPrevReference = false;
                do {
                    final StringBuilder literal
                            = new StringBuilder(replacement.substring(s, ret));
                    int yen = 0;
                    if (literal.length() > 0) {
                        for (int i = literal.length() - 1; literal.charAt(i) == '\\'; --i) {
                            ++yen;
                        }
                        escapeLiteralCharacters(literal);
                    }
                    s = ret + 2;
                    if ((yen % 2) == 0) {
                        // '$'がエスケープされていない場合は、番号をInteger型で格納する
                        replace.add(literal);
                        replace.add(Integer.parseInt(replacement.substring(ret + 1, s)));
                        hasPrevReference = true;
                    } else {
                        // '$'がエスケープされている場合は、文字列として格納する
                        replace.add(literal.append(replacement.substring(ret, s)).toString());
                    }
                } while ((ret = regex.search(replacement, e, s, e))>= 0);
                if (hasPrevReference) {
                    if (s < e && !replace.isEmpty()) {
                        replace.add(replacement.substring(s, e));
                    }
                    return true;
                    
                } else {
                    // 置換文字列に前方参照を含まない場合
                    escapeLiteralCharacters(replacement);
                    return false;
                }
            }
        } finally {
            regex.close();
        }
    }
    
    public String replaceAll(String replacement) {
        start = 0;
        return replace(new StringBuilder(replacement), -1, true);
     }

    public String replaceFirst(String replacement) {
        start = 0;
        return replace(new StringBuilder(replacement), 1, true);
    }

    public Matcher region(int start, int end) {
        this.start = start;
        this.range = end;
        return this;
    }

    public int regionStart() {
        return start;
    }

    public int regionEnd() {
        return range;
    }

    public boolean hasTransparentBounds() {
        return false;
    }

    public Matcher useTransparentBounds(boolean b) {
        throw new UnsupportedOperationException("Not supported yet");
//        hasTransparentBounds = b;
//        return this;
    }

    public boolean hasAnchoringBounds() {
        return true;
    }

    public Matcher useAnchoringBounds(boolean b) {
        throw new UnsupportedOperationException("Not supported yet");
    }

    @Override
    public String toString() {
        return pattern.pattern();
    }

    public boolean hitEnd() {
        throw new UnsupportedOperationException("Not supported yet");
    }

    public boolean requireEnd() {
        throw new UnsupportedOperationException("Not supported yet");
    }
}
