package onig4j.java.util.regex;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;
import onig4j.OnigOptionType;
import onig4j.OnigRegex;
import onig4j.OnigRegion;
import onig4j.OnigSyntaxType;

/**
 *
 * @author calico
 */
public final class Pattern implements Serializable {
    private static final long serialVersionUID = 1L;
    
    public static final int CANON_EQ = 128;
    public static final int CASE_INSENSITIVE = 2;
    public static final int COMMENTS = 4;
    public static final int DOTALL = 32;
    public static final int LITERAL = 16;
    /**
     * MEMO: 鬼車の行末記号は「\n」にのみマッチする
     */
    public static final int MULTILINE = 8;
    public static final int UNICODE_CASE = 64;
    public static final int UNIX_LINES = 1;
    
    /**
     * Unicodeエスケープシーケンス '\\uXXXX' を抽出する為の正規表現オブジェクト
     */
    private static final OnigRegex unicode = new OnigRegex("\\\\u\\h{4}");
    
    final OnigRegex regex;
//    private List<OnigOptionType> searchOptions;
    private final int flags;

    private Pattern(String pattern, int flags) {
        final int end = pattern.length();
        int ret = unicode.search(pattern, end, 0, end);
        if (ret >= 0) {
            // 正規表現にUnicodeエスケープシーケンスが存在する場合は文字に変換して置き換える
            final StringBuilder ptn = new StringBuilder();
            int start = 0;
            do {
                ptn.append(pattern.substring(start, ret));
                start = ret + 6;
                ptn.append((char) Integer.parseInt(pattern.substring(ret + 2, start), 16));
            } while ((start < end) && (ret = unicode.search(pattern, end, start, end)) >= 0);
            if (start < end) {
                ptn.append(pattern.substring(start));
            }
            pattern = ptn.toString();
        }
        
//        searchOptions = new ArrayList<OnigOptionType>(2);
//        searchOptions.add(OnigOptionType.ONIG_OPTION_NOTBOL);
//        searchOptions.add(OnigOptionType.ONIG_OPTION_NOTEOL);

        final List<OnigOptionType> options = new ArrayList<OnigOptionType>();
        if ((flags & CASE_INSENSITIVE) != 0) {
            options.add(OnigOptionType.ONIG_OPTION_IGNORECASE);
        }
        if ((flags & MULTILINE) != 0) {
            options.add(OnigOptionType.ONIG_OPTION_NEGATE_SINGLELINE);
        }
        if ((flags & DOTALL) != 0) {
            options.add(OnigOptionType.ONIG_OPTION_MULTILINE);
        }
        if ((flags & UNIX_LINES) == 0) {
            Logger.getLogger(Pattern.class.getName()).warning("Only the UNIX_LINES option is supported");
        }
        if ((flags & COMMENTS) != 0) {
            throw new UnsupportedOperationException("COMMENTS option is not supported yet");
        }
        if ((flags & LITERAL) != 0) {
            throw new UnsupportedOperationException("LITERAL option is not supported yet");
        }
        if ((flags & UNICODE_CASE) == 0) {
            Logger.getLogger(Pattern.class.getName()).warning("Only the UNICODE_CASE option is supported");
        }
        if ((flags & CANON_EQ) != 0) {
            throw new UnsupportedOperationException("CANON_EQ option is not supported yet");
        }

        this.regex = new OnigRegex(pattern, OnigSyntaxType.ONIG_SYNTAX_JAVA, options);
        this.flags = flags;
    }

    public static Pattern compile(String regex) {
        return new Pattern(regex, 0);
    }

    public static Pattern compile(String regex, int flags) {
        return new Pattern(regex, flags);
    }

    public int flags() {
        return flags;
    }

    public  Matcher matcher(CharSequence input) {
        return new Matcher(this, input);
    }

    public static boolean matches(String regex, CharSequence input) {
        return compile(regex).matcher(input).matches();
    }
    
    public String pattern() {
        return regex.pattern();
    }
    
    public static String quote(String s) {
        throw new UnsupportedOperationException("Not supported yet");
    }
    
    /**
     * 
     * @param input
     * @return The array of strings computed by splitting the input around matches of this pattern
     * @see #split(java.lang.CharSequence, int)
     */
    public String[] split(CharSequence input) {
        return split(input, 0);
    }
    
    public String[] split(CharSequence input, int limit) {
        final List<String> list = new ArrayList<String>();
        final OnigRegion region = new OnigRegion();
        final boolean isTrim = (limit == 0);
        int start = 0;
        final int end = input.length();
        int offset = 0;
        int ret;
//        OnigRegex negative = null;
        while ((ret = regex.search(input, end, start + offset, end, region)) >= 0) {
            list.add(input.subSequence(start, ret).toString());
            start = region.end(0);
            --limit;
            if (limit == 0) {
                break;
            }
            if (region.begin(0) == start) {
                // 行末記号（境界正規表現）の場合は無限ループに陥るのを防ぐために1文字ずらす
                offset = 1;
                if (start + offset >= end) {
                    break;
                }
                
//                if (negative == null) {
//                    EnumSet<OnigOptionType> options = regex.getOptionType();
////                    options.add(OnigOptionType.ONIG_OPTION_FIND_LONGEST);
//                    negative = new OnigRegex("[^" + regex.pattern() + "]", regex.getSyntaxType(), options);
//                }
//                if (ret != 0) {
//                    offset = 0;
//                }
//                ret = negative.match(input, end, start + offset);
//                if (ret > 0) {
//                    offset += ret;
//                }
//                if (start + offset >= end) {
//                    break;
//                }
////                ret = negative.search(input, end, start + offset, range, region);
////                start = ret + 1;
//                if (start + offset >= end) {
//                    break;
//                }
            }
        }
        if ((limit == 0 || ret < 0) && (start < end)) {
            list.add(input.subSequence(start, end).toString());
        }
        region.close();
//        if (negative != null) {
//            negative.close();
//        }
        
        if (isTrim && !list.isEmpty()) {
            int i = list.size() - 1;
            while (list.get(i).length() == 0) {
                // skip
                --i;
            }
            if (i + 1 < list.size()) {
                return list.subList(0, i + 1).toArray(new String[0]);
            }
        }
        return list.toArray(new String[0]);
    }
    
    /**
     * 
     * @return The string representation of this pattern
     * @see #pattern()
     */
    @Override
    public String toString() {
        return regex.pattern();
    }
}
