/**
 * @file bmpfile.h
 * @author Shinichiro Nakamura
 * @brief 小規模組み込みシステム向けBMP I/Oの実装。
 */

/*
 * ===============================================================
 *  Tiny BMP I/O Module
 *  Version 0.0.1
 * ===============================================================
 * Copyright (c) 2010-2011 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#ifndef BMPFILE_H
#define BMPFILE_H

#include "bmpint.h"

#ifdef __cplusplus
extern "C" {
#endif

/**
 * @brief BMPファイルを読み込む。
 * @details
 * この関数はBMPファイルが「どこに格納されているのか？」について全く感知しない。
 * また、読み込んだピクセルデータを「どこに格納するのか？」についても感知しない。
 * ユーザは、これら二つの抽象化された操作を提供する関数へのポインタを提供する必要がある。
 *
 * 例えば、小規模組み込みシステムにおいて、BMPファイルのイメージがSDカードやフラッシュ上に書かれているかもしれない。
 * また、読み込んだピクセルをSPI接続されたSRAMに格納したり、そのままディスプレイに表示したい事もあるだろう。
 * いずれにせよこれらの機能を実現するのは、この関数に渡すそれぞれの関数内部で実装詳細を提供すれば良い。
 *
 * @param func_fread ファイル読み込み関数。
 * @param extobj_func_fread ファイル読み込み関数に渡すパラメータ。
 * @param func_pixel_write ピクセル書き込み関数。
 * @param extobj_pixel_write ピクセル書き込み関数に渡すパラメータ。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int bmpfile_read(
        int (*func_fread)(void *buf, const unsigned int size, void *extobj_func_fread),
        void *extobj_func_fread,
        void (*func_pixel_write)(const int x, const int y, const uint8_t r, const uint8_t g, const uint8_t b, void *extobj_pixel_write),
        void *extobj_pixel_write);

/**
 * @brief BMPファイルを書き込む。
 * @details
 * この関数はBMPファイルが「どこに格納されているのか？」について全く感知しない。
 * また、書き込むためのピクセルデータが「どこに格納されているのか？」についても感知しない。
 * ユーザは、これら二つの抽象化された操作を提供する関数へのポインタを提供する必要がある。
 *
 * 例えば、小規模組み込みシステムにおいて、BMPファイルのイメージをSDカードやフラッシュ上に書きたいかもしれない。
 * また、書き込むピクセルはSPI接続されたSRAMに格納されているか、ディスプレイ上のメモリから直接取得したい事もあるだろう。
 * いずれにせよこれらの機能を実現するのは、この関数に渡すそれぞれの関数内部で実装詳細を提供すれば良い。
 *
 * @param func_fwrite ファイル書き込み関数。
 * @param extobj_func_fwrite ファイル書き込み関数に渡すパラメータ。
 * @param func_pixel_read ピクセル読み込み関数。
 * @param extobj_pixel_read ピクセル読み込み関数に渡すパラメータ。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int bmpfile_write(
        const int w,
        const int h,
        int (*func_fwrite)(const void *buf, const unsigned int size, void *extobj_func_fwrite),
        void *extobj_func_fwrite,
        void (*func_pixel_read)(const int x, const int y, uint8_t *r, uint8_t *g, uint8_t *b, void *extobj_pixel_read),
        void *extobj_pixel_read);

#ifdef __cplusplus
}
#endif

#endif

