/**
 * @file bmpimg.h
 * @author Shinichiro Nakamura
 * @brief 小規模組み込みシステム向けBMP I/Oの実装。
 */

/*
 * ===============================================================
 *  LCD Image Tool
 * ===============================================================
 * Copyright (c) 2010-2011 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#ifndef BMPIMG_H
#define BMPIMG_H

#include <stdint.h>

typedef void (*PIXEL_WRITER)(const int x, const int y, const uint8_t r, const uint8_t g, const uint8_t b, void *extobj_pixel_writer);
typedef void (*PIXEL_READER)(const int x, const int y, uint8_t *r, uint8_t *g, uint8_t *b, void *extobj_pixel_reader);
typedef int (*FILE_READER)(void *buf, const uint32_t size, void *extobj_func_fread);
typedef int (*FILE_WRITER)(const void *buf, const uint32_t size, void *extobj_func_fwrite);

/**
 * @brief BMP画像ハンドラの実装。
 */
typedef struct {
    int w;                  /**< 画像横方向サイズ。 */
    int h;                  /**< 画像縦方向サイズ。 */
    PIXEL_WRITER writer;    /**< Pixel Writer. */
    void *writer_extobj;    /**< External Object for the pixel writer. */
    PIXEL_READER reader;    /**< Pixel Reader. */
    void *reader_extobj;    /**< External Object for the pixel reader. */
} bmpimg_t;

#define BMPIMG_GET_W(BMPIMG)    ((BMPIMG)->w)
#define BMPIMG_GET_H(BMPIMG)    ((BMPIMG)->h)

/**
 * @brief 色定義。
 * @brief 8ビットでフルレンジ。
 */
typedef struct {
    unsigned char r; /**< 赤。 */
    unsigned char g; /**< 緑。 */
    unsigned char b; /**< 青。 */
} bmpcol_t;

#define BMPIMG_FONT_DOT_WIDTH   (5) /**< フォントの横方向ドット数。 */
#define BMPIMG_FONT_DOT_HEIGHT  (7) /**< フォンとの縦方向ドット数。 */

/**
 * @brief 色を設定する。
 *
 * @param COL 色定義。
 * @param R 赤。
 * @param G 緑。
 * @param B 青。
 */
#define BMPIMG_SET_COLOR(COL, R, G, B)  do { (COL).r = (unsigned char)(R); (COL).g = (unsigned char)(G); (COL).b = (unsigned char)(B); } while (0)

/**
 * @brief 文字列の幅を取得する。
 *
 * @param SIZE サイズ。
 * @param NSTR 文字列の長さ。
 *
 * @return 文字列の幅。
 */
#define BMPIMG_GET_STRING_WIDTH(SIZE, NSTR) \
    ((((SIZE) * (BMPIMG_FONT_DOT_WIDTH + 1)) * ((NSTR) - 1)) + ((SIZE) * (BMPIMG_FONT_DOT_WIDTH)))

/**
 * @brief 文字列の高さを取得する。
 *
 * @param SIZE サイズ。
 *
 * @return 文字列の高さ。
 */
#define BMPIMG_GET_STRING_HEIGHT(SIZE) \
    ((SIZE) * (BMPIMG_FONT_DOT_HEIGHT))

/**
 * @brief BMP画像をオープンする。
 *
 * @param p BMP画像ハンドラ。
 * @param w 画像横方向サイズ。
 * @param h 画像縦方向サイズ。
 * @param writer ピクセル書き込み関数。
 * @param writer_extobj ピクセル書き込み関数に渡すパラメータ。
 * @param reader ピクセル読み込み関数。
 * @param reader_extobj ピクセル読み込み関数に渡すパラメータ。
 */
void bmpimg_open(
        bmpimg_t *p,
        const int w,
        const int h,
        PIXEL_WRITER writer,
        void *writer_extobj,
        PIXEL_READER reader,
        void *reader_extobj);

/**
 * @brief ボックスを描画する。
 *
 * @param p BMP画像ハンドラ。
 * @param x1 始点X座標。
 * @param y1 始点Y座標。
 * @param x2 終点X座標。
 * @param y2 終点Y座標。
 * @param color 色。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int bmpimg_draw_box(bmpimg_t *p, const int x1, const int y1, const int x2, const int y2, bmpcol_t *color);

/**
 * @brief ボックスを描画する。
 *
 * @param p BMP画像ハンドラ。
 * @param x1 始点X座標。
 * @param y1 始点Y座標。
 * @param x2 終点X座標。
 * @param y2 終点Y座標。
 * @param color 色。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int bmpimg_fill_box(bmpimg_t *p, const int x1, const int y1, const int x2, const int y2, bmpcol_t *color);

/**
 * @brief 文字列を描画する。
 *
 * @param p BMP画像ハンドラ。
 * @param x 横方向描画開始ピクセル位置。
 * @param y 縦方向描画開始ピクセル位置。
 * @param size 1ピクセルのサイズ。
 * @param text 文字列。
 * @param color 描画色。
 */
int bmpimg_draw_string(bmpimg_t *p, const int x, const int y, const int size, const char *text, bmpcol_t *color);

/**
 * @brief 線を描画する。
 *
 * @param p BMP画像ハンドラ。
 * @param x1 始点X座標。
 * @param y1 始点Y座標。
 * @param x2 終点X座標。
 * @param y2 終点Y座標。
 * @param color 色。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int bmpimg_draw_line(bmpimg_t *p, const int x1, const int y1, const int x2, const int y2, bmpcol_t *color);

/**
 * @brief 点を描画する。
 *
 * @param p BMP画像ハンドラ。
 * @param x X座標。
 * @param y Y座標。
 * @param color 色。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int bmpimg_draw_pixel(bmpimg_t *p, const int x, const int y, bmpcol_t *color);

/**
 * @brief チェックボックスを描画する。
 *
 * @param p BMP画像ハンドラ。
 * @param x1 始点X座標。
 * @param y1 始点Y座標。
 * @param x2 終点X座標。
 * @param y2 終点Y座標。
 * @param state 選択状態。
 * @param color 色。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
void bmpimg_draw_checkbox(
        bmpimg_t *p,
        const int x1,
        const int y1,
        const int x2,
        const int y2,
        const int state,
        bmpcol_t *color);

/**
 * @brief プログレスバーを描画する。
 *
 * @param p BMP画像ハンドラ。
 * @param x1 始点X座標。
 * @param y1 始点Y座標。
 * @param x2 終点X座標。
 * @param y2 終点Y座標。
 * @param min 最小値。
 * @param max 最大値。
 * @param value 現在値。
 * @param color 色。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
void bmpimg_draw_progressbar(
        bmpimg_t *p,
        const int x1,
        const int y1,
        const int x2,
        const int y2,
        const int min,
        const int max,
        const int value,
        bmpcol_t *color);

/**
 * @brief BMPファイルから読み込む。
 *
 * @param p BMP画像ハンドラ。
 * @param reader ファイル読み込み関数。
 * @param reader_extobj ファイル読み込み関数に渡す外部オブジェクト。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int bmpimg_bmp_read(
        bmpimg_t *p,
        FILE_READER reader,
        void *reader_extobj);

/**
 * @brief BMPファイルに書き込む。
 *
 * @param p BMP画像ハンドラ。
 * @param writer ファイル書き込み関数。
 * @param writer_extobj ファイル書き込み関数に渡す外部オブジェクト。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int bmpimg_bmp_write(
        bmpimg_t *p,
        FILE_WRITER writer,
        void *writer_extobj);

/**
 * @brief BMP画像をクローズする。
 *
 * @param bit ビット深度。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int bmpimg_close(bmpimg_t *p);

#endif

