/**
 * @file lcdfont.c
 * @author Copyright(C) 2012 Shinichiro Nakamura
 */

/*
 * ===============================================================
 *  LCD Image Tool
 * ===============================================================
 * Copyright (c) 2012 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "lcdimg.h"
#include "lcdfont.h"
#include "util.h"

#define CODE_MIN    (0x00)
#define CODE_MAX    (0xFF)

#define CHAR_OFF    '.'
#define CHAR_ON     'o'
#define CHAR_DEF    '='

static int pickup_code(const char *text)
{
    char key[BUFSIZ];
    char value[BUFSIZ];
    char tmp[BUFSIZ];
    int i;

    /*
     * Copy the master data to a temporary variable.
     */
    strcpy(tmp, text);

    /*
     * Separate the key and value.
     */
    for (i = 0; i < (int)strlen(tmp); i++) {
        if (tmp[i] == CHAR_DEF) {
            tmp[i] = ' ';
        }
    }

    /*
     * Pickup the value.
     */
    if (sscanf(tmp, "%s %s", key, value) == 2) {
        if (strcmp("CODE", key) == 0) {
            int n;
            if (sscanf(value, "0x%02x", &n) == 1) {
                return n;
            }
            return -1;
        }
    }

    return -1;
}

int lcdfont_read(const char *filename, LCDFONT_READ_CALLBACK callback, void *extobj)
{
    char buf[BUFSIZ];
    FILE *fp;
    int code = -1;
    int y = 0;

    /*
     * Check the file name.
     */
    if (filename == NULL) {
        return -1;
    }
    if (strlen(filename) == 0) {
        return -1;
    }

    /*
     * Open the target file.
     */
    fp = fopen(filename, "r");
    if (fp == NULL) {
        return -1;
    }

    /*
     * Read the font data.
     */
    while (fgets(buf, sizeof(buf), fp) != NULL) {
        int tmp;
        util_remove_comment(buf);
        tmp = pickup_code(buf);
        if ((CODE_MIN <= tmp) && (tmp <= CODE_MAX)) {
            code = tmp;
        } else {
            if (FONT_X < strlen(buf)) {
                int x;
                for (x = 0; x < FONT_X; x++) {
                    if ((CODE_MIN <= code) && (code <= CODE_MAX)) {
                        callback(code, x, y, (buf[x] == CHAR_OFF) ? 0 : 1, extobj);
                    }
                }
                y++;
                if (FONT_Y <= y) {
                    y = 0;
                    code = -1;
                }
            }
        }
    }

    /*
     * Close the target file.
     */
    fclose(fp);

    return 0;
}

